/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>


void BCmain3d(int level)
{

    AllWinStruct  *AllWinData;
    LoadWinStruct *LoadWinData;
    SolveBlock    *SolBlock;
    
    int     num_levels;
    MGBlock *Meshes;   
    int     *BcTypeX, *BcTypeY, *BcTypeZ;
    double  *BcValueX,*BcValueY,*BcValueZ;
    
    int     num_loads;
    OneLoad *LoadList;
    int i,j,k,p,n,nsize;
    int nnx,nny,nnz;
    double la,lb,lc, lineload;
    int *Nodes, NumNodes;
    double Area;
    double val;
    
    int    TL;
    double Tval;
    double Texp;  
    double Tleft,Tright;  
       
    /*--------- end of declaration--------- */
    
    PutGetAllWinStruct(1, &AllWinData); 
    LoadWinData=AllWinData->LoadWinData;
    PutGetSolveBlock(1,&SolBlock); 
    
    num_loads=LoadWinData->num_loads3d;
    LoadList=LoadWinData->LoadList3d;
    
    num_levels=SolBlock->num_levels;
    Meshes=&(SolBlock->Meshes[level]);
    
    BcTypeX=Meshes->BcTypeX;
    BcTypeY=Meshes->BcTypeY;
    BcTypeZ=Meshes->BcTypeZ;
    
    BcValueX=Meshes->BcValueX;
    BcValueY=Meshes->BcValueY;
    BcValueZ=Meshes->BcValueZ;
    
    nnx=Meshes->nnx;
    nny=Meshes->nny;
    nnz=Meshes->nnz;
    
    la=Meshes->la;
    lb=Meshes->lb;
    lc=Meshes->lc;
   
    k=nnx*nny*nnz;
    for(i=0; i<k; i++)
    {
        BcTypeX[i]=0;
        BcTypeY[i]=0; 
        BcTypeZ[i]=0;
        BcValueX[i]=0.0;
        BcValueY[i]=0.0;   
        BcValueZ[i]=0.0;      
    } 
    
         /*      Boundary scheme       
         PrimFlag: 0:face 1:edge, 2:corner
         Faces ... NumFlag:  0-5 side number
         PhyFlag:  0:Ux  1:Uy  2: Uz  3:Fx  4:Fy  5:Fz
         BcVal:    value of B.C. 
         */
    NumNodes=0;
    if (NumNodes<(nnx*nny)) NumNodes=nnx*nny;
    if (NumNodes<(nnx*nnz)) NumNodes=nnx*nnz;
    if (NumNodes<(nny*nnz)) NumNodes=nny*nnz;
    
    
    InitVector_int(NumNodes, &Nodes);
    printf("Are you awake?\n");
    for(n=0; n<num_loads; n++)
    {
               
        printf("NumLoads: %i\n",n);
        if (0==(LoadList[n].PrimFlag)) 
        { 
            CollectNodesFace(LoadList[n].NumFlag, la, lb, lc,
                       nnx, nny, nnz, &NumNodes, Nodes, &Area);
            val=LoadList[n].BCVal;
            
            printf("NumNodes: %i\n",NumNodes);
            for(i=0; i<NumNodes; i++)
            {
                if (0==LoadList[n].PhyFlag) {BcTypeX[Nodes[i]]=1; BcValueX[Nodes[i]]=val;  }
                if (1==LoadList[n].PhyFlag) {BcTypeY[Nodes[i]]=1; BcValueY[Nodes[i]]=val;  }
                if (2==LoadList[n].PhyFlag) {BcTypeZ[Nodes[i]]=1; BcValueZ[Nodes[i]]=val;  }
            }
            val=val*Area;
            for(i=0; i<NumNodes; i++)
            {
                if (3==LoadList[n].PhyFlag) {BcTypeX[Nodes[i]]=0; BcValueX[Nodes[i]]+=val;  }
                if (4==LoadList[n].PhyFlag) {BcTypeY[Nodes[i]]=0; BcValueY[Nodes[i]]+=val;  }
                if (5==LoadList[n].PhyFlag) {BcTypeZ[Nodes[i]]=0; BcValueZ[Nodes[i]]+=val;  }
            }
        } 
    }                        
    free(Nodes);                                                                                 
    
    // reduce node force at edges (and corners)
    for(i=0; i<nnx; i++)
    for(j=0; j<nny; j+=(nny-1))
    for(k=0; k<nnz; k+=(nnz-1))
    {
        p=i+nnx*j+nnx*nny*k;
        if (0==BcTypeX[p]) BcValueX[p]/=2.0; 
        if (0==BcTypeY[p]) BcValueY[p]/=2.0;
        if (0==BcTypeZ[p]) BcValueZ[p]/=2.0;   
    }
    
    for(i=0; i<nnx; i+=(nnx-1))
    for(j=0; j<nny; j++)
    for(k=0; k<nnz; k+=(nnz-1))
    {
        p=i+nnx*j+nnx*nny*k;
        if (0==BcTypeX[p]) BcValueX[p]/=2.0; 
        if (0==BcTypeY[p]) BcValueY[p]/=2.0;
        if (0==BcTypeZ[p]) BcValueZ[p]/=2.0;   
    }
    
    for(i=0; i<nnx; i+=(nnx-1))
    for(j=0; j<nny; j+=(nny-1))
    for(k=0; k<nnz; k++)
    {
        p=i+nnx*j+nnx*nny*k;
        if (0==BcTypeX[p]) BcValueX[p]/=2.0; 
        if (0==BcTypeY[p]) BcValueY[p]/=2.0;
        if (0==BcTypeZ[p]) BcValueZ[p]/=2.0;   
    }
    
    for(i=0; i<nnx; i+=(nnx-1))
    for(j=0; j<nny; j+=(nny-1))
    for(k=0; k<nnz; k+=(nnz-1))
    {
        p=i+nnx*j+nnx*nny*k;
        if (0==BcTypeX[p]) BcValueX[p]*=2.0; 
        if (0==BcTypeY[p]) BcValueY[p]*=2.0;
        if (0==BcTypeZ[p]) BcValueZ[p]*=2.0;   
    }
    
    
    
    k=nnx*nny*nnz;
    for(i=0; i<k; i++)
    {
        if(1==BcTypeX[i]) Meshes->Ux[i]=BcValueX[i];  
        if(1==BcTypeY[i]) Meshes->Uy[i]=BcValueY[i];   
        if(1==BcTypeZ[i]) Meshes->Uz[i]=BcValueZ[i];       
    }
    
    return;
}

/*---------------------------------------------------------
	CollectNodesFace
----------------------------------------------------------*/
void CollectNodesFace(int face, double la, double lb, double lc,
                       int nnx, int nny, int nnz, int *NumNodes, 
                       int *Nodes, double *Area)
{
   int i,j,k,p,nnxnny;
   p=0;
   nnxnny=nnx*nny;
   
   if (4==face)
   {
       k=0;
       for(i=0; i<nnx; i++)
       for(j=0; j<nny; j++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=la*lb;
       *NumNodes=p;
   }
   
   if (2==face)
   {
       j=0;
       for(i=0; i<nnx; i++)
       for(k=0; k<nnz; k++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=la*lc;
       *NumNodes=p;
   }
   
   if (1==face)
   {
       i=(nnx-1);
       for(j=0; j<nny; j++)
       for(k=0; k<nnz; k++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=lb*lc;
       *NumNodes=p;
   }
   
   if (3==face)
   {
       j=(nny-1);
       for(i=0; i<nnx; i++)
       for(k=0; k<nnz; k++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=la*lc;
       *NumNodes=p;
   }
   
   if (0==face)
   {
       i=0;
       for(j=0; j<nny; j++)
       for(k=0; k<nnz; k++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=lb*lc;
       *NumNodes=p;
   }
   
   if (5==face)
   {
       k=(nnz-1);
       for(i=0; i<nnx; i++)
       for(j=0; j<nny; j++)
       {
            Nodes[p]=i+j*nnx+k*nnxnny;
            p++;
       }
       Area[0]=la*lb;
       *NumNodes=p;
   }
   
   return;
}
