/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*------------------------------------------------------------------------------
     InitMGBlock3d
-------------------------------------------------------------------------------*/
void InitMGBlock3d(double lx, double ly, double lz, 
                   int nex, int ney, int nez,
                   MGBlock *MG)
{   
    int nnsize, sizeBC;
    int nnx, nny, nnz;
    
    nnx=nex+1;
    nny=ney+1;
    nnz=nez+1;
    nnsize = nnx*nny*nnz;
    
    (MG->nnx) = nnx;
    (MG->nny) = nny;
    (MG->nnz) = nnz;
    (MG->nex) = nex;
    (MG->ney) = ney;
    (MG->nez) = nez;
    (MG->la)  = (lx/(double)nex);
    (MG->lb)  = (ly/(double)ney);
    (MG->lc)  = (lz/(double)nez);
    
    InitVector_double(nnsize, &(MG->MatFieldE));
    InitVector_int(nnsize, &(MG->MatFieldInt));
    
    InitVector_double(nnsize, &(MG->Ux));      
    InitVector_double(nnsize, &(MG->Uy));   
    InitVector_double(nnsize, &(MG->Uz));     
    
    InitVector_double(nnsize, &(MG->dUx));	 
    InitVector_double(nnsize, &(MG->dUy));
    InitVector_double(nnsize, &(MG->dUz));
    
    InitVector_double(nnsize, &(MG->Fx));      
    InitVector_double(nnsize, &(MG->Fy)); 
    InitVector_double(nnsize, &(MG->Fz)); 
    
    InitVector_double(nnsize, &(MG->Dx));      
    InitVector_double(nnsize, &(MG->Dy));
    InitVector_double(nnsize, &(MG->Dz));
     
    InitVector_double(nnsize, &(MG->Adx));
    InitVector_double(nnsize, &(MG->Ady));
    InitVector_double(nnsize, &(MG->Adz));		
    
    InitVector_double(nnsize, &(MG->Mpc));		 	 
    
    InitVector_double(nnsize, &(MG->SwapX));	 
    InitVector_double(nnsize, &(MG->SwapY));
    InitVector_double(nnsize, &(MG->SwapZ));
    
    InitVector_double(nnsize, &(MG->Swap));
    
    InitVector_double(nnsize, &(MG->Post1));
    InitVector_double(nnsize, &(MG->Post2));
    InitVector_double(nnsize, &(MG->Post3));
    InitVector_double(nnsize, &(MG->Post4));
    
    InitVector_double(nnsize, &(MG->Post5));
    InitVector_double(nnsize, &(MG->Post6));
    InitVector_double(nnsize, &(MG->Post7));
    InitVector_double(nnsize, &(MG->Post8));
    
    InitVector_double(nnsize, &(MG->EqStrain));
     
    InitVector_int(nnsize,      &(MG->BcTypeX));    
    InitVector_int(nnsize,      &(MG->BcTypeY));    
    InitVector_int(nnsize,      &(MG->BcTypeZ));   
    InitVector_double(nnsize,   &(MG->BcValueX)); 
    InitVector_double(nnsize,   &(MG->BcValueY));
    InitVector_double(nnsize,   &(MG->BcValueZ));
    
    InitVector_double(nnsize,   &(MG->Kappa)); 
    InitVector_double(nnsize,   &(MG->DamageOmega));
    
    
    if (nnx>nny) nnsize=nnx; else nnsize=nny;
    InitVector_double(nnsize,   &(MG->weights));
		
    return;
}


/*------------------------------------------------------------------------------
     InitMGBlock2and3d
-------------------------------------------------------------------------------*/
void InitMGBlock2and3d(double lx, double ly, double lz, 
                   int nex, int ney, int nez,
                   MGBlock *MG)
{   
    int nnsize, totsize, sizeBC;
    int nnx, nny, nnz;
    int ModelDim;
    int PostFactor;
    
    
    ModelDim=PutGetModelDim(1,123);
    
    if (2==ModelDim) nez=0;
    
    nnx=nex+1;
    nny=ney+1;
    nnz=nez+1;
    nnsize = nnx*nny*nnz;
    totsize = ModelDim*nnsize;
    
    MG->nnsize=nnsize;
    MG->totsize=totsize;
    
    
    (MG->nnx) = nnx;
    (MG->nny) = nny;
    (MG->nnz) = nnz;
    (MG->nex) = nex;
    (MG->ney) = ney;
    (MG->nez) = nez;
    (MG->la)  = (lx/(double)nex);
    (MG->lb)  = (ly/(double)ney);
    (MG->lc)  = (lz/(double)nez);
    
    InitVector_double(nnsize, &(MG->MatFieldE));
    InitVector_double(nnsize, &(MG->KnMatFieldE));
    InitVector_int(nnsize, &(MG->MatFieldInt));
    
    InitVector_double(totsize, &(MG->U));
    MG->Ux=&(MG->U[0]);
    MG->Uy=&(MG->U[nnsize]);
    if (3==ModelDim) MG->Uz=&(MG->U[2*nnsize]);
    
    InitVector_double(totsize, &(MG->dU));
    MG->dUx=&(MG->dU[0]);
    MG->dUy=&(MG->dU[nnsize]);
    if (3==ModelDim) MG->dUz=&(MG->dU[2*nnsize]);

    InitVector_double(totsize, &(MG->F));
    MG->Fx=&(MG->F[0]);
    MG->Fy=&(MG->F[nnsize]);
    if (3==ModelDim) MG->Fz=&(MG->F[2*nnsize]);
 
    InitVector_double(totsize, &(MG->D));
    MG->Dx=&(MG->D[0]);
    MG->Dy=&(MG->D[nnsize]);
    if (3==ModelDim) MG->Dz=&(MG->D[2*nnsize]);
    
    InitVector_double(totsize, &(MG->Ad));
    MG->Adx=&(MG->Ad[0]);
    MG->Ady=&(MG->Ad[nnsize]);
    if (3==ModelDim) MG->Adz=&(MG->Ad[2*nnsize]);	
    
    InitVector_double(nnsize, &(MG->Mpc));		 	 

    InitVector_double(totsize, &(MG->SwapXYZ));
    MG->SwapX=&(MG->SwapXYZ[0]);
    MG->SwapY=&(MG->SwapXYZ[nnsize]);
    if (3==ModelDim) MG->SwapZ=&(MG->SwapXYZ[2*nnsize]);
    
    InitVector_double(nnsize, &(MG->Swap));
    
    if (3==ModelDim) PostFactor=8; else PostFactor=4;
    
    InitVector_double(nnsize*PostFactor, &(MG->Post));
    
    MG->Post1=&(MG->Post[0]);
    MG->Post2=&(MG->Post[nnsize]);
    MG->Post3=&(MG->Post[nnsize*2]);
    MG->Post4=&(MG->Post[nnsize*3]);
    
    if (3==ModelDim)
    {
        MG->Post5=&(MG->Post[nnsize*4]);
        MG->Post6=&(MG->Post[nnsize*5]);
        MG->Post7=&(MG->Post[nnsize*6]);
        MG->Post8=&(MG->Post[nnsize*7]);
    }
    
    InitVector_double(nnsize, &(MG->EqStrain));

    InitVector_int(totsize, &(MG->BcType));
    MG->BcTypeX=&(MG->BcType[0]);
    MG->BcTypeY=&(MG->BcType[nnsize]);
    if (3==ModelDim) MG->BcTypeZ=&(MG->BcType[2*nnsize]);

    InitVector_double(totsize, &(MG->BcValue));
    MG->BcValueX=&(MG->BcValue[0]);
    MG->BcValueY=&(MG->BcValue[nnsize]);
    if (3==ModelDim) MG->BcValueZ=&(MG->BcValue[2*nnsize]);

    
    InitVector_double(nnsize,   &(MG->Kappa)); 
    InitVector_double(nnsize,   &(MG->DamageOmega));
    
    InitVector_int(nnsize, &(MG->ColorIndex));
    
    if (nnx>nny) nnsize=nnx; else nnsize=nny;
    InitVector_double(nnsize,   &(MG->weights));
    
    
		
    return;
}

/*------------------------------------------------------------------------------
     FreeMGBlock2and3d
-------------------------------------------------------------------------------*/

void FreeMGBlock2and3d(MGBlock *MG)
{   
   
    free(MG->MatFieldE);
    free(MG->KnMatFieldE);
    free(MG->MatFieldInt);

    
    free(MG->U);             
    free(MG->dU);	 
    free(MG->F);      
    free(MG->D);      
    free(MG->Ad);      	
    free(MG->Mpc); 	  
    free(MG->SwapXYZ);	 
   
    free(MG->Swap); 
    
    free(MG->Post); 

    free(MG->EqStrain); 
    
    free(MG->BcType);
    free(MG->BcValue); 
	
    free(MG->Kappa); 
    free(MG->DamageOmega);
   	
	free(MG->weights);
	free(MG->ColorIndex);
 	
    return;
}


/*------------------------------------------------------------------------------
     InitMGBlock2and3dPreCond
-------------------------------------------------------------------------------*/
void InitMGBlock2and3dPreCond(MGBlock *MG, MGBlock *PG)
{   
    int nnsize, totsize, sizeBC;
    int nnx, nny, nnz;
    int ModelDim;
    int PostFactor;
    
    
    ModelDim=PutGetModelDim(1,123);
    PG->nex=MG->nex;
    PG->ney=MG->ney;
    PG->nez=MG->nez;
    
    PG->nnx=MG->nnx;
    PG->nny=MG->nny;
    PG->nnz=MG->nnz;
    
    PG->nnsize  = MG->nnsize;
    PG->totsize = MG->totsize;
    
    nnsize  = PG->nnsize;
    totsize = PG->totsize;

    PG->la = MG->la;
    PG->lb = MG->lb;
    PG->lc = MG->lc; 

    InitVector_double(totsize, &(PG->U));
    PG->Ux=&(PG->U[0]);
    PG->Uy=&(PG->U[nnsize]);
    if (3==ModelDim) PG->Uz=&(PG->U[2*nnsize]);
    
    InitVector_double(totsize, &(PG->dU));
    PG->dUx=&(PG->dU[0]);
    PG->dUy=&(PG->dU[nnsize]);
    if (3==ModelDim) PG->dUz=&(PG->dU[2*nnsize]);

    InitVector_double(totsize, &(PG->F));
    PG->Fx=&(PG->F[0]);
    PG->Fy=&(PG->F[nnsize]);
    if (3==ModelDim) PG->Fz=&(PG->F[2*nnsize]);
 
    InitVector_double(totsize, &(PG->D));
    PG->Dx=&(PG->D[0]);
    PG->Dy=&(PG->D[nnsize]);
    if (3==ModelDim) PG->Dz=&(PG->D[2*nnsize]);
    
    InitVector_double(totsize, &(PG->Ad));
    PG->Adx=&(PG->Ad[0]);
    PG->Ady=&(PG->Ad[nnsize]);
    if (3==ModelDim) PG->Adz=&(PG->Ad[2*nnsize]);	 	 

    InitVector_double(totsize, &(PG->SwapXYZ));
    PG->SwapX=&(PG->SwapXYZ[0]);
    PG->SwapY=&(PG->SwapXYZ[nnsize]);
    if (3==ModelDim) PG->SwapZ=&(PG->SwapXYZ[2*nnsize]);
    
    InitVector_double(nnsize, &(PG->Swap)); 
    
    InitVector_double(totsize, &(PG->BcValue));
    PG->BcValueX=&(PG->BcValue[0]);
    PG->BcValueY=&(PG->BcValue[nnsize]);
    if (3==ModelDim) PG->BcValueZ=&(PG->BcValue[2*nnsize]);
    
    //LINKS !!!
    
    PG->MatFieldE    =MG->MatFieldE;
    PG->MatFieldInt  =MG->MatFieldInt;
    PG->Mpc          =MG->Mpc;
    PG->BcType       =MG->BcType;
    PG->BcTypeX      =MG->BcTypeX;
    PG->BcTypeY      =MG->BcTypeY;
    PG->BcTypeZ      =MG->BcTypeZ;
    PG->weights      =MG->weights;
    PG->ColorIndex   =MG->ColorIndex;
    
		
    return;
}

/*------------------------------------------------------------------------------
     FreeMGBlock2and3dPreCond
-------------------------------------------------------------------------------*/
void FreeMGBlock2and3dPreCond(MGBlock *MG)
{      
    free(MG->U);             
    free(MG->dU);	 
    free(MG->F);      
    free(MG->D);      
    free(MG->Ad);      	 
    free(MG->SwapXYZ);	 
    free(MG->Swap);  
    free(MG->BcValue);  
		
    return;
}

/*------------------------------------------------------------------------------
     InitPreMeshes
-------------------------------------------------------------------------------*/
int  InitPreMeshes()
{
    int i, num_levels;
    MGBlock *M, *P;
    SolveBlock *SolBlock;
    
    PutGetSolveBlock(1,&SolBlock); 
    num_levels=SolBlock->num_levels; 
    M=SolBlock->Meshes;
    if (1==SolBlock->PreMeshFlag) return 0;
    SolBlock->PreMeshFlag=1;         
    InitMGData (&P, num_levels);
    SolBlock->PreMesh=P;

    for (i=0; i<num_levels; i++)    
    {
        InitMGBlock2and3dPreCond(&(M[i]), &(P[i]));
    }
    return 0;
}


