/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>

/*********************************************************************************
                               MAIN PROGRAM OF POSTPROCESSING
********************************************************************************/
void PostMainLoop3d(void)
{    
    int     i,j;
    int     num_levels;
    int     show_level;
    MGBlock *Meshes, *M;
    double lambda;
    double Radius;
    SolveBlock *SolBlock;
    GraphicBlock *graphblock;
 
    AllWinStruct *AllWinData;
    SolveWinStruct *SolveWinData;

    PutGetAllWinStruct(1, &AllWinData); 
    SolveWinData=AllWinData->SolveWinData;    
    
    PutGetGraphicBlock(1, &graphblock); 
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    show_level=graphblock->show_level;
    M=&(Meshes[show_level]);
  
 if (0==graphblock->RotateFlag) // && 0!=graphblock->RotateCoor) 
 {
    if(0==strcmp("E-Modul",graphblock->SelectObject)) 
    ElemMatrix2ElemNodes3d(M->nnx, M->nny, M->nnz, M->MatFieldE,
              M->Post1, M->Post2, M->Post3, M->Post4, 
              M->Post5, M->Post6, M->Post7, M->Post8); 
    
    if(0==strcmp("Material Type",graphblock->SelectObject)) 
    ElemMatrixInt2ElemNodes3d(M->nnx, M->nny, M->nnz, M->MatFieldInt,
              M->Post1, M->Post2, M->Post3, M->Post4, 
              M->Post5, M->Post6, M->Post7, M->Post8); 
    
    if(0==strcmp("Color Index",graphblock->SelectObject)) 
    ElemMatrixInt2ElemNodes3d(M->nnx, M->nny, M->nnz, M->ColorIndex,
              M->Post1, M->Post2, M->Post3, M->Post4, 
              M->Post5, M->Post6, M->Post7, M->Post8); 
 
    if(0==strcmp("Displacement Ux",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes3d(M->nnx, M->nny, M->nnz, M->Ux,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8); 
              
    if(0==strcmp("Displacement Uy",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes3d(M->nnx, M->nny, M->nnz, M->Uy,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);

    if(0==strcmp("Displacement Uz",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes3d(M->nnx, M->nny, M->nnz, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);

    if(0==strcmp("Strain Exx",graphblock->SelectObject)) 
    PostStrain3d(0, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8); 
              
    if(0==strcmp("Strain Eyy",graphblock->SelectObject)) 
    PostStrain3d(1, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Strain Ezz",graphblock->SelectObject)) 
    PostStrain3d(2, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);

    if(0==strcmp("Strain Exy",graphblock->SelectObject)) 
    PostStrain3d(3, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
              
    if(0==strcmp("Strain Eyz",graphblock->SelectObject)) 
    PostStrain3d(4, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Strain Exz",graphblock->SelectObject)) 
    PostStrain3d(5, M->nnx, M->nny, M->nnz, 
                    M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Stress Sxx",graphblock->SelectObject))
    PostStress3d(0, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Stress Syy",graphblock->SelectObject))
    PostStress3d(1, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Stress Szz",graphblock->SelectObject))
    PostStress3d(2, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);

    if(0==strcmp("Stress Sxy",graphblock->SelectObject))
    PostStress3d(3, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Stress Syz",graphblock->SelectObject))
    PostStress3d(4, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);
    
    if(0==strcmp("Stress Sxz",graphblock->SelectObject))
    PostStress3d(5, M->nnx, M->nny, M->nnz, 
              M->MatFieldInt, M->MatFieldE, SolBlock->Material,    
              M->la, M->lb, M->lc, M->Ux, M->Uy, M->Uz,
              M->Post1, M->Post2, M->Post3, M->Post4,
              M->Post5, M->Post6, M->Post7, M->Post8);    
}
    
    if (1==graphblock->RotateFlag) 
    {
         if (0!=graphblock->RotateCoor) 
         {
             memcpy(&(M->Post1[0]), &(M->Post1[(graphblock->RotateCoor)*(M->nnx-1)*(M->nny-1)]),(M->nnx-1)*(M->nny-1)*sizeof(double));
             memcpy(&(M->Post2[0]), &(M->Post2[(graphblock->RotateCoor)*(M->nnx-1)*(M->nny-1)]),(M->nnx-1)*(M->nny-1)*sizeof(double));
             memcpy(&(M->Post3[0]), &(M->Post3[(graphblock->RotateCoor)*(M->nnx-1)*(M->nny-1)]),(M->nnx-1)*(M->nny-1)*sizeof(double));
             memcpy(&(M->Post4[0]), &(M->Post4[(graphblock->RotateCoor)*(M->nnx-1)*(M->nny-1)]),(M->nnx-1)*(M->nny-1)*sizeof(double));
         }
         
         graphblock->RotateCoor+=1;
         if ((M->nnz-1)==(graphblock->RotateCoor)) 
         {
              graphblock->NewObjectFlag==1;
              graphblock->RotateCoor=0;
         }
         printf("graphblock->RotateCoor : %i\n",graphblock->RotateCoor);
    }
    
              
/*

    if (1==(graphblock->PostSmoothFlag)) 
    {
        PostSmooth(M->nnx, M->nny, 
        M->Post1, M->Post2, M->Post3, M->Post4);
        printf("\nsmooth\n");
    }
    else printf("\nNONsmooth\n");
*/    
    return;
}

/*------------------------------------------------------------------------------
     ElemMatrix2ElemNodes3d
------------------------------------------------------------------------------*/
int ElemMatrix2ElemNodes3d(int nnx, int nny, int nnz, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j,p;
    size_t FullSize;
      
    FullSize=sizeof(double)*(nnx-1)*(nny-1)*(nnz-1);
    
        memcpy(Post1,M,FullSize);
        memcpy(Post2,M,FullSize);
        memcpy(Post3,M,FullSize);
        memcpy(Post4,M,FullSize);
        memcpy(Post5,M,FullSize);
        memcpy(Post6,M,FullSize);
        memcpy(Post7,M,FullSize);
        memcpy(Post8,M,FullSize);
        
    return(1);
}

/*------------------------------------------------------------------------------
     ElemMatrixInt2ElemNodes
------------------------------------------------------------------------------*/
int ElemMatrixInt2ElemNodes3d(int nnx, int nny, int nnz, int *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j,k,p;
    double tmp;
    size_t FullSize;
      
    FullSize=sizeof(double)*(nnx-1)*(nny-1)*(nnz-1);
    
    for(i=0;i<(nnx-1);i++)
    for(j=0;j<(nny-1);j++)
    for(k=0;k<(nnz-1);k++)
    {
        p=i+j*(nnx-1)+k*(nnx-1)*(nny-1);
        tmp=(double)M[p];
        Post1[p]=tmp;
        Post2[p]=tmp;
        Post3[p]=tmp;
        Post4[p]=tmp;
        Post5[p]=tmp;
        Post6[p]=tmp;
        Post7[p]=tmp;
        Post8[p]=tmp;
        
    }        
    return(1);
}

/*------------------------------------------------------------------------------
     NodeMatrix2ElemNodes3d
------------------------------------------------------------------------------*/
int NodeMatrix2ElemNodes3d(int nnx, int nny, int nnz, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j,k,p;
    int nex,ney;
    size_t nexSize;
    nex=nnx-1;   
    ney=nny-1;   
    nexSize=sizeof(double)*nex;
    
    for (k = 0; k < (nnz-1); k++)                         
    for (j = 0; j < (nny-1); j++) 
    {     
        memcpy(&(Post1[j*nex+k*nex*ney]),&(M[j*nnx+k*nnx*nny])      ,nexSize);
        memcpy(&(Post2[j*nex+k*nex*ney]),&(M[j*nnx+1+k*nnx*nny])    ,nexSize);
        memcpy(&(Post3[j*nex+k*nex*ney]),&(M[(j+1)*nnx+1+k*nnx*nny]),nexSize);
        memcpy(&(Post4[j*nex+k*nex*ney]),&(M[(j+1)*nnx+k*nnx*nny])  ,nexSize);
        memcpy(&(Post5[j*nex+k*nex*ney]),&(M[j*nnx+(k+1)*nnx*nny])      ,nexSize);
        memcpy(&(Post6[j*nex+k*nex*ney]),&(M[j*nnx+1+(k+1)*nnx*nny])    ,nexSize);
        memcpy(&(Post7[j*nex+k*nex*ney]),&(M[(j+1)*nnx+1+(k+1)*nnx*nny]),nexSize);
        memcpy(&(Post8[j*nex+k*nex*ney]),&(M[(j+1)*nnx+(k+1)*nnx*nny])  ,nexSize);
    }
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostStrain3d
------------------------------------------------------------------------------*/
int PostStrain3d(int component, int nnx, int nny, int nnz, 
                    double la, double lb, double lc, 
              double *Ux, double *Uy, double *Uz,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j,k,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex,ney,nez;
    int nnxy,nexy;
    double B1[144],B2[144],B3[144],B4[144],B5[144],B6[144],B7[144],B8[144];
    double U[24];
    double Eps[6];
    
    nex=nnx-1;
    ney=nny-1;
    nez=nnz-1;
    
    nnxy=nnx*nny;
    nexy=nex*ney;
    
    GetBatNode3d(la, lb, lc, 1, B1);
    GetBatNode3d(la, lb, lc, 2, B2);
    GetBatNode3d(la, lb, lc, 3, B3);
    GetBatNode3d(la, lb, lc, 4, B4);
    GetBatNode3d(la, lb, lc, 5, B5);
    GetBatNode3d(la, lb, lc, 6, B6);
    GetBatNode3d(la, lb, lc, 7, B7);
    GetBatNode3d(la, lb, lc, 8, B8);
        
    for (k = 0; k < nez; k++)       
    for (j = 0; j < ney; j++) 
    for (i = 0; i < nex; i++)   
    {
        p=i+j*nnx+(k)*nnxy;
        
        U[0]=Ux[p];
        U[1]=Uy[p];
        U[2]=Uz[p]; p++;
        
        U[3]=Ux[p];
        U[4]=Uy[p];
        U[5]=Uz[p]; p+=nnx;
        
        U[6]=Ux[p];
        U[7]=Uy[p];
        U[8]=Uz[p]; p--;
        
        U[9]=Ux[p];
        U[10]=Uy[p];
        U[11]=Uz[p];      

        p=i+j*nnx+(k+1)*nnxy;  
        
        U[12]=Ux[p];
        U[13]=Uy[p];
        U[14]=Uz[p]; p++;
        
        U[15]=Ux[p];
        U[16]=Uy[p];
        U[17]=Uz[p]; p+=nnx;
        
        U[18]=Ux[p];
        U[19]=Uy[p];
        U[20]=Uz[p]; p--;
        
        U[21]=Ux[p];
        U[22]=Uy[p];
        U[23]=Uz[p];   
        
        p=i+j*nex+k*nexy;        
        MatrixMultMatrix(6, 24, 1, B1, U, Eps); Post1[p]=Eps[component];
        MatrixMultMatrix(6, 24, 1, B2, U, Eps); Post2[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B3, U, Eps); Post3[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B4, U, Eps); Post4[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B5, U, Eps); Post5[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B6, U, Eps); Post6[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B7, U, Eps); Post7[p]=Eps[component]; 
        MatrixMultMatrix(6, 24, 1, B8, U, Eps); Post8[p]=Eps[component];  
    }
    
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostStress3d
------------------------------------------------------------------------------*/
int PostStress3d(int component, int nnx, int nny, int nnz, 
              int *MatFieldInt, double *MatFieldE, MaterialBlock *Material,
                    double la, double lb, double lc, 
              double *Ux, double *Uy, double *Uz,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j,k,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex,ney,nez;
    int nnxy,nexy;
    double B1[144],B2[144],B3[144],B4[144],B5[144],B6[144],B7[144],B8[144];
    double U[24];
    double Eps[6];
    double Sigma[6];
    double C[36];
    double E;
    int LastInt;
    double Mu;
    
    nex=nnx-1;
    ney=nny-1;
    nez=nnz-1;
    
    nnxy=nnx*nny;
    nexy=nex*ney;
    
    GetBatNode3d(la, lb, lc, 1, B1);
    GetBatNode3d(la, lb, lc, 2, B2);
    GetBatNode3d(la, lb, lc, 3, B3);
    GetBatNode3d(la, lb, lc, 4, B4);
    GetBatNode3d(la, lb, lc, 5, B5);
    GetBatNode3d(la, lb, lc, 6, B6);
    GetBatNode3d(la, lb, lc, 7, B7);
    GetBatNode3d(la, lb, lc, 8, B8);
    
    LastInt=0;
    Mu=Material[LastInt].Mu;
    BuildMaterialMatrix3d(Mu, C);
                    
    for (k = 0; k < nez; k++)       
    for (j = 0; j < ney; j++) 
    for (i = 0; i < nex; i++)   
    {
        p=i+j*nnx+(k)*nnxy;
        
        U[0]=Ux[p];
        U[1]=Uy[p];
        U[2]=Uz[p]; p++;
        
        U[3]=Ux[p];
        U[4]=Uy[p];
        U[5]=Uz[p]; p+=nnx;
        
        U[6]=Ux[p];
        U[7]=Uy[p];
        U[8]=Uz[p]; p--;
        
        U[9]=Ux[p];
        U[10]=Uy[p];
        U[11]=Uz[p];      

        p=i+j*nnx+(k+1)*nnxy;  
        
        U[12]=Ux[p];
        U[13]=Uy[p];
        U[14]=Uz[p]; p++;
        
        U[15]=Ux[p];
        U[16]=Uy[p];
        U[17]=Uz[p]; p+=nnx;
        
        U[18]=Ux[p];
        U[19]=Uy[p];
        U[20]=Uz[p]; p--;
        
        U[21]=Ux[p];
        U[22]=Uy[p];
        U[23]=Uz[p];   
        
        
        
        p=i+j*nex+k*nexy;        
        
        if (LastInt!=MatFieldInt[p])
        {
            LastInt=MatFieldInt[p];
            Mu=Material[LastInt].Mu;
            BuildMaterialMatrix3d(Mu, C);
        }
        
        E=MatFieldE[p];
        
        
        MatrixMultMatrix(6, 24, 1, B1, U, Eps); 
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma); 
        Post1[p]=Sigma[component]*E;
        MatrixMultMatrix(6, 24, 1, B2, U, Eps); 
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma);
        Post2[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B3, U, Eps); 
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma);
        Post3[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B4, U, Eps); 
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma);
        Post4[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B5, U, Eps);
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma); 
        Post5[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B6, U, Eps);
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma); 
        Post6[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B7, U, Eps);
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma); 
        Post7[p]=Sigma[component]*E; 
        MatrixMultMatrix(6, 24, 1, B8, U, Eps);
        MatrixMultMatrix(6, 6, 1, C, Eps, Sigma); 
        Post8[p]=Sigma[component]*E;  
    }
    
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostSmooth3d
------------------------------------------------------------------------------*/
int PostSmooth3d(int nnx, int nny, int nnz, 
                 double *Post1, double *Post2, double *Post3, double *Post4,
                 double *Post5, double *Post6, double *Post7, double *Post8)
{
    int i,j;
    int nex;
    double tmp;
       
    nex=nnx-1;       
              
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-2); i++)   
        {
            tmp=(Post2[i+j*nex]    + Post1[i+1+j*nex])*0.5;
            Post2[i+j*nex]  =tmp;
            Post1[i+1+j*nex]=tmp;  
            
            tmp=(Post3[i+j*nex]    + Post4[i+1+j*nex])*0.5;
            Post3[i+j*nex]  =tmp;
            Post4[i+1+j*nex]=tmp;  
        }
    }
    
    for (j = 0; j < (nny-2); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            tmp=(Post4[i+j*nex]    + Post1[i+(j+1)*nex])*0.5;
            Post4[i+j*nex]   = tmp;
            Post1[i+(j+1)*nex]=tmp;
            
            tmp=(Post3[i+j*nex]    + Post2[i+(j+1)*nex])*0.5;
            Post3[i+j*nex]   = tmp;
            Post2[i+(j+1)*nex]=tmp;  
        }
    }
    
                  	    
    return(1);


}

