/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyMainWindowUI.h"
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>

/*---------------------------------------------------------
	BuildMaterialMatrix
----------------------------------------------------------*/

void BuildMaterialMatrix3d(double mu, double *C)
{

   double k;
   int i;
   
   k=(1.0-mu)/((1.0+mu)*(1.0-2.0*mu));
   
   C[0]=1.0;
   C[1]=mu/(1.0-mu);
   C[2]=mu/(1.0-mu);
   C[3]=0.0;
   C[4]=0.0;
   C[5]=0.0;
   
   C[6]=mu/(1.0-mu);
   C[7]=1.0;
   C[8]=mu/(1.0-mu);
   C[9]=0.0;
   C[10]=0.0;
   C[11]=0.0;
   
   C[12]=mu/(1.0-mu);
   C[13]=mu/(1.0-mu);
   C[14]=1.0;
   C[15]=0.0;
   C[16]=0.0;
   C[17]=0.0;
   
   C[18]=0.0;
   C[19]=0.0;
   C[20]=0.0;
   C[21]=(1.0-2.0*mu)/(2.0*(1.0-mu));
   C[22]=0.0;
   C[23]=0.0;

   C[24]=0.0;
   C[25]=0.0;
   C[26]=0.0;
   C[27]=0.0;
   C[28]=(1.0-2.0*mu)/(2.0*(1.0-mu));
   C[29]=0.0;

   C[30]=0.0;
   C[31]=0.0;
   C[32]=0.0;
   C[33]=0.0;
   C[34]=0.0;
   C[35]=(1.0-2.0*mu)/(2.0*(1.0-mu));
   
   for(i=0;i<36;i++)
   C[i]*=k;

   return;    
}

/*---------------------------------------------------------
	GetBatXYZ3d
----------------------------------------------------------*/
 
void GetBatXYZ3d(double a, double b, double c, 
                 double x, double y, double z, double *BMatrix)
{
int i;
for(i=0; i<144; i++) BMatrix[i]=0.0;

BMatrix[0+6*0] = -0.10e1 / a + y / a / b + z / a / c - y * z / a / b / c;
BMatrix[0+6*3] = 0.10e1 / a - y / a / b - z / a / c + y * z / a / b / c;
BMatrix[0+6*6] = y / a / b - y * z / a / b / c;
BMatrix[0+6*9] = -y / a / b + y * z / a / b / c;
BMatrix[0+6*12] = -z / a / c + y * z / a / b / c;
BMatrix[0+6*15] = z / a / c - y * z / a / b / c;
BMatrix[0+6*18] = y * z / a / b / c;
BMatrix[0+6*21] = -y * z / a / b / c;
BMatrix[1+6*1] = -0.10e1 / b + x / a / b + z / b / c - x * z / a / b / c;
BMatrix[1+6*4] = -x / a / b + x * z / a / b / c;
BMatrix[1+6*7] = x / a / b - x * z / a / b / c;
BMatrix[1+6*10] = 0.10e1 / b - x / a / b - z / b / c + x * z / a / b / c;
BMatrix[1+6*13] = -z / b / c + x * z / a / b / c;
BMatrix[1+6*16] = -x * z / a / b / c;
BMatrix[1+6*19] = x * z / a / b / c;
BMatrix[1+6*22] = z / b / c - x * z / a / b / c;
BMatrix[2+6*2] = -0.10e1 / c + x / a / c + y / b / c - x * y / a / b / c;
BMatrix[2+6*5] = -x / a / c + x * y / a / b / c;
BMatrix[2+6*8] = -x * y / a / b / c;
BMatrix[2+6*11] = -y / b / c + x * y / a / b / c;
BMatrix[2+6*14] = 0.10e1 / c - x / a / c - y / b / c + x * y / a / b / c;
BMatrix[2+6*17] = x / a / c - x * y / a / b / c;
BMatrix[2+6*20] = x * y / a / b / c;
BMatrix[2+6*23] = y / b / c - x * y / a / b / c;
BMatrix[3+6*0] = -0.10e1 / b + x / a / b + z / b / c - x * z / a / b / c;
BMatrix[3+6*1] = -0.10e1 / a + y / a / b + z / a / c - y * z / a / b / c;
BMatrix[3+6*3] = -x / a / b + x * z / a / b / c;
BMatrix[3+6*4] = 0.10e1 / a - y / a / b - z / a / c + y * z / a / b / c;
BMatrix[3+6*6] = x / a / b - x * z / a / b / c;
BMatrix[3+6*7] = y / a / b - y * z / a / b / c;
BMatrix[3+6*9] = 0.10e1 / b - x / a / b - z / b / c + x * z / a / b / c;
BMatrix[3+6*10] = -y / a / b + y * z / a / b / c;
BMatrix[3+6*12] = -z / b / c + x * z / a / b / c;
BMatrix[3+6*13] = -z / a / c + y * z / a / b / c;
BMatrix[3+6*15] = -x * z / a / b / c;
BMatrix[3+6*16] = z / a / c - y * z / a / b / c;
BMatrix[3+6*18] = x * z / a / b / c;
BMatrix[3+6*19] = y * z / a / b / c;
BMatrix[3+6*21] = z / b / c - x * z / a / b / c;
BMatrix[3+6*22] = -y * z / a / b / c;
BMatrix[4+6*1] = -0.10e1 / c + x / a / c + y / b / c - x * y / a / b / c;
BMatrix[4+6*2] = -0.10e1 / b + x / a / b + z / b / c - x * z / a / b / c;
BMatrix[4+6*4] = -x / a / c + x * y / a / b / c;
BMatrix[4+6*5] = -x / a / b + x * z / a / b / c;
BMatrix[4+6*7] = -x * y / a / b / c;
BMatrix[4+6*8] = x / a / b - x * z / a / b / c;
BMatrix[4+6*10] = -y / b / c + x * y / a / b / c;
BMatrix[4+6*11] = 0.10e1 / b - x / a / b - z / b / c + x * z / a / b / c;
BMatrix[4+6*13] = 0.10e1 / c - x / a / c - y / b / c + x * y / a / b / c;
BMatrix[4+6*14] = -z / b / c + x * z / a / b / c;
BMatrix[4+6*16] = x / a / c - x * y / a / b / c;
BMatrix[4+6*17] = -x * z / a / b / c;
BMatrix[4+6*19] = x * y / a / b / c;
BMatrix[4+6*20] = x * z / a / b / c;
BMatrix[4+6*22] = y / b / c - x * y / a / b / c;
BMatrix[4+6*23] = z / b / c - x * z / a / b / c;
BMatrix[5+6*0] = -0.10e1 / c + x / a / c + y / b / c - x * y / a / b / c;
BMatrix[5+6*2] = -0.10e1 / a + y / a / b + z / a / c - y * z / a / b / c;
BMatrix[5+6*3] = -x / a / c + x * y / a / b / c;
BMatrix[5+6*5] = 0.10e1 / a - y / a / b - z / a / c + y * z / a / b / c;
BMatrix[5+6*6] = -x * y / a / b / c;
BMatrix[5+6*8] = y / a / b - y * z / a / b / c;
BMatrix[5+6*9] = -y / b / c + x * y / a / b / c;
BMatrix[5+6*11] = -y / a / b + y * z / a / b / c;
BMatrix[5+6*12] = 0.10e1 / c - x / a / c - y / b / c + x * y / a / b / c;
BMatrix[5+6*14] = -z / a / c + y * z / a / b / c;
BMatrix[5+6*15] = x / a / c - x * y / a / b / c;
BMatrix[5+6*17] = z / a / c - y * z / a / b / c;
BMatrix[5+6*18] = x * y / a / b / c;
BMatrix[5+6*20] = y * z / a / b / c;
BMatrix[5+6*21] = y / b / c - x * y / a / b / c;
BMatrix[5+6*23] = -y * z / a / b / c;


   return;    
} 

/*---------------------------------------------------------
	GetBatNode3d
----------------------------------------------------------*/
 
void GetBatNode3d(double a, double b, double c, 
                  int node, double *BMatrix)
{
   double x,y,z;
   
   x=a; y=b; z=c;
   
   switch (node)
   {
       case 1: x=0.0; y=0.0; z=0.0;    break;
       case 2:        y=0.0; z=0.0;    break;
       case 3:               z=0.0;    break;
       case 4: x=0.0;        z=0.0;    break;
       case 5: x=0.0; y=0.0;           break;
       case 6:        y=0.0;           break;
       case 7:                         break;
       case 8: x=0.0;                  break;
   }
   
   GetBatXYZ3d(a, b, c, x, y, z, BMatrix);
   
   
   return;
}

/*---------------------------------------------------------
	GetStiffnessMatrix3d
----------------------------------------------------------*/

void GetStiffnessMatrix3d(double mu, double a, double b, double c, double **KMatrix)
{
   double C[36];
   double B[144];
   double BT[144];
   double BTC[144];
   double Ktmp[576];
   double *K;
   double ip[2];
   int ix,iy,iz;
   int i;
   
   *KMatrix = (double *) malloc(sizeof(double)*(24*24));
   K=(*KMatrix);
   
   for(i=0; i<576; i++)
   K[i]=0.0;
   
   BuildMaterialMatrix3d(mu, C);
   
   ip[0]=0.211324865405;
   ip[1]=0.788675134595;
   
   for(ix=0; ix<2; ix++)
   for(iy=0; iy<2; iy++)
   for(iz=0; iz<2; iz++)
   {
       GetBatXYZ3d(a, b, c, (ip[ix]*a), (ip[iy]*b), (ip[iz]*c), B);
       TransposeMatrix(6,24,B,BT);
       MatrixMultMatrix(24,6,6,BT,C,BTC);
       MatrixMultMatrix(24,6,24,BTC,B,Ktmp);
       AddMatrix(24,24, K, Ktmp);            
   }
   return;   
}

