/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>

/*------------------------------------------------------------------------------
	BCMain
------------------------------------------------------------------------------*/
void BCmain(int level)
{

    AllWinStruct  *AllWinData;
    LoadWinStruct *LoadWinData;
    SolveBlock    *SolBlock;
    
    int     num_levels;
    MGBlock *Meshes;   
    int     *BcTypeX, *BcTypeY;
    double  *BcValueX, *BcValueY;
    
    int     num_loads;
    OneLoad *LoadList;
    int i,j,k;
    int nnx,nny;
    //int PF, NF;
    double la,lb, lineload;
    double w1, w2, w3, dj;
    double k1, k2;
    
    int    TL;
    double Tval;
    double Texp;  
    double Tleft,Tright;  
       
    /*--------- end of declaration--------- */
    
    PutGetAllWinStruct(1, &AllWinData); 
    LoadWinData=AllWinData->LoadWinData;
    PutGetSolveBlock(1,&SolBlock); 
    
    num_loads=LoadWinData->num_loads;
    LoadList=LoadWinData->LoadList;
    
    num_levels=SolBlock->num_levels;
    Meshes=&(SolBlock->Meshes[level]);
    
    BcTypeX=Meshes->BcTypeX;
    BcTypeY=Meshes->BcTypeY;
    BcValueX=Meshes->BcValueX;
    BcValueY=Meshes->BcValueY;
    
    nnx=Meshes->nnx;
    nny=Meshes->nny;
    la=Meshes->la;
    lb=Meshes->lb;
    
    //printf("\n(NormalMesh-BC) nnx: %i" ,nnx);
    
    // Reset all Values to Zero
    
    k=nny*nnx; 
    for(i=0; i<k; i++)
    {
        BcTypeX[i]=0;
        BcTypeY[i]=0; 
        BcValueX[i]=0.0;
        BcValueY[i]=0.0;        
    }
    
         /*      Boundary scheme       PrimFlag: 0:edge, 1:corner
                       2               NumFlag:  0-3 side number
                 3------------2        PhyFlag:  0:Ux  1:Uy  2:Fx  3:Fy
                 |            |        BcVal:    value of B.C.
              3  |            | 1
                 |            | 
                 0------------1
                       0
        */
    //if (level==num_levels-1) PrintLoadlist(LoadList,num_loads);
    
    for(i=0; i<num_loads; i++)
    {
        Tval=(LoadList[i].BCVal);
        Texp=(LoadList[i].BCPot);
               
        if (1==(LoadList[i].PrimFlag)) { TL=1; } 
        else 
        {
            switch (LoadList[i].NumFlag)
            {
                case 0: TL=nnx; break;
                case 1: TL=nny; break;
                case 2: TL=nnx; break;
                case 3: TL=nny; break;
            }
        }
                
        for(j=0; j<TL; j++)
        {
            switch (LoadList[i].NumFlag)
            {
                case 0: k=j;                     lineload=la; break;
                case 1: k=(nnx-1)+nnx*j;         lineload=lb; break;
                case 2: k=(nnx-1)-j+(nny-1)*nnx; lineload=la; break;
                case 3: k=0+(nny-1-j)*nnx;       lineload=lb; break;
            }
            
            switch (LoadList[i].PhyFlag)
            {
                case 0: BcTypeX[k]=1;  // Ux
                        BcValueX[k]+=Tval*pow(((double)(j)/(double)(TL-1)),Texp);
                        if (1==(LoadList[i].PrimFlag)) BcValueX[k]=Tval;
                        /*printf("LOAD: %i, %g, %g\n",LoadList[i].NumFlag,Texp,BcValueX[k]);*/ 
                        break;
                             
                case 1: BcTypeY[k]=1;  // Uy
                        BcValueY[k]+=Tval*pow(((double)(j)/(double)(TL-1)),Texp);
                        if (1==(LoadList[i].PrimFlag)) BcValueY[k]=Tval;
                        break;
                             
                case 2: if (BcTypeX[k]==1) break;   // Fx
                        BcTypeX[k]=0;
                        if (1==LoadList[i].PrimFlag) {BcValueX[k]=Tval; break;}
                        
                        if (0.0!=Tval)
                        {
                         dj=(double)j;
                         
                         k1=pow((lineload * dj),(Texp+2.0)) - pow((lineload * (dj-1.0)),(Texp+2.0));
                         k2=pow((lineload * dj),(Texp+1.0)) - pow((lineload * (dj-1.0)),(Texp+1.0));
                         w1=Tval/pow((lineload*((double)(TL-1))),Texp);
                         w2=1.0/(lineload*(Texp+2.0));
                         w3=(1.0-dj)/(Texp+1.0);
                         
                         Tleft = w1*(w2*k1+w3*k2);
                         
                         k1=pow((lineload * (dj+1.0)),(Texp+2.0)) - pow((lineload * dj),(Texp+2.0));
                         k2=pow((lineload * (dj+1.0)),(Texp+1.0)) - pow((lineload * dj),(Texp+1.0));
                         w2*=-1;
                         w3=(1.0+dj)/(Texp+1.0);
                         
                         Tright = w1*(w2*k1+w3*k2);
                       
                        
                         if(0==j)     BcValueX[k]+=Tright; 
                         if((TL-1)==j) BcValueX[k]+=Tleft; 
                         if(0<j && j<(TL-1)) BcValueX[k]+=(Tleft+Tright);
                        }
                        else {BcValueX[k]=0;}                
                        break;     
                
                case 3: if (BcTypeY[k]==1) break;   // Fy  
                        BcTypeY[k]=0; 
                        if (1==LoadList[i].PrimFlag) {BcValueY[k]=Tval; break;}
                        
                        if (0.0!=Tval)
                        {
                         dj=(double)j;
                         
                         k1=pow((lineload * dj),(Texp+2.0)) - pow((lineload * (dj-1.0)),(Texp+2.0));
                         k2=pow((lineload * dj),(Texp+1.0)) - pow((lineload * (dj-1.0)),(Texp+1.0));
                         w1=Tval/pow((lineload*((double)(TL-1))),Texp);
                         w2=1.0/(lineload*(Texp+2.0));
                         w3=(1.0-dj)/(Texp+1.0);
                         
                         Tleft = w1*(w2*k1+w3*k2);
                         
                         k1=pow((lineload * (dj+1.0)),(Texp+2.0)) - pow((lineload * dj),(Texp+2.0));
                         k2=pow((lineload * (dj+1.0)),(Texp+1.0)) - pow((lineload * dj),(Texp+1.0));
                         w2*=-1;
                         w3=(1.0+dj)/(Texp+1.0);
                         
                         Tright = w1*(w2*k1+w3*k2);
                       
                         if(0==j)     BcValueY[k]+=Tright; 
                         if((TL-1)==j) BcValueY[k]+=Tleft; 
                         if(0<j && j<(TL-1)) BcValueY[k]+=(Tleft+Tright);
                        }
                        else {BcValueY[k]=0;}
                        break;   
            }
        }
    } 
    
        
    k=(nnx-1)+(nny-1)*nnx;
    for(i=0; i<=k; i++)
    {
        if(1==BcTypeX[i]) Meshes->Ux[i]=BcValueX[i];  
        if(1==BcTypeY[i]) Meshes->Uy[i]=BcValueY[i];          
    }
    
    /*
    
    printf("\nBcTypeX\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%i ",BcTypeX[i+j*nnx]);
        
        }
        printf("\n");
    }
    
    printf("\nBcValueX\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%g ",BcValueX[i+j*nnx]);
        
        }
        printf("\n");
    } 
    getchar();    
    
    printf("\nBcTypeY\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%i ",BcTypeY[i+j*nnx]);
        
        }
        printf("\n");
    }                 
    
    printf("\nBcValueY\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%g ",BcValueY[i+j*nnx]);
        
        }
        printf("\n");
    }
    getchar();
    // */                        
                                                                                     
    return;
}


/*------------------------------------------------------------------------------
	SplineBCMain  //erste Spline RB-Routine ; behandelt einige RB falsch
------------------------------------------------------------------------------*/
void SplineBCmain()
{

    AllWinStruct  *AllWinData;
    LoadWinStruct *LoadWinData;
    SolveBlock    *SolBlock;
    
    int     num_levels;
    MGBlock *Meshes;   
    int     *BcTypeX, *BcTypeY;
    double  *BcValueX, *BcValueY;
    
    int     num_loads;
    OneLoad *LoadList;
    int i,j,k;
    int nnx,nny;
    double la,lb, lineload;
    
    int    TL;
    double Tval;
    double Texp;  
    double Tleft,Tright;  
       
    /*--------- end of declaration--------- */
    


    PutGetAllWinStruct(1, &AllWinData); 
    LoadWinData=AllWinData->LoadWinData;
    PutGetSolveBlock(1,&SolBlock); 
 
 
    num_loads=LoadWinData->num_loads;
    LoadList=LoadWinData->LoadList;
 
    num_levels=SolBlock->num_levels;
    Meshes=(SolBlock->SplineMesh);

    BcTypeX=Meshes->BcTypeX;
    BcTypeY=Meshes->BcTypeY;

    BcValueX=Meshes->BcValueX;
    BcValueY=Meshes->BcValueY;

    nnx=Meshes->nnx;printf("nnx: %i" ,nnx);
    nny=Meshes->nny;
    la=Meshes->la;
    lb=Meshes->lb;
   
    k=(nnx-1)+(nny-1)*nnx;
    for(i=0; i<=k; i++)
    {
        BcTypeX[i]=0;
        BcTypeY[i]=0; 
        BcValueX[i]=0.0;
        BcValueY[i]=0.0;        
    } 
    
         /*      Boundary scheme       PrimFlag: 0:edge, 1:corner
                       2               NumFlag:  0-3 side number
                 -------------         PhyFlag:  0:Ux  1:Uy  2:Fx  3:Fy
                 |            |        BcVal:    value of B.C.
              3  |            | 1
                 |            | 
                 -------------
                       0
        */
    

    for(i=0; i<num_loads; i++)
    {
               
        if (1==(LoadList[i].PrimFlag)) { TL=1; } 
        else 
        {
            switch (LoadList[i].NumFlag)
            {
                case 0: TL=nnx; break;
                case 1: TL=nny; break;
                case 2: TL=nnx; break;
                case 3: TL=nny; break;
            }
        }
                
        for(j=0; j<TL; j++)
        {
            switch (LoadList[i].NumFlag)
            {
                case 0: k=j;                     lineload=la; break;
                case 1: k=(nnx-1)+nnx*j;         lineload=lb; break;
                case 2: k=(nnx-1)-j+(nny-1)*nnx; lineload=la; break;
                case 3: k=0+(nny-1-j)*nnx;       lineload=lb; break;
            }
            
            Tval=(LoadList[i].BCVal);
            Texp=(LoadList[i].BCPot);
           
            /*if ( 
                   (TL!=1) 
                && (LoadList[i].PhyFlag>1) 
                && (    (0==k)
                    ||  ((nnx-1)==k)
                    ||  ((nnx*(nny-1))==k)
                    ||  (((nnx-1)+nnx*(nny-1))==k)
                   )
                )              Tval*=0.5;*/
            
            switch (LoadList[i].PhyFlag)
            {
                case 0: BcTypeX[k]=1; BcValueX[k]+=Tval*pow(((double)(j)/(double)(TL-1)),Texp);
                        if (1==(LoadList[i].PrimFlag)) BcValueX[k]=Tval; /*printf("LOAD: %i, %g, %g\n",LoadList[i].NumFlag,Texp,BcValueX[k]);*/ break;   // Ux  
                case 1: BcTypeY[k]=1; BcValueY[k]+=Tval*pow(((double)(j)/(double)(TL-1)),Texp);
                        if (1==(LoadList[i].PrimFlag)) BcValueY[k]=Tval; break;   // Uy  
                case 2: BcTypeX[k]=0; 
                        
                        Tval/=pow( (lineload*((double)(TL-1))) , Texp);
                        
                        if (0.0!=Tval)
                        {
                        Tleft = Tval*
                              (
                                (   
                                  (   ( 1.0 - ((double)j) ) 
                                    *  pow( lineload * ((double)j)  , (Texp+1.0) ) 
                                    /  (Texp+1.0)
                                   )
                                  +
                                  (     pow( lineload * ((double)j)  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )  
                                 )
                                 -
                                 (
                                   (   ( 1.0 - ((double)j) )
                                    *   pow( lineload * ((double)(j-1))  , (Texp+1.0) ) 
                                    /   (Texp+1.0)
                                   ) 
                                   +
                                   (     pow( lineload * ((double)(j-1))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )
                                  )
                                 );
                    Tright = Tval*
                              (
                                (   
                                  (   ( 1.0 + ((double)j) ) 
                                    *  pow( lineload * ((double)(j+1))  , (Texp+1.0) ) 
                                    /  (Texp+1.0)
                                   )
                                  -
                                  (     pow( lineload * ((double)(j+1))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )  
                                 )
                                 -
                                 (
                                   (   ( 1.0 + ((double)j) )
                                    *   pow( lineload * ((double)(j))  , (Texp+1.0) ) 
                                    /   (Texp+1.0)
                                   ) 
                                   -
                                   (     pow( lineload * ((double)(j))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )
                                  )
                                 );
                       
                        //nderung nach Kessel
                       // if(0==j)      BcValueX[k]+=Tleft; 
                       // if((TL-1)==j) BcValueX[k]+=Tright; 
                        if(0==j)      BcValueX[k]+=Tright; 
                        if((TL-1)==j) BcValueX[k]+=Tleft; 
                        
                        if(0<j && j<(TL-1)) BcValueX[k]+=(Tleft+Tright);
                        }
                        else
                        {
                        BcValueX[k]=0;    
                        }
                        //printf("Tval  %g Texp  %g  Tleft %g  Tright %g\n",Tval,Texp,Tleft,Tright);               
                        //printf("k: %i  BcValueX: %g\n",k,BcValueX[k]);                
                        break;   // Fx  
                
                case 3: BcTypeY[k]=0; 
                        
                        Tval/=pow( (lineload*((double)(TL-1))) , Texp);
                        
                        if (0.0!=Tval)
                        {
                        Tleft = Tval*
                              (
                                (   
                                  (   ( 1.0 - ((double)j) ) 
                                    *  pow( lineload * ((double)j)  , (Texp+1.0) ) 
                                    /  (Texp+1.0)
                                   )
                                  +
                                  (     pow( lineload * ((double)j)  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )  
                                 )
                                 -
                                 (
                                   (   ( 1.0 - ((double)j) )
                                    *   pow( lineload * ((double)(j-1))  , (Texp+1.0) ) 
                                    /   (Texp+1.0)
                                   ) 
                                   +
                                   (     pow( lineload * ((double)(j-1))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )
                                  )
                                 );
                    Tright = Tval*
                              (
                                (   
                                  (   ( 1.0 + ((double)j) ) 
                                    *  pow( lineload * ((double)(j+1))  , (Texp+1.0) ) 
                                    /  (Texp+1.0)
                                   )
                                  -
                                  (     pow( lineload * ((double)(j+1))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )  
                                 )
                                 -
                                 (
                                   (   ( 1.0 + ((double)j) )
                                    *   pow( lineload * ((double)(j))  , (Texp+1.0) ) 
                                    /   (Texp+1.0)
                                   ) 
                                   -
                                   (     pow( lineload * ((double)(j))  , (Texp+2.0) )
                                     /  (Texp+2.0)
                                     /   lineload
                                   )
                                  )
                                 );
                        // nderung nach Kessel
                        //if(0==j)      BcValueY[k]+=Tleft; 
                        //if((TL-1)==j)  BcValueY[k]+=Tright; 
                        if(0==j)       BcValueY[k]+=Tright; 
                        if((TL-1)==j)  BcValueY[k]+=Tleft; 
                        
                        if(0<j && j<(TL-1)) BcValueY[k]+=(Tleft+Tright);
                        }
                        else
                        {
                        BcValueY[k]=0;    
                        }
                        
                        break;   // Fy  
            }
        }
    } 
    
        
    k=(nnx-1)+(nny-1)*nnx;
    for(i=0; i<=k; i++)
    {
        if(1==BcTypeX[i]) Meshes->Ux[i]=BcValueX[i];  
        if(1==BcTypeY[i]) Meshes->Uy[i]=BcValueY[i];          
    }
    /*
    printf("\nBcTypeX\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%i ",BcTypeX[i+j*nnx]);
        
        }
        printf("\n");
    }
    
    printf("\nBcTypeY\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%i ",BcTypeY[i+j*nnx]);
        
        }
        printf("\n");
    }
    
    printf("\nBcValueX\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%g ",BcValueX[i+j*nnx]);
        
        }
        printf("\n");
    }                  
    
    printf("\nBcValueY\n");
    for (j=nny-1; j>=0; j--)
    {
        for(i=0; i<nnx; i++)
        {
            printf("%g ",BcValueY[i+j*nnx]);
        
        }
        printf("\n");
    }
    */                           
                                                                                     
    return;
}


/*------------------------------------------------------------------------------
	SplineBCmain_K   !neu -n32-!
------------------------------------------------------------------------------*/
void SplineBCmain_K()
{

    AllWinStruct  *AllWinData;
    LoadWinStruct *LoadWinData;
    SolveBlock    *SolBlock;
    BSplineBlock  *BSP;
    SomeElements  *SpElemsX;
    SomeElements  *SpElemsY;
    SomeElements  *SpElems;
    SomeElements  *SpElemsN;
    MGBlock       *NormalMesh;
    MGBlock       *SplineMesh;
    OneLoad       *LoadList;
       
    int     *BcTypeX,   *BcTypeY;
    int     *SpBcTypeX, *SpBcTypeY;
    int     *DispType;
    double  *BcValueX,  *BcValueY;
    double  *SpBcValueX,*SpBcValueY;
        
    int     num_levels;    
    int     num_loads;
    int     num_disp;
    int     numcoeff;
    int     numpoly, numpolyN;
    int     center, centerN;
    int     orderK;
    int     numXi;
    
    int     i, j, k, ks, ii, jj, kk;
    int     nnx, nny, nex, ney;
    int     Spnnx, Spnny, Spnex, Spney;
    int     SpTL, TL, SpTLe, TLe;
    int     EType, ETypeN, elmt, iw1;
    double  la, lb, Spla, Splb;
    double  len, Nlen, Splen;
    double  w1, w2, w3, w4, xwert;
    double  *A,*Su,*Nu;  
    
    int    PrF, NuF, PhF; 
    double Tval;
    double Texp;
       
    /*--------- end of declaration--------- */
    
    PutGetAllWinStruct(1, &AllWinData); 
    LoadWinData=AllWinData->LoadWinData;
    SortLoads(LoadWinData);
    PutGetSolveBlock(1,&SolBlock); 
    
    num_loads=LoadWinData->num_loads;
    num_disp=LoadWinData->num_disp;
    LoadList=LoadWinData->LoadList;  
    DispType=LoadWinData->DispType;
    
    num_levels=SolBlock->num_levels;
    NormalMesh=&(SolBlock->Meshes[num_levels-1]);  //spter ndern
    SplineMesh=(SolBlock->SplineMesh);
    
    BcTypeX=NormalMesh->BcTypeX;
    BcTypeY=NormalMesh->BcTypeY;
    BcValueX=NormalMesh->BcValueX;
    BcValueY=NormalMesh->BcValueY;
    
    SpBcTypeX=SplineMesh->BcTypeX;
    SpBcTypeY=SplineMesh->BcTypeY;
    SpBcValueX=SplineMesh->BcValueX;
    SpBcValueY=SplineMesh->BcValueY;

    nnx=NormalMesh->nnx; printf("\n(Splines-BC) normalMesh nnx: %i" ,nnx);
    nny=NormalMesh->nny; printf("\n(Splines-BC) normalMesh nny: %i" ,nny);
    nex=NormalMesh->nex;
    ney=NormalMesh->ney;
        
    Spnnx=SplineMesh->nnx; printf("\n(Splines-BC) SplineMesh nnx: %i" ,Spnnx);
    Spnny=SplineMesh->nny; printf("\n(Splines-BC) SplineMesh nny: %i" ,Spnny);
    
    BSP=&(SolBlock->BSplines);   
    SpElemsX=&(BSP->SplineElemsX); 
    SpElemsY=&(BSP->SplineElemsY);
    orderK=BSP->orderK;
        
    la=NormalMesh->la;
    lb=NormalMesh->lb;
    Spla=SplineMesh->la;
    Splb=SplineMesh->lb;
        
    // Reset all Values to Zero
    
    ks=Spnny*Spnnx;
    for(i=0; i<ks; i++)
    {
        SpBcTypeX[i]=0;
        SpBcTypeY[i]=0; 
        SpBcValueX[i]=0.0;
        SpBcValueY[i]=0.0;        
    }     
    
    
    /*      Boundary scheme        PrimFlag: 0:edge, 1:corner
                   2               NumFlag:  0-3 side number
             -------------         PhyFlag:  0:Ux  1:Uy  2:Fx  3:Fy
             |            |        BcVal:    value of B.C.
          3  |            | 1
             |            | 
             -------------
                   0
    */
    
    //PrintLoadlist(LoadList,num_loads);
    
    //Aufbringung von Verschiebung-RB an den Seiten
    
    for(i=0; i<8; i++) // Schleife ber Kanten x-Richtung i=0..3 y-Richtung i=4..7
    {
        if ( floor(((double)i)/2.0) == ceil(((double)i)/2.0) ) ii=0;
        else ii=1;
        
        if (i > 3) jj=1; else jj=0;
        
        switch (ii)              
        {
         case 0: SpTL=Spnnx; break;
         case 1: SpTL=Spnny; break;
        }
        
        // Verschiebung in x-Richtung i=0..3  in y-Richtung i=4..7
        
        if (DispType[i] > 0)  // Verschiebung an dieser Kante vorhanden
        {
            InitVector_double(SpTL,&Nu);
        
            for(j=0; j<num_disp; j++) // Schleife ber Lastnummern nur Verschiebung
            {
                PrF=(LoadList[j].PrimFlag);
                NuF=(LoadList[j].NumFlag);
                PhF=(LoadList[j].PhyFlag);
                Tval=(LoadList[j].BCVal);
                Texp=(LoadList[j].BCPot);
                
                if (0==PrF && (i-jj*4)==NuF && jj==PhF) // nur bei passender Kante und passender Verschiebung Schleife ausfhren
                for(ks=0; ks<SpTL; ks++)        // Schleife ber Knoten
                {
                  if (0.0!=Tval) Nu[ks]+=Tval*pow(((double)(ks)/(double)(SpTL-1)),Texp);
                }
            }
         
            if (DispType[i]==2) // Verschiebung ungleich Null an dieser Kante
            {
                InitVector_double(SpTL,&Su);
                InitVector_double(SpTL*SpTL,&A);
                
                // Ermittlung von A
                
                switch (ii)              
                {
                 case 0: TLe=nex; len=la; Splen=Spla; SpElems=SpElemsX; break;
                 case 1: TLe=ney; len=lb; Splen=Splb; SpElems=SpElemsY; break;
                }
                                
                numpoly = (SpElems->numpoly);
                center  = (SpElems->center);                
                
                for(j=0; j<SpTL; j++)     // Schleife ber Knoten
                {
                 w1=(double)j*Splen;
                 w2=w1/len;
                 
                 if (floor(w2)==w2 && w2 > 0) elmt=(int)floor(w2)-1;
                 else elmt=(int)floor(w2);
                 
                 xwert=w1-(double)elmt*len;
                 if (elmt > TLe-1) { elmt=TLe-1; xwert=len; }
                 
                 if (elmt <= center) EType=elmt-center;
                 else
                 {
                  iw1=nex-1-center;
                  if (elmt >= iw1) EType=elmt-iw1;
                  else EType=0;   
                 }
                 
                 for (k=0; k<numpoly; k++)   // Schleife ber Polynome
                 {
                  ks=j+SpTL*(k+elmt);
                  A[ks]=PolyVal(SpElems->Element[EType+center].Polynom[k],xwert);
                 }
                }
                //printf("\n\n Matrix A: \n");  PrintMatrixT(SpTL,SpTL,A); getchar();
                  
                kk=GaussSeidelGLSloeser(A,Su,Nu,SpTL);  // lse GLS
                
                if (kk > 0) // GLSloeser hat keine Lsung gefunden
                {
                 printf("\n\n ein LF am Rand %i konnte nicht bercksichtigt werden !\n",(i-jj*4));
                 printf(" < ENTER > \n"); getchar();
                 SetVectorToZero_double(SpTL,Su);
                }
                
                free (A);
                free (Nu);
                Nu=Su;
                Su=NULL;
            }
            
            for(j=0; j<SpTL; j++)     // Schleife ber Knoten
            {            
                switch (i-jj*4)
                {
                    case 0: ks=j;                           break;
                    case 1: ks=(Spnnx-1)+Spnnx*j;           break;
                    case 2: ks=(Spnnx-1)-j+(Spnny-1)*Spnnx; break;
                    case 3: ks=0+(Spnny-1-j)*Spnnx;         break;
                }
                
                switch (jj)
                {
                    case 0: SpBcTypeX[ks]=1;
                            SpBcValueX[ks]+=Nu[j];
                            SplineMesh->Ux[ks]+=Nu[j];       break;
                    case 1: SpBcTypeY[ks]=1;
                            SpBcValueY[ks]+=Nu[j];
                            SplineMesh->Uy[ks]+=Nu[j];       break;                                           
                }
                
            }
            //  printf("\n\n Durchgang %i /Vektor Nu: \n",i);  PrintMatrixT(SpTL,1,Nu); getchar();
            free (Nu);
        }
                 
    } 
    
    // Aufbringung der Verschiebungen an den Ecken
    
    for(j=0; j<num_disp; j++) // Schleife ber Lastnummern nur Verschiebung
    {
     if (1==LoadList[j].PrimFlag) // Ecke
     {
      switch (LoadList[j].NumFlag) //Eckennummer
      {
          case 0: ks=0;               break;
          case 1: ks=Spnnx-1;         break;
          case 2: ks=Spnny*Spnnx-1;   break;
          case 3: ks=(Spnny-1)*Spnnx; break;
      } 
      
      switch (LoadList[j].PhyFlag) // ux oder uy
      {
          case 0: SpBcTypeX[ks]=1;
                  SpBcValueX[ks]=LoadList[j].BCVal;
                  SplineMesh->Ux[ks]=LoadList[j].BCVal; break;
          case 1: SpBcTypeY[ks]=1;
                  SpBcValueY[ks]=LoadList[j].BCVal;
                  SplineMesh->Uy[ks]=LoadList[j].BCVal; break;                                           
      }
     }
    }
    
    // Aufbringung von Kraft-RB an den Seiten und Ecken
    
    for(i=num_disp; i<num_loads; i++)
    {
        Tval=(LoadList[i].BCVal);
        Texp=(LoadList[i].BCPot);
        
        if (0==LoadList[i].PrimFlag) // nur Kantenkrfte auftragen
        {
         numXi=(int)ceil((ceil(fabs(Texp))+(double)orderK+2.0)/2.0);
         numXi=GetSamplePointsToBSP(BSP,numXi);
         
         numpoly = (SpElemsX->numpoly);
         numcoeff = (SpElemsX->numcoeff);
         center  = (SpElemsX->center);
         
         switch (LoadList[i].NumFlag) //Kantennummer
         {
            case 0: TLe=nex; SpElems=SpElemsX; SpElemsN=SpElemsY; len=la; break;
            case 1: TLe=ney; SpElems=SpElemsY; SpElemsN=SpElemsX; len=lb; break;
            case 2: TLe=nex; SpElems=SpElemsX; SpElemsN=SpElemsY; len=la; break;
            case 3: TLe=ney; SpElems=SpElemsY; SpElemsN=SpElemsX; len=lb; break;
         }
         
         for(j=0; j<TLe; j++) // Schleife ber Elemente NormalMesh 
         {
            
            if (j <= center) EType=(j-center); // welcher Elementtyp
            else
            {
             iw1=TLe-1-center;
             if (j >= iw1) EType=(j-iw1);
             else EType=0;   
            }
            
            for(ii=0; ii<numpoly; ii++) // Schleife ber Knoten im Element Richtung Lastverlauf
            {
              
              //Auswertung des Integrals (Lastfkt*Formfkt)  =w4
              
              w4=0;
              
              for(kk=0; kk<numXi; kk++)  // Schleife ber Sttzstellen
              {
               w1=len/2*(BSP->Xi[kk])+len/2;
               if (0.0==Texp) w2=(BSP->Gi[kk])*Tval;
               else w2=(BSP->Gi[kk])*Tval/pow((len*TLe),Texp)*pow((w1+(double)j*len),Texp);
               w3=PolyVal(SpElems->Element[EType+center].Polynom[ii],w1);
               w4+=w2*w3;
              }
              
              w4*=len/2.0;
              
              //Zuweisung des Wertes w4
              
              switch (LoadList[i].NumFlag) //Kantennummer
              {
                  case 0: ks=j+ii;                  break;
                  case 1: ks=Spnnx*(1+j+ii)-1;      break;
                  case 2: ks=Spnnx*Spnny-1-ii-j;    break;
                  case 3: ks=Spnnx*(Spnny-1-ii-j);  break;
              }
              
              switch (LoadList[i].PhyFlag) //x- oder y-Richtung
              {
                  case 2: if (SpBcTypeX[ks]==0)  SpBcValueX[ks]+=w4;   
                          break;
                  case 3: if (SpBcTypeY[ks]==0)  SpBcValueY[ks]+=w4; 
                          break;
              }
             
            }  
          //printf("\n\n SpBcValueY: \n");
          //PrintMatrixT(Spnnx,Spnny,SpBcValueY);
          //getchar();
         }
        }
        
        if (1==LoadList[i].PrimFlag) // nur Eckkrfte auftragen
        {
         switch (LoadList[i].NumFlag)  // welche Ecke
         {
            case 0: ks=0;               break;
            case 1: ks=Spnnx-1;         break;
            case 2: ks=Spnny*Spnnx-1;   break;
            case 3: ks=(Spnny-1)*Spnnx; break;
         }
         switch (LoadList[i].PhyFlag)  //x- oder y-Richtung
         {
            case 2: if (SpBcTypeX[ks]==0) SpBcValueX[ks]+=Tval;
                    break;
            
            case 3: if (SpBcTypeY[ks]==0) SpBcValueY[ks]+=Tval;
                    break;
         }
        }
    }
    
    //Kontrollausgabe
    /*
    printf("\n\n BcTypeX: \n");                     // Type X
    PrintMatrixIntegerT(nnx,nny,BcTypeX);
    
    printf("\n\n SpBcTypeX: \n");
    PrintMatrixIntegerT(Spnnx,Spnny,SpBcTypeX);
    
    getchar();    
    
    printf("\n\n BcValueX: \n");                   // Value X
    PrintMatrixT(nnx,nny,BcValueX);
    
    printf("\n\n SpBcValueX: \n");
    PrintMatrixT(Spnnx,Spnny,SpBcValueX);    

    getchar();    
    
    printf("\n\n BcTypeY: \n");                     // Type Y
    PrintMatrixIntegerT(nnx,nny,BcTypeY);
    
    printf("\n\n SpBcTypeY: \n");
    PrintMatrixIntegerT(Spnnx,Spnny,SpBcTypeY);    
    
    getchar();
        
    printf("\n\n BcValueY: \n");                   // Value Y
    PrintMatrixT(nnx,nny,BcValueY);
        
    printf("\n\n SpBcValueY: \n");
    PrintMatrixT(Spnnx,Spnny,SpBcValueY);
    
    getchar();
    //*/
    
    return;
}

/*------------------------------------------------------------------------------
	PrintLoadlist   !neu -n43-!
------------------------------------------------------------------------------*/

void PrintLoadlist(OneLoad *LoadList, int num_loads)
{
     int i;
     
     printf("\n\nBoundary Conditions in Loadlist:\n");
     for(i=0; i<num_loads; i++)
     {
      printf("\n%2i. BC:  ",(i+1));
      if (0==LoadList[i].PrimFlag) printf("Edge  =");  
      else printf("Corner=");
      printf("%i  ",(LoadList[i].NumFlag+1));
      if (0==LoadList[i].PhyFlag) printf("Ux= ");
      if (1==LoadList[i].PhyFlag) printf("Uy= ");
      if (2==LoadList[i].PhyFlag) printf("Fx= ");
      if (3==LoadList[i].PhyFlag) printf("Fy= ");
      printf("%g  ",(LoadList[i].BCVal));
      printf("*(z^%g)",(LoadList[i].BCPot));	
     }
}

