/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <iostream>
using namespace std;

/*------------------------------------------------------------------------------
	 AllocBSplineMem  !neu -n04-!
------------------------------------------------------------------------------*/
void AllocBSplineMem(SomeElements *SpElems, int numElements, int numpoly, int numcoeff)
{   
    int i,j;
    
     SpElems->numpoly  = numpoly;
     SpElems->numcoeff = numcoeff;
     SpElems->numElements = numElements;
     SpElems->center = (int)floor((double)numElements/2.0);
     
    (SpElems->Element)=(SomePolynoms*) malloc(numElements*sizeof(SomePolynoms));
    
    for(i=0; i<numElements; i++) 
    {
        InitPolynomStruct(&(SpElems->Element[i].Polynom), numpoly);
        
        for(j=0; j<numpoly; j++)
        { 
          InitPolynomFunction( &(SpElems->Element[i].Polynom[j]), numcoeff);
        }
    }
    return;
}


/*------------------------------------------------------------------------------
	 FreeBSplineMem  !neu -n06-!
------------------------------------------------------------------------------*/
void FreeBSplineMem(SomeElements *SpElems)
{   
    int i,j;
    int numElements;
    int numpoly;
    
    numpoly = (SpElems->numpoly);
    numElements = (SpElems->numElements);
    
    for(i=0; i<numElements; i++) 
    {       
        for(j=0; j<numpoly; j++)
        { 
          DeletePolynomFunction(&(SpElems->Element[i].Polynom[j]));
        }
    }
    
    for(i=0; i<numElements; i++) 
    {
      free(SpElems->Element[i].Polynom);
    }
    
    free(SpElems->Element);
    
    return;
}


/*------------------------------------------------------------------------------
	 Helpfunction omega for SetOrderkBSplines  !neu -n09-!
------------------------------------------------------------------------------*/
void omega( double *F, int k, int p, int e, int ne )
{
    double xj1,xjk1;
    double xj2,xjk2;  
    
    // Teil 1 
    
    if (-k+p+1 < -e) xj1=(double)(-e);
    else xj1=(double)(-k+p+1);     
      
    if (p+1 < -e) xjk1=(double)(-e);
    else xjk1=(double)(p+1);  
    if (p+1 > ne-e) xjk1=(double)(ne-e); 
         
 
    if (xjk1-xj1 == 0.0) F[0]=0.0; 
    else F[0]=xj1/(xjk1-xj1)+1.0;
    
    if (xjk1-xj1 == 0.0) F[1]=0.0; 
    else F[1]=-1.0/(xjk1-xj1);
    
    // Teil 2
         
    if (-k+p < -e) xj2=(double)(-e);
    else xj2=(double)(-k+p);     
      
    if (p < -e) xjk2=(double)(-e);
    else xjk2=(double)(p);  
    if (p > ne-e) xjk2=(double)(ne-e);  

    
    if (xjk2-xj2 == 0.0) F[2]=0.0; 
    else F[2]=-xj2/(xjk2-xj2);
     
    if (xjk2-xj2 == 0.0) F[3]=0.0; 
    else F[3]=1.0/(xjk2-xj2);
   
    return;
}


/*------------------------------------------------------------------------------
	SetOrderkBSplines  !neu -n08-!
------------------------------------------------------------------------------*/
void SetOrderkBSplines(BSplineBlock *BSP)
{
    int i,ip,ic,k;
    int e,p;
    int numElements;
    int numpoly;    
    int numcoeff;
    double la,lb;
    double F[4];
    double a,b,c,d;
    double zw;
    SomeElements *SpElemsX;
    SomeElements *SpElemsY;
    SomeElements *SpElemsXdX;
    SomeElements *SpElemsYdY;
    
    SpElemsX=&(BSP->SplineElemsX); 
    SpElemsY=&(BSP->SplineElemsY);
    SpElemsXdX=&(BSP->SplineElemsXdX); 
    SpElemsYdY=&(BSP->SplineElemsYdY);
       
    numElements = (SpElemsX->numElements);    
    numcoeff = (SpElemsX->numcoeff);
    numpoly = (SpElemsX->numpoly);
    
    la = (BSP->la);
    lb = (BSP->lb);

    // x-direction , first values

    for(i=0; i<numElements; i++) 
    { 
     (SpElemsX->Element[i].Polynom[0].coefficient[0]) = 1.0;
    }
    
    // x-direction , all other values

    for(i=0; i<numElements; i++)
    {
      for(k=1; k<numpoly; k++)       
      {
        for(ip=k; ip>=0; ip--)       
        {
          omega( &F[0], k, ip, i, numElements );  
               
          for(ic=k; ic>=0; ic--)       
          {         
            a=(SpElemsX->Element[i].Polynom[ip].coefficient[ic]);
            
            if (ic < 1) b=0.0;
            else b=(SpElemsX->Element[i].Polynom[ip].coefficient[ic-1]);
            
            if (ip < 1) c=0.0;
            else c=(SpElemsX->Element[i].Polynom[ip-1].coefficient[ic]);
            
            if (ip < 1 || ic < 1) d=0.0;
            else d=(SpElemsX->Element[i].Polynom[ip-1].coefficient[ic-1]);
            
            
            zw=a*F[0]+b*F[1]+c*F[2]+d*F[3];
            (SpElemsX->Element[i].Polynom[ip].coefficient[ic])=zw;
            
          }
        }
      }
    }
    
    // y-direction , copy from x-direction * 1/(lb^ic)
   
    for(i=0; i<numElements; i++)
    {
        for(ip=0; ip<numpoly; ip++)
        {
            for(ic=0; ic<numcoeff; ic++)
            {
              a=(SpElemsX->Element[i].Polynom[ip].coefficient[ic]);
              (SpElemsY->Element[i].Polynom[ip].coefficient[ic]) = a/pow((double)lb,(double)ic);
            }
        }
    }

    // x-direction , * 1/(la^ic)

    for(i=0; i<numElements; i++)
    {
        for(ip=0; ip<numpoly; ip++)
        {
            for(ic=0; ic<numcoeff; ic++)
            {
              a=(SpElemsX->Element[i].Polynom[ip].coefficient[ic]);
              (SpElemsX->Element[i].Polynom[ip].coefficient[ic]) = a/pow((double)la,(double)ic);
            }
        }
    }
    
    // x-direction derivative
    
    for(i=0; i<numElements; i++)
    {
        for(ip=0; ip<numpoly; ip++)
        {
            for(ic=0; ic<numcoeff; ic++)
            {
              a=(SpElemsX->Element[i].Polynom[ip].coefficient[ic]);
              (SpElemsXdX->Element[i].Polynom[ip].coefficient[ic]) = a;
            }
            DerivePolynomStruct (&(SpElemsXdX->Element[i].Polynom[ip]));

        }
    }    
    SpElemsXdX->numcoeff = numcoeff-1;
  
    // y-direction derivative
    
    for(i=0; i<numElements; i++)
    {
        for(ip=0; ip<numpoly; ip++)
        {
            for(ic=0; ic<numcoeff; ic++)
            {
              a=(SpElemsY->Element[i].Polynom[ip].coefficient[ic]);
              (SpElemsYdY->Element[i].Polynom[ip].coefficient[ic]) = a;
            }                        
            DerivePolynomStruct (&(SpElemsYdY->Element[i].Polynom[ip]));

        }
    }    
    SpElemsYdY->numcoeff = numcoeff-1; 
       
    /*/Kontrollausgabe
    e=0;
    while (e > -1)
    {
      cout <<  "\n Element=";
      cin >> e;
      cout <<  "\n Polynom=";
      cin >> p; 
           
      if (PrintSplinePolynomStruct( &(BSP->SplineElemsX),e-1,p-1) == 0) cout << "\n\nEingabefehler";
      if (PrintSplinePolynomStruct( &(BSP->SplineElemsXdX),e-1,p-1) == 0) cout << "\n\nEingabefehler";
      //if (PrintSplinePolynomStruct( &(BSP->SplineElemsY),e-1,p-1) == 0) cout << "\n\nEingabefehler";
      //if (PrintSplinePolynomStruct( &(BSP->SplineElemsYdY),e-1,p-1) == 0) cout << "\n\nEingabefehler";
      cout <<  "\n\n\n";
    }
    //*/
    
    return;
}


/*------------------------------------------------------------------------------
	AllocateMatrices_K  !neu -n11-!  for stiffness matrices
------------------------------------------------------------------------------*/

void AllocateMatrices_K(BSplineBlock *BSP)
{
   int    i,j;
   int    lastK, orderK, numXi;
   int    numElements, lastnumElems, numpoly, dof;
   int    BSPMatrBuildFlag;
   double *v;
   
   AllWinStruct *AllWinData;
   PutGetAllWinStruct(1, &AllWinData);
      
   BSPMatrBuildFlag=BSP->BSPMatrBuildFlag;
   
   lastK=BSP->lastK;
   lastnumElems=lastK*2-1;   
                 
   orderK=BSP->orderK;   
   numpoly=BSP->SplineElemsX.numpoly;
   numElements=BSP->SplineElemsX.numElements;
   
   dof=numpoly*numpoly*2;
   numXi=BSP->numXi;   
   
   //delete old matrices
   
   if (BSPMatrBuildFlag == 1)          
   {
    free(BSP->B);
    free(BSP->BT);
    free(BSP->Nx);
    free(BSP->Ny);

    for(i=0; i<lastnumElems; i++) 
    {    
       for(j=0; j<lastnumElems; j++)
           {
                free(BSP->K[i][j]);
           }        
    }
    
    for(i=0; i<lastnumElems; i++) 
    {
       free(BSP->K[i]);
    }

    free(BSP->K);
   }

   //allocate new matrices
   
   InitVector_double(3*dof,&v); BSP->B  =v;
   InitVector_double(3*dof,&v); BSP->BT =v;
   InitVector_double(dof,&v);   BSP->Nx =v;
   InitVector_double(dof,&v);   BSP->Ny =v;


   BSP->K=(double***) malloc(numElements*sizeof(double**));
   
   for(i=0; i<numElements; i++) 
   { 
       BSP->K[i]=(double**) malloc(numElements*sizeof(double*));
       for(j=0; j<numElements; j++)
           { 
            if (AllWinData->SolveWinData->YMsmoothFlag == 1)
            {InitVector_double(dof*dof*(numXi*numXi+1),&v);  (BSP->K[i][j]) =v;}
            else
            {InitVector_double(dof*dof,&v);  (BSP->K[i][j]) =v;}
           }   
   }
   
   BSP->BSPMatrBuildFlag = 1;
   
   return;    
}


/*------------------------------------------------------------------------------
	BuildMaterialMatrix
------------------------------------------------------------------------------*/

void BuildMaterialMatrix(double mu, double *C)
{

   C[0]=1.0/(1.0-mu*mu);
   C[1]=mu/(1.0-mu*mu);
   C[2]=0.0;
   C[3]=mu/(1.0-mu*mu);
   C[4]=1.0/(1.0-mu*mu);
   C[5]=0.0;
   C[6]=0.0;
   C[7]=0.0;
   C[8]=(1.0-mu)/2.0/(1.0-mu*mu);

   return;    
}


/*------------------------------------------------------------------------------
	BuildStiffnessForBSplines3  !neu -n12-! nutzt Spiegelsymmetrie aus
------------------------------------------------------------------------------*/

void BuildStiffnessForBSplines3(BSplineBlock *BSP)
{
     int i,j,xi,xj,half;
     int k,numXi;
     int dof;
     int numElements, numpoly, center;
     int TypeX, TypeY;
     double la,lb;
     double x,y;
     double thickness, G;
     double *B, *K;

     AllWinStruct *AllWinData;

     PutGetAllWinStruct(1, &AllWinData);
     thickness=AllWinData->GeomWinData->lz;
     k=BSP->orderK;
     numXi=BSP->numXi;
     
     numElements=BSP->SplineElemsX.numElements;
     numpoly=BSP->SplineElemsX.numpoly;
     center=BSP->SplineElemsX.center;
     dof=numpoly*numpoly*2;

     la=BSP->la;
     lb=BSP->lb;

     B=BSP->B;

     InitVector_double((dof*dof), &K);

     // Erzeugung erstes viertel
     for(i=0; i<=center; i++)
     {
     for(j=0; j<=center; j++)
         {
          TypeX=i-center; TypeY=j-center;
          for(xi=0; xi<numXi; xi++)
             {
             for(xj=0; xj<numXi; xj++)
                 {
                  x=la/2.0+(BSP->Xi[xi])*la/2.0;
                  y=lb/2.0+(BSP->Xi[xj])*lb/2.0;

                  GetBatXY_K(TypeX,TypeY,BSP,x,y);

                  G=(BSP->Gi[xi])*(BSP->Gi[xj]);
                  BTCBandAddKreplace(BSP,K,(BSP->K[i][j]),G);
                 }
             }
          MatrixMultScalar(dof,dof,(BSP->K[i][j]),(thickness*(la/2.0)*(lb/2.0)));
          //PrintMatrixToFile(dof,dof,(BSP->K[i][j]),"./Kmatrix.txt"); 
          //getchar();          
         }
     }

     for(i=center+1; i<numElements; i++)
     {
     for(j=0; j<=center; j++)
         {
          TypeX=i-center;
          ReflectVertical(BSP->K[center-TypeX][j],BSP->K[i][j],numpoly);
         }
     }

     for(i=0; i<numElements; i++)
     {
     for(j=center+1; j<numElements; j++)
         {
          TypeY=j-center;
          ReflectHorizontal(BSP->K[i][center-TypeY],BSP->K[i][j],numpoly);
         }
     }     
     /*/Kontrollausgabe
     for(j=0; j<numElements; j++)
     { 
      for(i=0; i<numElements; i++)
      {
       TypeX=i-center; TypeY=j-center;
     
       PrintMatrixToFile(dof*dof,1,(BSP->K[center+TypeX][center+TypeY]));
       printf("\n\n la=%f  lb=%f ",la,lb);
       printf("\n\n TypeX=%i  TypeY=%i ",TypeX,TypeY);
       printf("\n i=%i j=%i   Durchgang=%i\n",i,j,i+1+j*numElements);
       printf("\n......................................................");
       //PrintMatrixT(dof,dof,(BSP->K[center+TypeX][center+TypeY]));
       getchar();
      }
     }
     //*/

     free(K);
     return;
}

/*------------------------------------------------------------------------------
	BuildStiffnessForBSplines3a  fr multi-Phase E-Modul-Anwendung 
------------------------------------------------------------------------------*/

void BuildStiffnessForBSplines3a(BSplineBlock *BSP)
{
     int i,j,xi,xj;
     int numXi;
     int dof;
     int numElements, numpoly, center;
     int TypeX, TypeY;
     double la,lb;
     double x,y;
     double thickness, G;
     double *B, *BT, *C, *BTC, *K, *Kt, *Kg;

     AllWinStruct *AllWinData;

     PutGetAllWinStruct(1, &AllWinData);
     thickness=AllWinData->GeomWinData->lz;
     
     numXi=BSP->numXi;
     numElements=BSP->SplineElemsX.numElements;
     numpoly=BSP->SplineElemsX.numpoly;
     center=BSP->SplineElemsX.center;
     dof=numpoly*numpoly*2;
     
     la=BSP->la;
     lb=BSP->lb;
     
     B=BSP->B;
     BT=BSP->BT;
     C=BSP->C;     
     InitVector_double((dof*3), &BTC);
     //InitVector_double((dof*dof), &Kt);
     
     for(i=0; i<numElements; i++)
     { 
     for(j=0; j<numElements; j++)
         {
         TypeX=i-center; TypeY=j-center;
         Kg=&(BSP->K[i][j][numXi*numXi*dof*dof]);        
         for(xi=0; xi<numXi; xi++)
             { 
             for(xj=0; xj<numXi; xj++)
                 {              
                  x=la/2+(BSP->Xi[xi])*la/2;       
                  y=lb/2+(BSP->Xi[xj])*lb/2;
                  
                  GetBTatXY_K(TypeX,TypeY,BSP,x,y);
                  GetBatXY_K(TypeX,TypeY,BSP,x,y);
                  
                  MatrixMultMatrix(dof,3,3,BT,C,BTC);
                  K=&(BSP->K[i][j][(xi+numXi*xj)*dof*dof]);
                  MatrixMultMatrix(dof,3,dof,BTC,B,K);              
                        
                  G=(BSP->Gi[xi])*(BSP->Gi[xj])*(thickness*(la/2)*(lb/2));
                  MatrixMultScalar(dof,dof, K, G);
                  AddMatrix(dof,dof,Kg,K);
                  //AddMatrix(dof,dof,Kt,K);
                 }
             }
         //PrintMatrixToFile(dof,dof,Kt,"./Kmatrix.txt"); 
         //getchar();
         //MatrixMultScalar(dof,dof, Kt, 0);    
         }
     }
     
     free(BTC);
     //free(Kt);
     return;
}




/*------------------------------------------------------------------------------
	 BTCBandAddKreplace  !neu -n29-!  for stiffness matrices 
------------------------------------------------------------------------------*/

void BTCBandAddKreplace(BSplineBlock *BSP, double *K, double *KM, double G)
{
     int dof, dofd, numpoly;
     int i,j;
     double *B,*C;
        
     B=BSP->B;
     C=BSP->C;
          
     numpoly=BSP->SplineElemsX.numpoly;
     dof=numpoly*numpoly;
     dofd=dof*2;
     
     for(i=0; i<(dofd); i++)  //row
     for(j=0; j<(dofd); j++)  //col
     {
        K[j*dofd+i]=B[j*3+2]*B[i*3+2]*G;
        KM[j*dofd+i] += K[j*dofd+i]*C[8];
        if ( i >= dof && j <  dof ) KM[j*dofd+i] += K[(j+dof)*dofd+(i-dof)]*C[3];
        if ( i >= dof && j >= dof ) KM[j*dofd+i] += K[(j-dof)*dofd+(i-dof)]*C[0];             
     }    
     
     for(i=0; i<(dof ); i++)  //row
     for(j=0; j<(dofd); j++)  //col
     {
        if ( j < dof ) KM[j*dofd+i] += K[(j+dof)*dofd+(i+dof)]*C[4];
        else KM[j*dofd+i] += K[(j-dof)*dofd+(i+dof)]*C[1];             
     } 
         
     return;
}


/*------------------------------------------------------------------------------
	 ReflectHorizontal  !neu -n30-!  for stiffness matrices
------------------------------------------------------------------------------*/

void ReflectHorizontal( double *Kold, double *Knew, int numpoly)
{
     int i,j,ii,jj;
     int nps, dof, dofd;
     int a,b;
     
     nps=numpoly-1;
     dof=numpoly*numpoly;
     dofd=numpoly*numpoly*2;
     
     for(i=0; i<numpoly; i++)
     for(j=0; j<numpoly; j++)
     for(ii=0; ii<numpoly; ii++)
     for(jj=0; jj<numpoly; jj++)
     {
     a=j+i*dofd  + jj*numpoly + ii*dofd*numpoly;
     b=j+i*dofd  - (jj-nps)*numpoly -(ii-nps)*dofd*numpoly;
     
     Knew[a]=Kold[b];
     Knew[a+dof]=-Kold[b+dof];
     Knew[a+dof*dofd]=-Kold[b+dof*dofd];
     Knew[a+dof*(dofd+1)]=Kold[b+dof*(dofd+1)];
     }
     return;
}


/*------------------------------------------------------------------------------
	 ReflectVertical  !neu -n31-!  for stiffness matrices
------------------------------------------------------------------------------*/

void ReflectVertical( double *Kold, double *Knew, int numpoly)
{
     int i,j,ii,jj;
     int nps, dof, dofd;
     int a,b;
     
     nps=numpoly-1;
     dof=numpoly*numpoly;
     dofd=numpoly*numpoly*2;
     
     for(i=0; i<numpoly; i++)
     for(j=0; j<numpoly; j++)
     for(ii=0; ii<numpoly; ii++)
     for(jj=0; jj<numpoly; jj++)
     {
     a=j+i*dofd + jj*numpoly + ii*dofd*numpoly;
     b=-(j-nps) -(i-nps)*dofd + jj*numpoly + ii*dofd*numpoly;
     
     Knew[a]=Kold[b];
     Knew[a+dof]=-Kold[b+dof];
     Knew[a+dof*dofd]=-Kold[b+dof*dofd];
     Knew[a+dof*(dofd+1)]=Kold[b+dof*(dofd+1)];
     }
     return;
}


/*------------------------------------------------------------------------------
	 GetBatXY_K   !neu -n27-!
------------------------------------------------------------------------------*/
 
void GetBatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y)
/*******************************************************************************
type defines if the corresponding element is 
in the middle or at the boundary ... etc.
TypeY: 
 1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
 0      (K_1st_boundaryelement, K_centerelement       , K_1st_boundaryelement)
-1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
TypeX:            -1                      0                       1    
*******************************************************************************/
{
   int i,j;
   SomePolynoms *PNx, *PNy, *PNDx, *PNDy;
   double *B;
   int numpoly, center, npsq;

   center=BSP->SplineElemsX.center;
   numpoly=BSP->SplineElemsX.numpoly;
   npsq=numpoly*numpoly;

   PNx=&(BSP->SplineElemsX.Element[center+TypeX]);
   PNy=&(BSP->SplineElemsY.Element[center+TypeY]);

   PNDx=&(BSP->SplineElemsXdX.Element[center+TypeX]);
   PNDy=&(BSP->SplineElemsYdY.Element[center+TypeY]);

   B=BSP->B;
   
   for(j=0; j<numpoly; j++)
   for(i=0; i<numpoly; i++)
   {   
       B[(i+numpoly*j)*3]=PolyVal(PNDx->Polynom[i], x) *  PolyVal(PNy->Polynom[j] , y);
       B[(i+numpoly*j)*3 +(npsq*3+2)]=B[(i+numpoly*j)*3];
       
       B[(i+numpoly*j)*3+2]=PolyVal(PNx->Polynom[i] , x) *  PolyVal(PNDy->Polynom[j], y);
       B[(i+numpoly*j)*3+1+(npsq*3)]=B[(i+numpoly*j)*3+2];       
   }
       
   return;    
} 


/*------------------------------------------------------------------------------
	 GetBTatXY_K    !neu -n28-!
------------------------------------------------------------------------------*/
 
void GetBTatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y)
/*******************************************************************************
type defines if the corresponding element is 
in the middle or at the boundary ... etc.
TypeY: 
 1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
 0      (K_1st_boundaryelement, K_centerelement       , K_1st_boundaryelement)
-1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
TypeX:            -1                      0                       1    
*******************************************************************************/
{
   int i,j;
   SomePolynoms *PNx, *PNy, *PNDx, *PNDy;
   double *BT;
   int numpoly, center, npsq;

   center=BSP->SplineElemsX.center;
   numpoly=BSP->SplineElemsX.numpoly;
   npsq=numpoly*numpoly;

   PNx=&(BSP->SplineElemsX.Element[center+TypeX]);
   PNy=&(BSP->SplineElemsY.Element[center+TypeY]);

   PNDx=&(BSP->SplineElemsXdX.Element[center+TypeX]);
   PNDy=&(BSP->SplineElemsYdY.Element[center+TypeY]);

   BT=BSP->BT;

   for(i=0; i<numpoly; i++) 
   for(j=0; j<numpoly; j++)
   {   
       BT[i+numpoly*j]          =PolyVal(PNDx->Polynom[i], x) *  PolyVal(PNy->Polynom[j] , y);
       BT[i+numpoly*j +(npsq*5)]=BT[i+numpoly*j];
       
       BT[i+numpoly*j +(npsq*3)]=PolyVal(PNx->Polynom[i] , x) *  PolyVal(PNDy->Polynom[j], y);
       BT[i+numpoly*j +(npsq*4)]=BT[i+numpoly*j +(npsq*3)];       
   }

   return;    
} 


/*------------------------------------------------------------------------------
	 GetNatXY_K    !neu -n26-!
------------------------------------------------------------------------------*/
 
void GetNatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y)
/*******************************************************************************
type defines if the corresponding element is 
in the middle or at the boundary ... etc.
TypeY: 
 1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
 0      (K_1st_boundaryelement, K_centerelement       , K_1st_boundaryelement)
-1      (K_1st_boundaryelement, K_1st_boundaryelement , K_1st_boundaryelement)
TypeX:            -1                      0                       1    
*******************************************************************************/
{
   int i,j,k;
   SomePolynoms *PNx, *PNy;
   double *Nx,*Ny;
   int numpoly, center;

   center=BSP->SplineElemsX.center;
   numpoly=BSP->SplineElemsX.numpoly;
   PNx=&(BSP->SplineElemsX.Element[center+TypeX]);
   PNy=&(BSP->SplineElemsY.Element[center+TypeY]);
   
   Nx=BSP->Nx;
   Ny=BSP->Ny;
 
   for(i=0; i<numpoly; i++) 
   for(j=0; j<numpoly; j++)
   {
       Nx[i+numpoly*j] =  PolyVal(PNx->Polynom[i],x) * PolyVal(PNy->Polynom[j],y);
       Ny[i+numpoly*j+(numpoly*numpoly)] =  Nx[i+numpoly*j];
   }

   return;    
} 


/*------------------------------------------------------------------------------
	InitOrderkBSplines  !neu -n03-!
------------------------------------------------------------------------------*/
int InitOrderkBSplines()
{
    int k;
    int numpoly, numcoeff, numElements;
    SolveBlock *SolBlock;
    BSplineBlock *BSP;
    
    //end of declaration
    
    PutGetSolveBlock(1,&SolBlock);
    BSP=&(SolBlock->BSplines);
    
    k=BSP->orderK; 
    
    //Berechnung der notwendigen Werte    
    numElements=k*2-1;  //
    numpoly=k+1;
    numcoeff=k+1;

    //Fallunterscheidung:
    if (BSP->initSplineFlag != 0)   
    {
    FreeBSplineMem( &(BSP->SplineElemsX) );
    FreeBSplineMem( &(BSP->SplineElemsY) ); 
    FreeBSplineMem( &(BSP->SplineElemsXdX) );
    FreeBSplineMem( &(BSP->SplineElemsYdY) );
    }

    AllocBSplineMem( &(BSP->SplineElemsX), numElements, numpoly, numcoeff );
    AllocBSplineMem( &(BSP->SplineElemsY), numElements, numpoly, numcoeff ); 
    AllocBSplineMem( &(BSP->SplineElemsXdX), numElements, numpoly, numcoeff );
    AllocBSplineMem( &(BSP->SplineElemsYdY), numElements, numpoly, numcoeff );
    
    BSP->initSplineFlag=1;

return 1;
}









