/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*---------------------------------------------------------
	BSplineMain2
----------------------------------------------------------*/

void BSplineMain2()
{
double lx, ly, mu;
int num_elem_x, num_elem_y, num_levels, length, numpoly, dof;
MGBlock *NormalMesh, *SplineMesh;
SolveBlock *SolBlock;

    
   /************** Get information of solver block *******************/
   PutGetSolveBlock(1,&SolBlock);
   num_levels=SolBlock->num_levels; 
   NormalMesh=&(SolBlock->Meshes[num_levels-1]);
   
   num_elem_x=(NormalMesh->nex)   +1  ;
   num_elem_y=(NormalMesh->ney)   +1  ;
   
   lx=((double)(NormalMesh->la))*((double)(NormalMesh->nex)) ;
   ly=((double)(NormalMesh->lb))*((double)(NormalMesh->ney)) ;
   
   
   /************** Create Spline Mesh ******************************/
   InitMGData (&SplineMesh, 1);
   SolBlock->SplineMesh=SplineMesh;
   InitMGBlock2and3d(lx, ly, 1.0, num_elem_x, num_elem_y, 0, SplineMesh);
   PutGetSolveBlock(1,&SolBlock);
   SplineMesh=SolBlock->SplineMesh;
   
   
   /************** Initialize Spline Basis ******************************/  
   InitQuadraticBSplines();
   SetQuadraticBSplines(&(SolBlock->BSplines.Polynomials[0]),(NormalMesh->la));
   SetQuadraticBSplines(&(SolBlock->BSplines.Polynomials[6]),(NormalMesh->lb));
   mu=0.2;
   BuildMaterialMatrix(mu, SolBlock->BSplines.C);
   SolBlock->BSplines.la=NormalMesh->la;
   SolBlock->BSplines.lb=NormalMesh->lb;
   numpoly=3;
   SolBlock->BSplines.numpoly=numpoly;
   AllocateMatrices(&(SolBlock->BSplines));
   
   BuildStiffnessForBSplines2(&(SolBlock->BSplines));
       
   
   /************** Boundary conditions ******************************/  
   SplineBCmain();
   
   
   /****************** Start Solver ********************************/  
   CGSolverForBSplines(SolBlock, NormalMesh, SplineMesh, 
                        SolBlock->BSplines.K11,
                        SolBlock->BSplines.K12,
                        SolBlock->BSplines.K13,
                        SolBlock->BSplines.K21,
                        SolBlock->BSplines.K22,
                        SolBlock->BSplines.K23,
                        SolBlock->BSplines.K31,
                        SolBlock->BSplines.K32,
                        SolBlock->BSplines.K33); 
   return;    
}


/*---------------------------------------------------------
	AllocateMatrices
----------------------------------------------------------*/

void AllocateMatrices(BSplineBlock *BSP)
{
   int    n, dof;
   double *v;
   
   n=BSP->numpoly;
   dof=n*n*2;
   
   InitVector_double(3*dof,&v); BSP->B  =v;
   InitVector_double(3*dof,&v); BSP->BT =v;
   InitVector_double(3*dof,&v); BSP->B  =v;
   
   InitVector_double(dof*dof,&v); BSP->K11 =v;
   InitVector_double(dof*dof,&v); BSP->K12 =v;
   InitVector_double(dof*dof,&v); BSP->K13 =v;
   InitVector_double(dof*dof,&v); BSP->K21 =v;
   InitVector_double(dof*dof,&v); BSP->K22 =v;
   InitVector_double(dof*dof,&v); BSP->K23 =v;
   InitVector_double(dof*dof,&v); BSP->K31 =v;
   InitVector_double(dof*dof,&v); BSP->K32 =v;
   InitVector_double(dof*dof,&v); BSP->K33 =v;
   
   InitVector_double(dof,&v); BSP->Nx =v;
   InitVector_double(dof,&v); BSP->Ny =v;
   
   return;    
}


/*---------------------------------------------------------
	GetBTatXY
----------------------------------------------------------*/
 
void GetBTatXY(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y)
/*****************************************************
type defines if the corresponding element is 
in the middle or at the boundary ... etc.
TypeY: 
2   (K31, K32, K33)
1   (K21, K22, K23)
0   (K11, K12, K13)
TypeX:0 -  1  - 2    
******************************************************/
{
int i,j,k;
SomePolynoms *PNx, *PNy, *PNDx, *PNDy;
double *BT;
int numpoly;
double *Nx,*Ny; //


   PNx=&(BSP->Polynomials[TypeX]);
   PNy=&(BSP->Polynomials[TypeY+6]);

   PNDx=&(BSP->Polynomials[TypeX+3]);
   PNDy=&(BSP->Polynomials[TypeY+9]);

   BT=BSP->BT;
   numpoly=BSP->numpoly;

   InitVector_double((numpoly*numpoly), &Nx); //Nx and Ny are not the shape functions!
   InitVector_double((numpoly*numpoly), &Ny);

   for(i=0; i<numpoly; i++) 
   for(j=0; j<numpoly; j++)
   {
       Nx[i+numpoly*j] =  PolyVal(PNDx->Polynom[i], x)  *   PolyVal(PNy->Polynom[j] , y);
       Ny[i+numpoly*j] =  PolyVal(PNx->Polynom[i] , x)  *   PolyVal(PNDy->Polynom[j], y);
       
       BT[i+numpoly*j]                      =Nx[i+numpoly*j];
       BT[i+numpoly*j  +(numpoly*numpoly*3)]=Ny[i+numpoly*j];
       BT[i+numpoly*j  +(numpoly*numpoly*4)]=Ny[i+numpoly*j];
       BT[i+numpoly*j  +(numpoly*numpoly*5)]=Nx[i+numpoly*j];
   }

   free(Nx);
   free(Ny);
   return;    
} 
/*---------------------------------------------------------
	GetNatXY
----------------------------------------------------------*/
 
void GetNatXY(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y)
/*****************************************************
type defines if the corresponding element is 
in the middle or at the boundary ... etc.
TypeY: 
2   (K31, K32, K33)
1   (K21, K22, K23)
0   (K11, K12, K13)
TypeX:0 -  1  - 2    
******************************************************/
{
int i,j,k;
SomePolynoms *PNx, *PNy;
double *Nx,*Ny;
int numpoly;


   PNx=&(BSP->Polynomials[TypeX]);
   PNy=&(BSP->Polynomials[TypeY+6]);
   
   Nx=BSP->Nx;
   Ny=BSP->Ny;
 
   numpoly=BSP->numpoly;


   for(i=0; i<numpoly; i++) 
   for(j=0; j<numpoly; j++)
   {
       Nx[i+numpoly*j] =  PolyVal(PNx->Polynom[i], x)  *   PolyVal(PNy->Polynom[j] , y);
       Ny[i+numpoly*j+(numpoly*numpoly)] =  Nx[i+numpoly*j];
   }

   return;    
} 


/*---------------------------------------------------------
	BuildStiffnessForBSplines2
----------------------------------------------------------*/

void BuildStiffnessForBSplines2(BSplineBlock *BSP)
{
   int im, jm, j, numpoly;
   double x, y;
   double la, lb, A;
   double i1,i2;
   double *B, *BT, *C, *BTC, *K;
   int    dof;
   double *K11, *K12, *K13, *K21, *K22, *K23, *K31, *K32, *K33;
   
   i1=0.211324865405;
   i2=0.788675134595;
   
   la=BSP->la;
   lb=BSP->lb;
   A=la*lb;
   
   B=BSP->B;
   BT=BSP->BT;
   C=BSP->C;
   
   numpoly=BSP->numpoly;
   dof=numpoly*numpoly*2;
   InitVector_double(dof*3, &BTC);
   InitVector_double((dof*dof), &K);
   
   K11=BSP->K11;    K12=BSP->K12;    K13=BSP->K13;
   K21=BSP->K21;    K22=BSP->K22;    K23=BSP->K23;
   K31=BSP->K31;    K32=BSP->K32;    K33=BSP->K33;
   
   for(im=0; im<numpoly; im++)
   for(jm=0; jm<numpoly; jm++) 
   for(j=0; j<4; j++) 
   {
       switch (j)
       {
           case 0: x=i1*la;y=i1*lb; break;
           case 1: x=i2*la;y=i1*lb; break;
           case 2: x=i1*la;y=i2*lb; break;
           case 3: x=i2*la;y=i2*lb; break;
       }
       

       GetBTatXY(im, jm, BSP, x, y);
       TransposeMatrix(dof,3,BT,B);            
       MatrixMultMatrix(dof,3,3,BT,C,BTC);
       MatrixMultMatrix(dof,3,dof,BTC,B,K);
            
       switch (im+3*jm) // Shape functions of y; 1=a; 2=b; 3=c
       {
           case 0: AddMatrix(dof,dof, K11, K); break;
           case 1: AddMatrix(dof,dof, K12, K); break;
           case 2: AddMatrix(dof,dof, K13, K); break;
           case 3: AddMatrix(dof,dof, K21, K); break; 
           case 4: AddMatrix(dof,dof, K22, K); break;
           case 5: AddMatrix(dof,dof, K23, K); break;
           case 6: AddMatrix(dof,dof, K31, K); break;
           case 7: AddMatrix(dof,dof, K32, K); break;  
           case 8: AddMatrix(dof,dof, K33, K); break;
       }           
   }
   printf("\nBT\n");
   PrintMatrix(18,3,BT);
   
   
   MatrixMultScalar(dof,dof, K11, A);
   MatrixMultScalar(dof,dof, K12, A);
   MatrixMultScalar(dof,dof, K13, A);
   MatrixMultScalar(dof,dof, K21, A);
   MatrixMultScalar(dof,dof, K22, A);
   MatrixMultScalar(dof,dof, K23, A);
   MatrixMultScalar(dof,dof, K31, A);
   MatrixMultScalar(dof,dof, K32, A);
   MatrixMultScalar(dof,dof, K33, A);
   
   free(BTC);
   free(K);

   return;    
}




/*---------------------------------------------------------
	TransformSplinesOnMesh2
----------------------------------------------------------*/

// direction=0 means x-direction
// direction=1 means y-direction

int TransformSplinesOnMesh2(int direction,   int nnx, int nny, 
                            double la, double lb, double *SUx, double *SUy,
                            BSplineBlock *BSP, 
                            double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,m,n;
    int nex,ney,nns;
    int TypeX,TypeY;
    double U[18],tmp[1], *Ux, *Uy;
 
    
    nex=nnx-1;
    ney=nny-1;
    nns=nnx+1;
    Ux=SUx;
    Uy=SUy;
            
       for (j = 0; j < ney; j++) 
       for (i = 0; i < nex; i++)   
       {
         TypeX=1;
         if(0==i) TypeX=0;
         if((nex-1)==i) TypeX=2;
         
         TypeY=1;
         if(0==j) TypeY=0;
         if((ney-1)==j) TypeY=2;
         
            U[0]=Ux[i+j*nns];
            U[1]=Ux[1+i+j*nns];
            U[2]=Ux[2+i+j*nns];
            
            U[3]=Ux[i+(j+1)*nns];
            U[4]=Ux[1+i+(j+1)*nns];
            U[5]=Ux[2+i+(j+1)*nns];
            
            U[6]=Ux[i+(j+2)*nns];
            U[7]=Ux[1+i+(j+2)*nns];
            U[8]=Ux[2+i+(j+2)*nns];
            
            U[9]=Uy[i+j*nns];
            U[10]=Uy[1+i+j*nns];
            U[11]=Uy[2+i+j*nns];
            
            U[12]=Uy[i+(j+1)*nns];
            U[13]=Uy[1+i+(j+1)*nns];
            U[14]=Uy[2+i+(j+1)*nns];
            
            U[15]=Uy[i+(j+2)*nns];
            U[16]=Uy[1+i+(j+2)*nns];
            U[17]=Uy[2+i+(j+2)*nns];
         
            GetNatXY(TypeX, TypeY, BSP, 0.0, 0.0);
            if (0==direction)
            MatrixMultMatrix(1,18,1,BSP->Nx,U,tmp); else
            MatrixMultMatrix(1,18,1,BSP->Ny,U,tmp);
            //NUy[i+j*nnx]=tmp[0];
            Post1[i+j*nex]=tmp[0];
          
            GetNatXY(TypeX, TypeY, BSP, la, 0.0);
            if (0==direction)
            MatrixMultMatrix(1,18,1,BSP->Nx,U,tmp); else
            MatrixMultMatrix(1,18,1,BSP->Ny,U,tmp);
            //NUy[i+1+j*nnx]=tmp[0];
            Post2[i+j*nex]=tmp[0];
          
            GetNatXY(TypeX, TypeY, BSP, la, lb);
            if (0==direction)
            MatrixMultMatrix(1,18,1,BSP->Nx,U,tmp); else
            MatrixMultMatrix(1,18,1,BSP->Ny,U,tmp);
            //NUy[i+1+(j+1)*nnx]=tmp[0];
            Post3[i+j*nex]=tmp[0];
            
            GetNatXY(TypeX, TypeY, BSP, 0.0, lb);
            if (0==direction)
            MatrixMultMatrix(1,18,1,BSP->Nx,U,tmp); else
            MatrixMultMatrix(1,18,1,BSP->Ny,U,tmp);
            //NUy[i+(j+1)*nnx]=tmp[0];
            Post4[i+j*nex]=tmp[0];
          
       }
    return(1);
}

/*---------------------------------------------------------
	TransformSplinesOnMesh3
----------------------------------------------------------*/

// type=0 means strain Exx 
// type=1 means strain Eyy
// type=2 means strain Exy
// type=3 means strain Sxx 
// type=4 means strain Syy
// type=5 means strain Sxy


int TransformSplinesOnMesh3(int type,   int nnx, int nny, 
                            double la, double lb, double *SUx, double *SUy,
                            double *MatFieldE,
                            BSplineBlock *BSP, 
                            double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,m,n;
    int nex,ney,nns;
    int TypeX,TypeY;
    double U[18],tmp[3], *Ux, *Uy, tmp2[3], v ;
    double B[54];
    double *C;
 
    
    nex=nnx-1;
    ney=nny-1;
    nns=nnx+1;
    Ux=SUx;
    Uy=SUy;
    C=BSP->C;
            
       for (j = 0; j < ney; j++) 
       for (i = 0; i < nex; i++)   
       {
         TypeX=1;
         if(0==i) TypeX=0;
         if((nex-1)==i) TypeX=2;
         
         TypeY=1;
         if(0==j) TypeY=0;
         if((ney-1)==j) TypeY=2;
         
            U[0]=Ux[i+j*nns];
            U[1]=Ux[1+i+j*nns];
            U[2]=Ux[2+i+j*nns];
            
            U[3]=Ux[i+(j+1)*nns];
            U[4]=Ux[1+i+(j+1)*nns];
            U[5]=Ux[2+i+(j+1)*nns];
            
            U[6]=Ux[i+(j+2)*nns];
            U[7]=Ux[1+i+(j+2)*nns];
            U[8]=Ux[2+i+(j+2)*nns];
            
            U[9]=Uy[i+j*nns];
            U[10]=Uy[1+i+j*nns];
            U[11]=Uy[2+i+j*nns];
            
            U[12]=Uy[i+(j+1)*nns];
            U[13]=Uy[1+i+(j+1)*nns];
            U[14]=Uy[2+i+(j+1)*nns];
            
            U[15]=Uy[i+(j+2)*nns];
            U[16]=Uy[1+i+(j+2)*nns];
            U[17]=Uy[2+i+(j+2)*nns];
   // Post1      
            GetBTatXY(TypeX, TypeY, BSP, 0.0, 0.0); TransposeMatrix(18,3,BSP->BT,B);
            MatrixMultMatrix(3,18,1,B,U,tmp); 
            switch(type)
            {
             case 0: v=tmp[0]; break;
             case 1: v=tmp[1]; break;
             case 2: v=tmp[2]; break;
             case 3: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[0]*MatFieldE[i+j*nex];break;
             case 4: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[1]*MatFieldE[i+j*nex];break;
             case 5: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[2]*MatFieldE[i+j*nex];break;
            }
            Post1[i+j*nex]=v;
            
   // Post2             
            GetBTatXY(TypeX, TypeY, BSP, la, 0.0); TransposeMatrix(18,3,BSP->BT,B);
            MatrixMultMatrix(3,18,1,B,U,tmp); 
            switch(type)
            {
             case 0: v=tmp[0]; break;
             case 1: v=tmp[1]; break;
             case 2: v=tmp[2]; break;
             case 3: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[0]*MatFieldE[i+j*nex];break;
             case 4: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[1]*MatFieldE[i+j*nex];break;
             case 5: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[2]*MatFieldE[i+j*nex];break;
            }
            Post2[i+j*nex]=v;

   // Post3              
            GetBTatXY(TypeX, TypeY, BSP, la, lb); TransposeMatrix(18,3,BSP->BT,B);
            MatrixMultMatrix(3,18,1,B,U,tmp); 
            switch(type)
            {
             case 0: v=tmp[0]; break;
             case 1: v=tmp[1]; break;
             case 2: v=tmp[2]; break;
             case 3: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[0]*MatFieldE[i+j*nex];break;
             case 4: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[1]*MatFieldE[i+j*nex];break;
             case 5: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[2]*MatFieldE[i+j*nex];break;
            }
            Post3[i+j*nex]=v;

   // Post4              
            GetBTatXY(TypeX, TypeY, BSP, 0.0, lb); TransposeMatrix(18,3,BSP->BT,B);
            MatrixMultMatrix(3,18,1,B,U,tmp); 
            switch(type)
            {
             case 0: v=tmp[0]; break;
             case 1: v=tmp[1]; break;
             case 2: v=tmp[2]; break;
             case 3: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[0]*MatFieldE[i+j*nex];break;
             case 4: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[1]*MatFieldE[i+j*nex];break;
             case 5: MatrixMultMatrix(3,3,1,C,tmp,tmp2); v=tmp2[2]*MatFieldE[i+j*nex];break;
            }
            Post4[i+j*nex]=v;            

          
       }
    return(1);
}


