/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

//neue Datei


/*******************************************************************************
	BSplineMain3    !neu -n02-!
*******************************************************************************/

void BSplineMain3()
{
   double lx, ly, mu;
   int Spnex, Spney, num_levels;
   int numXi, orderK;
   int YMsmoothFlag, CircleElementFlag, CircleIntpntFlag;
   AllWinStruct *AllWinData;
   MGBlock *NormalMesh, *SplineMesh;
   SolveBlock *SolBlock;
   BSplineBlock *BSP;
    
   PutGetSolveBlock(1,&SolBlock);   
   PutGetAllWinStruct(1, &AllWinData);
   BSP=&(SolBlock->BSplines);   
   
   /************** Get information of solver block ****************************/
   YMsmoothFlag=AllWinData->SolveWinData->YMsmoothFlag;
   CircleElementFlag=AllWinData->SolveWinData->CircleElementFlag;
   CircleIntpntFlag=AllWinData->SolveWinData->CircleIntpntFlag;
   num_levels=SolBlock->num_levels;
   NormalMesh=&(SolBlock->Meshes[num_levels-1]);   
   BSP->orderK=AllWinData->SolveWinData->BSpOrderK;  //get values
   orderK=BSP->orderK;
   //mu=AllWinData->MesoWinData->Mu_Matrix;
   mu=0.2;
   SolBlock->BSplines.la=NormalMesh->la;   
   SolBlock->BSplines.lb=NormalMesh->lb;
   
   if ((NormalMesh->nex) < (orderK*2-1)) 
   {
    printf("\n\n\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
    printf("\n\n\n x-direction number of elements (%i) is too small for ",(NormalMesh->nex));                     
    printf("\n order %i B-Splines / no solution / %i needed",orderK,(orderK*2-1));
    printf("\n\n\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
    return;
   }    
   
   if ((NormalMesh->ney) < (orderK*2-1)) 
   {
    printf("\n\n\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
    printf("\n\n\n y-direction number of elements (%i) is too small for ",(NormalMesh->ney));                     
    printf("\n order %i B-Splines / no solution / %i needed",orderK,(orderK*2-1));
    printf("\n\n\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
    return;
   }     
   
   Spnex=(NormalMesh->nex)+orderK-1 ;
   Spney=(NormalMesh->ney)+orderK-1 ;
   
   lx=((double)(NormalMesh->la))*((double)(NormalMesh->nex)) ;
   ly=((double)(NormalMesh->lb))*((double)(NormalMesh->ney)) ;

   /************** create smoothed Youngs Modulus Field **********************/
   if (YMsmoothFlag == 1) 
   { 
      if (1 == CircleElementFlag || 1 == CircleIntpntFlag)
          createKnMatFieldCircle();
      else
          createKnMatFieldE(NormalMesh);
   }
   
   /************** Create Spline Mesh *****************************************/
   if(0==AllWinData->SolveWinData->ContFlag)
   {
       if (1==SolBlock->SpMeshDefined) 
       {SplineMesh=SolBlock->SplineMesh; FreeMGBlock(SplineMesh);}
       if (0==SolBlock->SpMeshDefined) InitMGData (&SplineMesh, 1);
       InitMGBlock(lx, ly, Spnex, Spney, SplineMesh);
       SolBlock->SpMeshDefined=1;
       SolBlock->SplineMesh=SplineMesh;
   }
   else
   {
       SplineMesh=SolBlock->SplineMesh; 
   }
   /************** Initialize Spline Basis ************************************/
   numXi=orderK+2;
   numXi=GetSamplePointsToBSP(BSP,numXi);
     
   InitOrderkBSplines();
   SetOrderkBSplines(BSP);
   BuildMaterialMatrix(mu, SolBlock->BSplines.C);
   AllocateMatrices_K(&(SolBlock->BSplines));
   if (YMsmoothFlag == 1)
   {BuildStiffnessForBSplines3a(&(SolBlock->BSplines));}
   else
   {BuildStiffnessForBSplines3(&(SolBlock->BSplines));}
   //printf("\n\n\n!!!!!!!! Steifigkeitsmatrizen erzeugt !!!!!!!!!!!!!!!!!");
   //getchar();
   if(1==AllWinData->SolveWinData->ContFlag) goto ContinueComputation;
   /************** Boundary conditions ****************************************/
   SplineBCmain_K();
   numXi=GetSamplePointsToBSP(BSP,numXi);
   BSP->lastK=BSP->orderK;
   /****************** Start Solver *******************************************/
   ContinueComputation:
   CGSolverForBSplines_K(SolBlock, NormalMesh, SplineMesh, YMsmoothFlag, numXi);
   
   /**************** interne Ergebnisausgabe **********************************/
   /*
   printf("\n Vektor Fx: \n");
   //PrintMatrixT((SplineMesh->nnx)*(SplineMesh->nny),1,SplineMesh->Fx);
   PrintMatrixT((SplineMesh->nnx)*(SplineMesh->nny),1,SplineMesh->Fx);
   getchar();
   printf("\n Vektor Fy: \n");
   PrintMatrixT((SplineMesh->nnx)*(SplineMesh->nny),1,SplineMesh->Fy);
   getchar();
   */
   return;
}



/*------------------------------------------------------------------------------
	TransformSplinesOnMesh2_K   !neu -n39-!
------------------------------------------------------------------------------*/

// direction=0 means x-direction
// direction=1 means y-direction

void TransformSplinesOnMesh2_K(int direction,   int nex, int ney,
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             BSplineBlock *BSP,
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int iw1;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double *U,tmp[1];
    
    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;
        
    InitVector_double(Ksize,&U);
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                if (0==direction) U[k]=Ux[ks];
                else U[k+KsizeH]=Uy[ks];
             }  
            }
    
            GetNatXY_K(TypeX, TypeY, BSP, 0.0, 0.0);
            if (0==direction) MatrixMultMatrix(1,Ksize,1,BSP->Nx,U,tmp); 
                        else  MatrixMultMatrix(1,Ksize,1,BSP->Ny,U,tmp);
            Post1[i+j*nex]=tmp[0];
    
            GetNatXY_K(TypeX, TypeY, BSP,  la, 0.0);
            if (0==direction) MatrixMultMatrix(1,Ksize,1,BSP->Nx,U,tmp); 
                        else  MatrixMultMatrix(1,Ksize,1,BSP->Ny,U,tmp);
            Post2[i+j*nex]=tmp[0];    

            GetNatXY_K(TypeX, TypeY, BSP,  la,  lb);
            if (0==direction) MatrixMultMatrix(1,Ksize,1,BSP->Nx,U,tmp); 
                        else  MatrixMultMatrix(1,Ksize,1,BSP->Ny,U,tmp);
            Post3[i+j*nex]=tmp[0];     
    
            GetNatXY_K(TypeX, TypeY, BSP, 0.0,  lb);
            if (0==direction) MatrixMultMatrix(1,Ksize,1,BSP->Nx,U,tmp); 
                        else  MatrixMultMatrix(1,Ksize,1,BSP->Ny,U,tmp);
            Post4[i+j*nex]=tmp[0];     
       }
    }
    free(U);
}

/*------------------------------------------------------------------------------
	TransformSplinesOnMesh3_K   !neu -n40-!
------------------------------------------------------------------------------*/

// type=0 means strain Exx
// type=1 means strain Eyy
// type=2 means strain Exy
// type=3 means strain Sxx
// type=4 means strain Syy
// type=5 means strain Sxy


void TransformSplinesOnMesh3_K(int type,   int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *MatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int iw1;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double BU[3],CBU[3],v;
    double *U,*C,*B;

    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    C=BSP->C;
    B=BSP->B;
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;

    InitVector_double(Ksize,&U);
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                U[k]=Ux[ks];  // 1.0; //
                U[k+KsizeH]=Uy[ks];
             }  
            }
            // Kontrollausgabe
            /*
            if (BSP->help==1 && i==0 && j==0)
            {  
             printf("\n Element 3,3: \n");
             printf(" TypeX= %i     TypeY= %i  \n",TypeX, TypeY);
             printf(" Center= %i\n",center);
             
             printf("\n Post1:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, 0.0);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
                          
             printf("\n Post2:    ");
             GetBatXY_K(TypeX, TypeY, BSP, la, 0.0);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             printf("\n Post3:    ");
             GetBatXY_K(TypeX, TypeY, BSP, la, lb);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             printf("\n Post4:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, lb);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             getchar();
            }
            
            if (BSP->help==1 && i==4 && j==3)
            {  
             printf("\n Element 4,3: \n");
             printf(" TypeX= %i     TypeY= %i  \n",TypeX, TypeY);
             printf(" Center= %i\n",center);
             
             printf("\n Post1:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, 0.0);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             printf("\n Post4:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, lb);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             getchar();
            }
            
            if (BSP->help==320 && i==3 && j==3)
            {  
             printf("\n Element 3,3: \n");
             printf(" TypeX= %i     TypeY= %i  \n",TypeX, TypeY);
             printf(" Center= %i\n",center);
             
             printf("\n Post2:    ");
             GetBatXY_K(TypeX, TypeY, BSP, la, 0.0);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             printf("\n Post3:    ");
             GetBatXY_K(TypeX, TypeY, BSP, la, lb);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             getchar();
            }
            
            if (BSP->help==320 && i==4 && j==3)
            {  
             printf("\n Element 4,3: \n");
             printf(" TypeX= %i     TypeY= %i  \n",TypeX, TypeY);
             printf(" Center= %i\n",center);
             
             printf("\n Post1:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, 0.0);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             printf("\n Post4:    ");
             GetBatXY_K(TypeX, TypeY, BSP, 0.0, lb);
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             PrintMatrixT(1,3,BU);
             
             getchar();
            }            
            
            */
            
            for (k = 0; k < 4; k++)  // Post1 bis Post4
            {    
             switch(k)
             {
              case 0: GetBatXY_K(TypeX, TypeY, BSP, 0.0, 0.0);break;
              case 1: GetBatXY_K(TypeX, TypeY, BSP,  la, 0.0);break;
              case 2: GetBatXY_K(TypeX, TypeY, BSP,  la,  lb);break;
              case 3: GetBatXY_K(TypeX, TypeY, BSP, 0.0,  lb);break;
             }
             
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             
             switch(type)
             {
              case 0: v=BU[0]; break;
              case 1: v=BU[1]; break;
              case 2: v=BU[2]; break;
              case 3: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[0]*MatFieldE[i+j*nex];break;
              case 4: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[1]*MatFieldE[i+j*nex];break;
              case 5: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[2]*MatFieldE[i+j*nex];break;
             }             
             
             switch(k)
             {
              case 0: Post1[i+j*nex]=v; break;
              case 1: Post2[i+j*nex]=v; break;
              case 2: Post3[i+j*nex]=v; break;
              case 3: Post4[i+j*nex]=v; break;
             }             
            } 
        }
    }
    free(U);
}


/*------------------------------------------------------------------------------
	TransformSplinesOnMesh3a
------------------------------------------------------------------------------*/

// type=0 means strain Exx
// type=1 means strain Eyy
// type=2 means strain Exy
// type=3 means strain Sxx
// type=4 means strain Syy
// type=5 means strain Sxy
// only for use with Sxx,Syy,Sxy related multiple phase elements

void TransformSplinesOnMesh3a(int type,   int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *KnMatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int iw1, zw;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double BU[3],CBU[3],v;
    double *U,*C,*B;

    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    C=BSP->C;
    B=BSP->B;
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;

    InitVector_double(Ksize,&U);
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                U[k]=Ux[ks];  // 1.0; //
                U[k+KsizeH]=Uy[ks];
             }  
            }
                        
            for (k = 0; k < 4; k++)  // Post1 bis Post4
            {    
             switch(k)
             {
              case 0: GetBatXY_K(TypeX, TypeY, BSP, 0.0, 0.0); zw=j;       break;
              case 1: GetBatXY_K(TypeX, TypeY, BSP,  la, 0.0); zw=j+1;     break;
              case 2: GetBatXY_K(TypeX, TypeY, BSP,  la,  lb); zw=j+nex+2; break;
              case 3: GetBatXY_K(TypeX, TypeY, BSP, 0.0,  lb); zw=j+nex+1; break;
             }
             
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             
             switch(type)
             {
              case 0: v=BU[0]; break;
              case 1: v=BU[1]; break;
              case 2: v=BU[2]; break;
              case 3: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[0]*KnMatFieldE[i+j*nex+zw];break;
              case 4: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[1]*KnMatFieldE[i+j*nex+zw];break;
              case 5: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[2]*KnMatFieldE[i+j*nex+zw];break;
             }             
             
             switch(k)
             {
              case 0: Post1[i+j*nex]=v; break;
              case 1: Post2[i+j*nex]=v; break;
              case 2: Post3[i+j*nex]=v; break;
              case 3: Post4[i+j*nex]=v; break;
             }             
            } 
        }
    }
    free(U);
}

/*------------------------------------------------------------------------------
	TransformSplinesOnMesh4   FinerSplineMesh-Auswertung der Verschiebungen
------------------------------------------------------------------------------*/

// direction=0 means x-direction
// direction=1 means y-direction

void TransformSplinesOnMesh4(int direction, int Factor,  int nex, int ney,
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             BSplineBlock *BSP,
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int ii,jj;
    int x_limit,y_limit;
    int x_pos,y_pos;
    int iw1;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double las,lbs;
    double *U,tmp[1];
    
    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;
        
    InitVector_double(Ksize,&U);
    
    las=la/(double)Factor;
    lbs=lb/(double)Factor;
    
    x_limit=Factor*nex;
    y_limit=Factor*ney;
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                if (0==direction) U[k]=Ux[ks];
                else U[k+KsizeH]=Uy[ks];
             }  
            }
            
            for (jj = 0; jj < Factor+(j+1==ney); jj++)
            for (ii = 0; ii < Factor+(i+1==nex); ii++)
            {
                GetNatXY_K(TypeX, TypeY, BSP, (ii*las), (jj*lbs));
                if (0==direction) MatrixMultMatrix(1,Ksize,1,BSP->Nx,U,tmp); 
                            else  MatrixMultMatrix(1,Ksize,1,BSP->Ny,U,tmp);
                
                x_pos=i*Factor+ii;   y_pos=j*Factor+jj;
                
                if (x_pos != x_limit && y_pos != y_limit)         
                Post1[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor]=tmp[0];
                
                if (x_pos != 0 && y_pos != y_limit)         
                Post2[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor-1]=tmp[0];
                
                if (x_pos != 0 && y_pos != 0)         
                Post3[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor-1]=tmp[0];
                
                if (x_pos != x_limit && y_pos != 0)         
                Post4[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor]=tmp[0];
            }
       }
    }
    free(U);
}


/*------------------------------------------------------------------------------
	TransformSplinesOnMesh5  FinerSplineMesh-Auswertung mit normalem E-Modul Feld
------------------------------------------------------------------------------*/

// type=0 means strain Exx
// type=1 means strain Eyy
// type=2 means strain Exy
// type=3 means strain Sxx
// type=4 means strain Syy
// type=5 means strain Sxy


void TransformSplinesOnMesh5(int type, int Factor,  int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *MatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int ii,jj;
    int iw1;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double las,lbs;
    double BU[3],CBU[3],v;
    double *U,*C,*B;

    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    C=BSP->C;
    B=BSP->B;
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;

    InitVector_double(Ksize,&U);
    
    las=la/(double)Factor;
    lbs=lb/(double)Factor;
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                U[k]=Ux[ks];  // 1.0; //
                U[k+KsizeH]=Uy[ks];
             }  
            }
            
            for (jj = 0; jj <= Factor; jj++)
            for (ii = 0; ii <= Factor; ii++)
            {
             GetBatXY_K(TypeX, TypeY, BSP, (ii*las), (jj*lbs));
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
            
             switch(type)
             {
              case 0: v=BU[0]; break;
              case 1: v=BU[1]; break;
              case 2: v=BU[2]; break;
              case 3: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[0]*MatFieldE[i+j*nex];break;
              case 4: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[1]*MatFieldE[i+j*nex];break;
              case 5: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[2]*MatFieldE[i+j*nex];break;
             }            
            
             if (ii != Factor && jj != Factor)         
             Post1[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor]=v;
                
             if (ii != 0 && jj != Factor)         
             Post2[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor-1]=v;
                
             if (ii != 0 && jj != 0)         
             Post3[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor-1]=v;
                
             if (ii != Factor && jj != 0)         
             Post4[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor]=v;
            } 
        }
    }
    free(U);
}


/*------------------------------------------------------------------------------
	TransformSplinesOnMesh5a  FinerSplineMesh-Auswertung mit multi-phase E-Modul Feld
------------------------------------------------------------------------------*/

// type=0 means strain Exx
// type=1 means strain Eyy
// type=2 means strain Exy
// type=3 means strain Sxx
// type=4 means strain Syy
// type=5 means strain Sxy


void TransformSplinesOnMesh5a(int type, int Factor,  int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *KnMatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4)
{
    int i,j,k,ks,kx,ky;
    int ii,jj;
    int iw1;
    int TypeX,TypeY;
    int orderK;
    int center;
    int Ksize, KsizeH;
    double x,y,las,lbs;
    double BU[3],CBU[3],v;
    double E, E1, E2, E3, E4;
    double *U,*C,*B;
    double CircleIntpntFlag;
    AllWinStruct *AllWinData;
    double xpos, ypos;   
    
    PutGetAllWinStruct(1, &AllWinData);
    
    CircleIntpntFlag=AllWinData->SolveWinData->CircleIntpntFlag;
    
    orderK=BSP->orderK;
    center=BSP->SplineElemsX.center;
    C=BSP->C;
    B=BSP->B;
    
    KsizeH=(orderK+1)*(orderK+1);
    Ksize=2*KsizeH;

    InitVector_double(Ksize,&U);
    
    las=la/(double)Factor;
    lbs=lb/(double)Factor;
    
    for (j = 0; j < ney; j++)
    {
        if (j <= center) TypeY=(j-center); // welcher Elementtyp y-Richtung
        else
        {
         iw1=ney-1-center;
         if (j >= iw1) TypeY=(j-iw1);
         else TypeY=0;   
        }        
        
        for (i = 0; i < nex; i++)
        {    
            if (i <= center) TypeX=(i-center); // welcher Elementtyp x-Richtung
            else
            {
             iw1=nex-1-center;
             if (i >= iw1) TypeX=(i-iw1);
             else TypeX=0;   
            }
            
            for (ky = 0; ky < (orderK+1); ky++)  //U global zu U lokal
            {
             for (kx = 0; kx < (orderK+1); kx++)
             {    
                k = kx + (orderK+1)*ky;
                ks=(i+Spnnx*j) + (kx + Spnnx*ky);
                
                U[k]=Ux[ks];  // 1.0; //
                U[k+KsizeH]=Uy[ks];
             }  
            }
            
            E1=KnMatFieldE[i+j*nex+j];
            E2=KnMatFieldE[i+j*nex+j+1];
            E3=KnMatFieldE[i+j*nex+j+nex+2];
            E4=KnMatFieldE[i+j*nex+j+nex+1];
            
            for (jj = 0; jj <= Factor; jj++)
            for (ii = 0; ii <= Factor; ii++)
            {
             GetBatXY_K(TypeX, TypeY, BSP, (ii*las), (jj*lbs));
             MatrixMultMatrix(3,Ksize,1,B,U,BU);
             
             x=(double)ii/(double)Factor;
             y=(double)jj/(double)Factor;
             E=y*(x*(E3-E4-E2+E1)+E4-E1)+x*(E2-E1)+E1;
             
             if (1==CircleIntpntFlag)
             {
                 xpos=la*( ((double) (i)) + x);
                 ypos=lb*( ((double) (j)) + y);  
                 CircleFunction(xpos,ypos,&E);          
             }
             
             switch(type)
             {
              case 0: v=BU[0]; break;
              case 1: v=BU[1]; break;
              case 2: v=BU[2]; break;
              case 3: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[0]*E; break;
              case 4: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[1]*E; break;
              case 5: MatrixMultMatrix(3,3,1,C,BU,CBU); v=CBU[2]*E; break;
             }            
            
             if (ii != Factor && jj != Factor)         
             Post1[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor]=v;
                
             if (ii != 0 && jj != Factor)         
             Post2[i*Factor+j*nex*Factor*Factor+ii+jj*nex*Factor-1]=v;
                
             if (ii != 0 && jj != 0)         
             Post3[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor-1]=v;
                
             if (ii != Factor && jj != 0)         
             Post4[i*Factor+j*nex*Factor*Factor+ii+(jj-1)*nex*Factor]=v;
            } 
        }
    }
    free(U);
}


/*------------------------------------------------------------------------------
	 createKnMatFieldE
------------------------------------------------------------------------------*/

void createKnMatFieldE(MGBlock *MG)
{
    int i,j;
    int nnx,nny;
    int nex,ney;
    double zw, factor;
    
    double *E_elems;
    double *E_nodes;
    
    E_elems=MG->MatFieldE;
    E_nodes=MG->KnMatFieldE;
    
    nnx=MG->nnx;
    nny=MG->nny;
    nex=MG->nex;
    ney=MG->ney;
    
    SetVectorToZero_double(nnx*nny,  E_nodes);
    
    for (j = 0; j < ney; j++)
    {
      for (i = 0; i < nex; i++)
      {
        zw=E_elems[i+j*nex];
        E_nodes[ i   + j   *nnx]+=zw; 
        E_nodes[(i+1)+ j   *nnx]+=zw; 
        E_nodes[(i+1)+(j+1)*nnx]+=zw; 
        E_nodes[ i   +(j+1)*nnx]+=zw;    
      }
    }
    
    for (j = 0; j < nny; j++)
    {
      for (i = 0; i < nnx; i++)
      {
        factor=0.25;
        if (0==i || (nnx-1)==i) factor*=2.0;
        if (0==j || (nny-1)==j) factor*=2.0;
        E_nodes[i+j*nnx]*=factor;    
      }
    }
    
    /*// Testausdruck (mit niedriger Elementanzahl durchfhren)
    printf("\n Element E-Modul    ");
    PrintMatrixT(nex,ney,E_elems);
    
    printf("\n Knoten E-Modul    ");
    PrintMatrixT(nnx,nny,E_nodes);    
    
    getchar();
    //*/
}

/*------------------------------------------------------------------------------
	 createKnMatFieldCircle
------------------------------------------------------------------------------*/

void createKnMatFieldCircle()
{
    int i,j;
    int nnx,nny;
    int nex,ney;
    int num_levels;
    int type;
    double la,lb;
    double zw, factor;
    
    double *E_elems;
    double *E_nodes;
    double xpos,ypos;
    double radius;
    double EModul;
    
   MGBlock *MG;
   SolveBlock *SolBlock;
    
   PutGetSolveBlock(1,&SolBlock); 
    
   num_levels=SolBlock->num_levels;
   MG=&(SolBlock->Meshes[num_levels-1]); 
    
    E_elems=MG->MatFieldE;
    E_nodes=MG->KnMatFieldE;
    
    nnx=MG->nnx;
    nny=MG->nny;
    nex=MG->nex;
    ney=MG->ney;
    la=MG->la;
    lb=MG->lb;
    
    printf("nnx: %i",nnx);
    
    SetVectorToZero_double(nnx*nny,  E_nodes);
       
    for (j = 0; j < nny; j++)
    {
      for (i = 0; i < nnx; i++)
      {
        xpos=((double)i)*la;
        ypos=((double)j)*lb;
        CircleFunction(xpos, ypos, &EModul);
        E_nodes[i+j*nnx]=EModul;
      }
    }
    
    //memcpy(MG->Fx,E_nodes,(nnx*nny)*sizeof(double));
    for (j = 0; j < ney; j++)
    {
      for (i = 0; i < nex; i++)
      {
        E_elems[i+j*nex]=0.25*(E_nodes[i    +    j*nnx]
                              +E_nodes[(i+1)+    j*nnx]
                              +E_nodes[(i+1)+(j+1)*nnx]
                              +E_nodes[i    +(j+1)*nnx]);
      }
    }
    
    
    
    return;
}

/*------------------------------------------------------------------------------
	 CircleFunction
------------------------------------------------------------------------------*/

void CircleFunction(double xpos, double ypos, double *EModul)
{
    
    double distance;
    double x,y;
    
    distance=sqrt( (xpos-64.0)*(xpos-64.0) + (ypos-64.0)*(ypos-64.0) );
    
    if (distance>=5) {(*EModul)=100000.0; return;}
    if (distance<=3) {(*EModul)=200000.0; return;}
    (*EModul)=(3-distance)*50000.0+200000.0; return;
    
    /*
    if (distance>=10.0) {(*EModul)=100000.0; return;}
    if (distance<=6.0) {(*EModul)=200000.0; return;}
    x=(distance-6.0)/4.0;
    y=(-2.0*x+3.0)*x*x;
    (*EModul)=-y*100000.0+200000.0; return;  
    */
    /*
    if (xpos>=60.0) {(*EModul)=200000.0; return;}
    if (xpos<=40.0) {(*EModul)=100000.0; return;}
    x=(xpos-40.0)/20.0;
    y=(-2.0*x+3.0)*x*x;
    y=x;
    (*EModul)=y*100000.0+100000.0; return;
    */
    /*
    if (xpos>=56.25) {(*EModul)=200000.0; return;}
    if (xpos<=43.75) {(*EModul)=100000.0; return;}
    x=(xpos-43.75)/(4.0*3.125);
    y=(-2.0*x+3.0)*x*x;
    y=x;
    (*EModul)=y*100000.0+100000.0; return;
    */
    if (xpos>=72.0) {(*EModul)=200000.0; return;}
    if (xpos<=56.0) {(*EModul)=100000.0; return;}
    x=(xpos-56.0)/(16.0);
    y=(-2.0*x+3.0)*x*x;
    y=x;
    (*EModul)=y*100000.0+100000.0; return;    
    
}



