/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*********************************************************************************
                            Basic routines
********************************************************************************/

/*-----------------------------------------------------------------------------
	InitVector_int
-------------------------------------------------------------------------------*/
int InitVector_int(int nn, int **M)
{
    *M = (int *) malloc(sizeof(int)*(nn));
    memset((*M), 0, nn*sizeof(int));
    return(1); 
}



/*-----------------------------------------------------------------------------
	InitVector_double
-------------------------------------------------------------------------------*/
int InitVector_double(int nn, double **M)
{
    *M = (double *) malloc(sizeof(double)*(nn));
    memset((*M), 0, nn*sizeof(double));
    return(1); 
}

/*-----------------------------------------------------------------------------
	InitVector_float
-------------------------------------------------------------------------------*/
int InitVector_float(int nn, float **M)
{
    *M = (float *) malloc(sizeof(float)*(nn));
    memset((*M), 0, nn*sizeof(float));
    return(1); 
}

/*-----------------------------------------------------------------------------
	InitVector_char
-------------------------------------------------------------------------------*/
int InitVector_char(int nn, unsigned char **M)
{
    *M = (unsigned char *) malloc(sizeof(unsigned char)*(nn));
    memset((*M), '0', nn*sizeof(char));
    return(1); 
}



/*-----------------------------------------------------------------------------
	SetVectorToZero_int
-------------------------------------------------------------------------------*/
int SetVectorToZero_int(int nn,  int *M)
{   
    memset(M, 0, nn*sizeof(int));
    return(1);
}



/*-----------------------------------------------------------------------------
	SetVectorToZero_double
-------------------------------------------------------------------------------*/
int SetVectorToZero_double(int nn,  double *M)
{
    memset(M, 0, nn*sizeof(double));
    return(1);
}





/*-------------------------------------------------------------------------------
	InitializeUx
-------------------------------------------------------------------------------*/

int InitializeUx(int nnx, int nny, double *Ux)
{
  int i;
  int j;
  double step;
   
    step = 1.0/(nnx-1.0);
    for(j = 0;j < nny;j++)
        for(i = 0;i < nnx;i++)
            Ux[i+j*nnx] = step * i; 
    
    return(1);
}
/*-------------------------------------------------------------------------------
	StartVectorZero
-------------------------------------------------------------------------------*/

int StartVectorZero(int nnx, int nny, double *Ux)
{
  int i;
  int j;
  double step;
   
    step = 1.0/(nnx-1.0);
    for(j = 1;j < (nny-1);j++)
        for(i = 1;i < (nnx-1);i++)
            Ux[i+j*nnx] = 0; 
    
    return(1);
}

/*-------------------------------------------------------------------------------
	InitializeUy
-------------------------------------------------------------------------------*/

int InitializeUy(int nnx, int nny, double *Uy)
{
  int i;
  int j;
  double step;
   
    step = 1.0/(nny-1.0);
    for(j = 0;j < nny;j++)
        for(i = 0;i < nnx;i++)
            Uy[i+j*nnx] = step * j; 
    
    return(1);
}


/*---------------------------------------------------------
	ScalarProduct
----------------------------------------------------------*/

int ScalarProduct(int n, double *v1, double *v2, double *alpha)
{
    int i;
    double tmp;
    double *V1,*V2;
    int n1, n2, t;
    
    tmp=0.0;
    V1=v1;
    V2=v2;
    
    for(i=0; i<n; i++)    
    {
      tmp+=(*V1)*(*V2);V1++;V2++; 
    }
    
    (*alpha)=tmp;
    return(1);
}

/*---------------------------------------------------------
	VectorMult
----------------------------------------------------------*/

int VectorMult(int n, double *v1, double *v2, double *v3)
{
    int i;
    double tmp;
    double *V1,*V2,*V3;
    

    V1=v1;
    V2=v2;
    V3=v3;
    
    for(i=0; i<n; i++)    
    {
      V3[i]=V1[i]*V2[i];
    }
    
   
    return(1);
}

/*--------------------------------------------------------------------------------
	FindMinMaxVector
--------------------------------------------------------------------------------*/
int FindMinMaxVector(int length, double *vector, double *minimum, double *maximum)
{
   int i; 
   double min,max;
   
   min=vector[0];
   max=vector[0];
   
   for(i = 0;i < length;i++)
   {
	if (vector[i]<min) min=vector[i];
	if (vector[i]>max) max=vector[i];
   }
   
   *minimum=min;
   *maximum=max;

return(1);
}

/*------------------------------------------------------------------------------
	PlotDoubleMatrix
-------------------------------------------------------------------------------*/
#if 0
int PlotDoubleMatrix(int d1, int d2, double winx, double winy, double min, double max, 
                     MGBlock *Mesh, double *PlotMatrix, int *colors)
{
    int i,j;
    int nex,ney;
    double la,lb;
    double *M;
    double diff;
    int mycolor1;
    
    la=Mesh->la;
    lb=Mesh->lb;
    nex=Mesh->nnx;
    ney=Mesh->nny;
    M=PlotMatrix;
    
     
    diff=max-min;
    
    for(i = 0; i < nex; i++)
    {
        for(j = 0; j < ney; j++)
        {
       	    mycolor1=floor((colors[0]-2)/diff*(M[i+nex*j]-min))+1;
	    	    
	    g2_pen(d1, colors[mycolor1]);
	    g2_filled_rectangle(d1, (i)*la, (j)*lb, (i+1)*la,(j+1)*lb) ;
        }
    } 
   
   return(1);
}
#endif

/*-------------------------------------------------------------------------------
	PrintDoubleMatrix
---------------------------------------------------------------------------------*/
int PrintDoubleMatrix(MGBlock *Mesh)
{
    int i,j;
    int nex,ney;
    double *M;
    
    nex=Mesh->nex;
    ney=Mesh->ney;
    M=Mesh->MatFieldE;
    
    printf("\n\n");
    
    for (i=0; i<nex; i++)
    {	
	for (j=0; j<ney; j++)
	{   
	    printf(" %6.2f",M[i+j*nex]);
	}   
	printf("\n");
    }
    
    printf("\n");
    return (1);
}


/*-------------------------------------------------------------------------------
	PrintMatrix
---------------------------------------------------------------------------------*/
int PrintMatrix(int m, int n, double *M)
{
    int i,j;
       
    printf("\n");
    for(j=0; j<n; j++)
    { 
        printf("\n");
        for(i=0; i<m; i++)
        {
             printf("%9.5f ",M[i+j*m]);
        }
    }
	
    return (1);
}

/*-------------------------------------------------------------------------------
	PrintMatrixInteger
---------------------------------------------------------------------------------*/
int PrintMatrixInteger(int m, int n, int *M)
{
    int i,j;
       
    printf("\n");
    for(j=0; j<n; j++)
    { 
        printf("\n");
        for(i=0; i<m; i++)
        {
             printf("%i ",M[i+j*m]);
        }
    }
	
    return (1);
}


/*------------------------------------------------------------------------------
	PrintMatrixT   !abgendert -a10-!
------------------------------------------------------------------------------*/
int PrintMatrixT(int row, int col, double *M)
{
    int i,j;
       
    printf("\n");
    for(i=0; i<row; i++)
    { 
        printf("\n");
        for(j=0; j<col; j++)
        {
             printf("%2.2f ",M[i+j*row]);
             //printf("%9.5f ",M[i+j*row]);
             //printf("%12.11f ",M[i+j*row]);
        }
    }
	
    return (1);
}


/*------------------------------------------------------------------------------
    PrintMatrixToFile  !neu -n41-!
------------------------------------------------------------------------------*/
int PrintMatrixToFile(int row, int col, double *M, char name[20])
{
    int i,j; 
    FILE *outp ;
    
    printf("\n\n.......Print Matrix to File %s.......\n",name);
    //outp = fopen("./matrix.txt","wb");   
    outp = fopen(name,"wb");  
    
    if (outp == NULL)  
   	{   
  		printf("Error: cannot open the file ");
   	    getchar();
   	    return(0);
   	}
    else
    {        
        for(i=0; i<row; i++)
        { 
            
            for(j=0; j<col; j++)
            {
                 //fprintf(outp,"%g",M[i+j*row]);
                 fprintf(outp,"%.13g",M[i+j*row]);
                 //fprintf(outp,"%9.5f",M[i+j*row]);
                 //fprintf(outp,"%12.11f",M[i+j*row]); 
                 if (j==col-1) fprintf(outp," \n");
                 else fprintf(outp,"\t");              
            }
        }
        fclose(outp);   
    }
    return(1);
}


/*-------------------------------------------------------------------------------
	PrintMatrixIntegerT  !abgendert -a11-!
---------------------------------------------------------------------------------*/
int PrintMatrixIntegerT(int row, int col, int *M)
{
    int i,j;
       
    printf("\n");
    for(i=0; i<row; i++)
    { 
        printf("\n");
        for(j=0; j<col; j++)
        {
             printf("%i ",M[i+j*row]);
        }
    }
	
    return (1);
}


/*------------------------------------------------------------------------------
	MatrixMultScalar
------------------------------------------------------------------------------*/

void MatrixMultScalar(int row,int col, double *M, double S)
{
   int m,n;
   
   for(m=0; m<row; m++)
   for(n=0; n<col; n++)
   M[m+row*n]*=S;  
   
   return;    
}


/*------------------------------------------------------------------------------
	MyControl
------------------------------------------------------------------------------*/
int MyControl(void)
{
    int Frames;
    double RelError;
    int errflag1,errflag2;

    errflag1=0;
    errflag2=0;
    GraphicBlock *graphblock;
    SolveBlock *SolBlock;
    
    ChangeTimeFlag(11);
    PutGetGraphicBlock(1, &graphblock);
    (graphblock->framecount)++;
    (graphblock->RunFlag)=0;
     if (1==(graphblock->TMP)) {(graphblock->RunFlag)=1;(graphblock->TMP)=0;}
    
    if((graphblock->framecount) >= (graphblock->Frames))
    {   
        PutGetSolveBlock(1, &SolBlock); 
        (graphblock->framecount)=0;
        

        ChangeTimeFlag(6);        
        if (SolBlock->ErrorData.ErrFlag[0]) {errflag2+=1; errflag1+=ComputeEuclideanNorm();}
        if (SolBlock->ErrorData.ErrFlag[1]) {errflag2+=1; errflag1+=ComputeRelEuclideanError();}
        if (SolBlock->ErrorData.ErrFlag[2]) {errflag2+=1; errflag1+=ComputeRelEnergyError();}
        if (errflag1==errflag2 && errflag2!=0){graphblock->RunFlag=3;}
        //printf("FLAG1: %i FLAG2:= %i", errflag1, errflag2);
        ChangeTimeFlag(11);
        PrintErrorData();
        PrintTimeData();
      
      //RelError=PostErrorEnergy()/PostSystemEnergy();
        //printf("%i   error:%g  time[s]:%g  Rel. Error %g\n",SolBlock->i, SolBlock->err2, mytime, RelError);
        //printf("\nRel Error %g???\n", RelError);
        //if(RelError<1E-06) return(1);
        
        //PostMainLoop();
        graphblock->NewObjectFlag=1; 
        UpdateGraphicWindow(); 
            
        if (0!=(graphblock->RunFlag)) 
        {
            if (1==(graphblock->RunFlag)) {PrintCheckTimeData();MyPause();} //PAUSE
            if (3==(graphblock->RunFlag)) { //return 1;} //STOP
            SaveTwoValues((double)SolBlock->i,0.0);return 1;} //STOP
            if (2==(graphblock->RunFlag)) {graphblock->RunFlag=1;} //STEP
        }
   }
   return 0;
}



/*------------------------------------------------------------------------------
	MyPause
------------------------------------------------------------------------------*/
void MyPause()
{
    GraphicBlock *graphblock;
    PutGetGraphicBlock(1, &graphblock);
    while(1==(graphblock->RunFlag))
    {
       Fl::wait();
    }
}


/*------------------------------------------------------------------------------
     OpenMyWindow
------------------------------------------------------------------------------*/
#if 0
int OpenMyWindow(int xsize, int ysize, int *device_num1, int *device_num2)
{
    int d1,d2;
    d2=g2_open_X11(xsize,ysize); 
    d1=g2_open_vd();			
    g2_attach(d1, d2);
    g2_set_auto_flush(d1,0); 
    (*device_num1)=d1;
    (*device_num2)=d2;		
    return(1); 
}
#endif

/*------------------------------------------------------------------------------
     SaveTwoValues
------------------------------------------------------------------------------*/
void SaveTwoValues(double v1, double v2) 
{   
    int i;
    FILE    *outp ;
    char filename[30];
       
        
    strcpy(filename,"test2.dat");
    outp = fopen(filename,"a");  
    
   	if (outp == NULL)  
   	{   
  		printf("Error: cannot open the file ");
   	    //getchar();
   	}
 
 	else 
	{   
	    
   		fprintf(outp,"%g   %g\n",v1,v2);	
        fclose(outp);
	}
}


/*------------------------------------------------------------------------------
	GaussSeidelGLSloeser    liefert x als Lsung von A*x=b    !neu -n34-! 
	
	Rckmeldung: 0 - korrekt gelst mit weniger als 1.0E-12 Schritten
	             1 - ein Diagonalelement von A ist gleich Null
	             2 - mit 1.0E10 Schritten wurde die Fehlerschranke
	                 nicht unterschritten
------------------------------------------------------------------------------*/

int GaussSeidelGLSloeser(double *A, double *x, double *b, int GLsize)
{
    int i,j,k;
    double w1, w2, w3;
    double *H, *R;
    InitVector_double(GLsize,&H);
    InitVector_double(GLsize,&R);
    /* Kontrollausgabe
    printf("\n\n Matrix A: \n");
    PrintMatrixT(GLsize,GLsize,A);
    
    printf("\n\n Vektor b: \n");
    PrintMatrixT(GLsize,1,b);    
    */
    
    for(i=0; i<GLsize; i++)
    { 
     if (0.0==A[i+GLsize*i]) return 1;
    }
    
    k=0;
    w3=1;
    
    while(w3>1.0E-12)
    {
        for(i=0; i<GLsize; i++)  //solver
        {
            w1=A[i+GLsize*i];
            x[i]=b[i]/w1;
                    
            for(j=0; j<GLsize; j++)
            {     
             if (i!=j) x[i]-=A[i+GLsize*j]*x[j]/w1;
            }
            
        }
        k++;
        
        MatrixMultMatrix(GLsize,GLsize,1,A,x,H);  //Fehlerkriterium
        MatrixMultScalar(GLsize,1,H,-1.0);
        memcpy(R, b, sizeof(double)*GLsize);
        AddMatrix(GLsize,1,R,H);
        ScalarProduct(GLsize, R, R, &w3);
        w3=sqrt(w3);
        
        /*/ Kontrollausgabe
        printf("\n\n Durchgang %i , delta = %9.5f , Vektor x: \n",k,w3);
        PrintMatrixT(GLsize,1,x);
        getchar();
        //*/  
              
        if (k>(int)1.0E10) return 2;
    }
    free(H);
    free(R);
    return 0;
}
