/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*---------------------------------------------------------
	CGSolver
----------------------------------------------------------*/

void CGSolver(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes)
{
int i, j, iii, length;
double tmp1, tmp2, tmp3;
double err1;

        GetRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
                M->MatFieldInt, SolBlock->Material,
                M->Ux,  M->Uy, M->Fx, M->Fy, 
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 

        length=(M->nnx)*(M->nny);
        memcpy(M->Dx, M->Fx, sizeof(double)*length);
        memcpy(M->Dy, M->Fy, sizeof(double)*length);
        
        for(i=0; i<100000; i++)    
        {
            
            ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->Fx, &tmp1);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->Fy, &tmp2);
            tmp1=tmp1+tmp2; 
            
            GetRestoringForceIncrement2(
              M->nnx, M->nny,M->MatFieldE, 
              M->MatFieldInt, SolBlock->Material,
        	  M->Dx,  M->Dy, M->Adx, M->Ady, 
              M->BcTypeX,
              M->BcTypeY, M->BcValueX, M->BcValueY);
            
        	
            ScalarProduct(((M->nnx)*(M->nny)), M->Dx, M->Adx, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Dy, M->Ady, &tmp3);
            tmp2=(tmp2+tmp3);   
            
            tmp3=tmp1/tmp2; //tmp3 entspricht alpha
            
            for(j=0; j<length; j++) 
            {
                (M->Ux[j])=(M->Ux[j])+tmp3*(M->Dx[j]);
                (M->Uy[j])=(M->Uy[j])+tmp3*(M->Dy[j]);
            
                (M->Fx[j])=(M->Fx[j])-tmp3*(M->Adx[j]);
                (M->Fy[j])=(M->Fy[j])-tmp3*(M->Ady[j]);
            }
        	  
       	    ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->Fx, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->Fy, &tmp3);
            tmp2=tmp2+tmp3; 
            
            tmp3=tmp2/tmp1; //tmp3 entspricht beta
            
            for(j=0; j<length; j++) 
            {
                (M->Dx[j])=(M->Fx[j])+tmp3*(M->Dx[j]);
                (M->Dy[j])=(M->Fy[j])+tmp3*(M->Dy[j]);
            }   
            

            SolBlock->i=i; 
            iii=MyControl();
            if (1==iii) return;
        }
   return;    
}
/*---------------------------------------------------------
	PCGSolver
----------------------------------------------------------*/

void PCGSolver(double *K_matrix, SolveBlock *SolBlock, MGBlock *M)
{
int i, j, iii, length;
double tmp1, tmp2, tmp3;
double err1; 
double Omega;
       Omega=1.0;
               GetRestoringForce(M->nnx, M->nny, K_matrix, M->MatFieldE, 
        	  M->Ux,  M->Uy, M->Fx,  M->Fy, &err1, &err1, 
              M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY);
        length=(M->nnx)*(M->nny);
        //memcpy(M->Dx, M->Fx, sizeof(double)*length);
        //memcpy(M->Dy, M->Fy, sizeof(double)*length);
        JacobiPreconditioner(M->nnx, M->nny, K_matrix, M->MatFieldE, M->Mpc);
        VectorMult(length, M->Fx, M->Mpc, M->Dx);
        VectorMult(length, M->Fy, M->Mpc, M->Dy);
        
        for(i=0; i<100000; i++)    
        {
            //GetRestoringForce(M->nnx, M->nny, K_matrix, M->MatFieldE, 
        	//  M->Ux,  M->Uy, M->Fx,  M->Fy, &err1, &err1, 
            //  M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->dUy);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->dUx);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->Adx);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->Ady);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->SwapX);
            SetVectorToZero_double((M->nnx)*(M->nny),  M->SwapY);
            
            for(j=0; j<3; j++)    
            GetLocalDisplacements_GS_real(M->nnx, M->nny, K_matrix, M->MatFieldE, 
            M->Adx, M->Ady, M->dUx,  M->dUy, M->SwapX, M->SwapY, &err1,&err1, Omega,
            M->BcTypeX, M->BcTypeY, M->Fx, M->Fy); 
            
            //VectorMult(length, M->Fx, M->Mpc, M->dUx);
            //VectorMult(length, M->Fy, M->Mpc, M->dUy);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->dUx, &tmp1);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->dUy, &tmp2);
            tmp1=tmp1+tmp2; 
            
            GetRestoringForceIncrement(M->nnx, M->nny, K_matrix, M->MatFieldE, 
        	  M->Dx,  M->Dy, M->Adx, M->Ady, &err1, &err1, M->BcTypeX,
              M->BcTypeY, M->BcValueX, M->BcValueY);
        	
            ScalarProduct(((M->nnx)*(M->nny)), M->Dx, M->Adx, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Dy, M->Ady, &tmp3);
            tmp2=tmp2+tmp3;   
            
            tmp3=tmp1/tmp2; //tmp3 entspricht alpha
            
            for(j=0; j<length; j++) 
            {
                (M->Ux[j])=(M->Ux[j])+tmp3*(M->Dx[j]);
                (M->Uy[j])=(M->Uy[j])+tmp3*(M->Dy[j]);
            
                (M->Fx[j])=(M->Fx[j])-tmp3*(M->Adx[j]);
                (M->Fy[j])=(M->Fy[j])-tmp3*(M->Ady[j]);
            }
        	
            //VectorMult(length, M->Fx, M->Mpc, M->dUx);
            //VectorMult(length, M->Fy, M->Mpc, M->dUy);  
       	    ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->dUx, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->dUy, &tmp3);
            tmp2=tmp2+tmp3; 
            
            tmp3=tmp2/tmp1; //tmp3 entspricht beta
            
            for(j=0; j<length; j++) 
            {
                (M->Dx[j])=(M->dUx[j])+tmp3*(M->Dx[j]);
                (M->Dy[j])=(M->dUy[j])+tmp3*(M->Dy[j]);
            }   
            
            
            /* P=&(Meshes[num_levels-2]);         
            GetLocalDisplacements_GS_Q9(M->nnx, M->nny, K_matrixQ9, P->MatFieldE, 
                M->dUx, M->dUy, M->Ux,  M->Uy, M->Fx, M->Fy, &err1,&err1, Omega,
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
            TransForceOnCoarseMeshQ4onQ9(M->nnx, M->nny, M->Fx, M->Fy,
                              P->nnx, P->nny, P->Fx, P->Fy);
            TransForceOnCoarseMesh(M->nnx, M->nny, M->Fx, M->Fy,
                              P->nnx, P->nny, P->SwapX, P->SwapY);
            
            SolBlock->err2=err2; */
            SolBlock->i=i; 
            iii=MyControl();
            if (1==iii) return;
        }
 return;
}
    
    

/*---------------------------------------------------------
	JacobiPreconditioner
----------------------------------------------------------*/

int JacobiPreconditioner(int nnx, int nny, double *K, double *MatFieldE, double *Mpc)
{
    register int i,j;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double tmp;

    nex=nnx-1;
    ney=nny-1;
    tmp=0.0;
      
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
         
         
         }
         Esum=E1+E2+E3+E4;
         tmp+=Esum;
         
         }
    }
    tmp=tmp/(nnx*nny);
    
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
         
         
         }
         Esum=E1+E2+E3+E4;
         Mpc[i+j*nnx]=1.0;///(K[0]*Esum);
         
         }
    }
   
    return(1);
}


/*---------------------------------------------------------
	CGSolver3d
----------------------------------------------------------*/

void CGSolver3d(SolveBlock *SolBlock, MGBlock *M)
{
int i, j, iii, length;
double tmp1, tmp2, tmp3, tmp4;
double err1;

        GetRestoringForce3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        	  M->MatFieldInt, M->Ux,  M->Uy, M->Uz, M->Fx,  M->Fy, M->Fz,
              M->BcTypeX, M->BcTypeY, M->BcTypeZ, M->BcValueX, M->BcValueY, M->BcValueZ);
        length=(M->nnx)*(M->nny)*(M->nnz);
        memcpy(M->Dx, M->Fx, sizeof(double)*length);
        memcpy(M->Dy, M->Fy, sizeof(double)*length);
        memcpy(M->Dz, M->Fz, sizeof(double)*length);
        
        for(i=0; i<100000; i++)    
        {
            
            ScalarProduct(length, M->Fx, M->Fx, &tmp1);
            ScalarProduct(length,  M->Fy, M->Fy, &tmp2);
            tmp1=tmp1+tmp2;
            ScalarProduct(length,  M->Fz, M->Fz, &tmp2); 
            tmp1=tmp1+tmp2;
            printf("tmp1: %f\n",tmp1);
            
            GetRestoringForceIncrement3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        	  M->MatFieldInt,  M->Dx,  M->Dy, M->Dz, M->Adx, M->Ady, M->Adz,
              M->BcTypeX, M->BcTypeY, M->BcTypeZ, M->BcValueX, M->BcValueY, M->BcValueZ);
        	
            ScalarProduct(length, M->Dx, M->Adx, &tmp2);
            ScalarProduct(length, M->Dy, M->Ady, &tmp3);
            ScalarProduct(length, M->Dz, M->Adz, &tmp4);
            tmp2=-(tmp2+tmp3+tmp4); 
            printf("tmp2: %f\n",tmp2);   
            
            tmp3=tmp1/tmp2; //tmp3 entspricht alpha
            printf("alpha: %f\n",tmp3);
            
            for(j=0; j<length; j++) 
            {
                (M->Ux[j])=(M->Ux[j])+tmp3*(M->Dx[j]);
                (M->Uy[j])=(M->Uy[j])+tmp3*(M->Dy[j]);
                (M->Uz[j])=(M->Uz[j])+tmp3*(M->Dz[j]);
            
                (M->Fx[j])=(M->Fx[j])+tmp3*(M->Adx[j]);
                (M->Fy[j])=(M->Fy[j])+tmp3*(M->Ady[j]);
                (M->Fz[j])=(M->Fz[j])+tmp3*(M->Adz[j]);
            }
        	  
       	    ScalarProduct(length, M->Fx, M->Fx, &tmp2);
            ScalarProduct(length, M->Fy, M->Fy, &tmp3);
            ScalarProduct(length, M->Fz, M->Fz, &tmp4);
            tmp2=tmp2+tmp3+tmp4; 
            
            tmp3=tmp2/tmp1; //tmp3 entspricht beta
            
            for(j=0; j<length; j++) 
            {
                (M->Dx[j])=(M->Fx[j])+tmp3*(M->Dx[j]);
                (M->Dy[j])=(M->Fy[j])+tmp3*(M->Dy[j]);
                (M->Dz[j])=(M->Fz[j])+tmp3*(M->Dz[j]);
            }   
            
            
            /* P=&(Meshes[num_levels-2]);         
            GetLocalDisplacements_GS_Q9(M->nnx, M->nny, K_matrixQ9, P->MatFieldE, 
                M->dUx, M->dUy, M->Ux,  M->Uy, M->Fx, M->Fy, &err1,&err1, Omega,
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
            TransForceOnCoarseMeshQ4onQ9(M->nnx, M->nny, M->Fx, M->Fy,
                              P->nnx, P->nny, P->Fx, P->Fy);
            TransForceOnCoarseMesh(M->nnx, M->nny, M->Fx, M->Fy,
                              P->nnx, P->nny, P->SwapX, P->SwapY);
            
            SolBlock->err2=err2; */
            SolBlock->i=i; 
            iii=MyControl();
            if (1==iii) return;
        }
   return;    
}

/*---------------------------------------------------------
	GetRestoringForce3d
----------------------------------------------------------*/

int GetRestoringForce3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double *BcValueX, double *BcValueY, double *BcValueZ)
{
    int i,j,k,m;
    int nex, ney, nez, nns;
    double E;
    double U[24],F[24];
    double *K;
    int p;
    
    nex=nnx-1;
    ney=nny-1;
    nez=nnz-1;
    
    SetVectorToZero_double((nnx*nny*nnz), Fx);
    SetVectorToZero_double((nnx*nny*nnz), Fy);
    SetVectorToZero_double((nnx*nny*nnz), Fz);
    
    for (k = 0; k < nez; k++) 
    for (j = 0; j < ney; j++) 
    {     
        for (i = 0; i < nex; i++)   
        {
            p=i+nex*j+nex*ney*k;
            K=Material[MatFieldInt[p]].K;
            
            p=i+nnx*j+nnx*nny*(k);
            U[0]=Ux[p];
            U[1]=Uy[p];
            U[2]=Uz[p];
            
            p++;
            U[3]=Ux[p];
            U[4]=Uy[p];
            U[5]=Uz[p];
            
            p+=nnx;
            U[6]=Ux[p];
            U[7]=Uy[p];
            U[8]=Uz[p];
            
            p--;
            U[9]=Ux[p];
            U[10]=Uy[p];
            U[11]=Uz[p];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            U[12]=Ux[p];
            U[13]=Uy[p];
            U[14]=Uz[p];
            
            p++;
            U[15]=Ux[p];
            U[16]=Uy[p];
            U[17]=Uz[p];
            
            p+=nnx;
            U[18]=Ux[p];
            U[19]=Uy[p];
            U[20]=Uz[p];
            
            p--;
            U[21]=Ux[p];
            U[22]=Uy[p];
            U[23]=Uz[p];
            
            MatrixMultMatrix(24,24,1,K,U,F);
            E= MatFieldE[i + j*nex + k*nex*ney];
            MatrixMultScalar(24,1, F, E);
          
            p=i+nnx*j+nnx*nny*(k);
            Fx[p]+=F[0];
            Fy[p]+=F[1];
            Fz[p]+=F[2];
            
            p++;
            Fx[p]+=F[3];
            Fy[p]+=F[4];
            Fz[p]+=F[5];
            
            p+=nnx;
            Fx[p]+=F[6];
            Fy[p]+=F[7];
            Fz[p]+=F[8];
            
            p--;
            Fx[p]+=F[9];
            Fy[p]+=F[10];
            Fz[p]+=F[11];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            Fx[p]+=F[12];
            Fy[p]+=F[13];
            Fz[p]+=F[14];
            
            p++;
            Fx[p]+=F[15];
            Fy[p]+=F[16];
            Fz[p]+=F[17];
            
            p+=nnx;
            Fx[p]+=F[18];
            Fy[p]+=F[19];
            Fz[p]+=F[20];
            
            p--;
            Fx[p]+=F[21];
            Fy[p]+=F[22];
            Fz[p]+=F[23];
            
           // for(m=0;m<24;m++)
           // if (0!=F[m]) printf("f[m]: %f \n",F[m]);  
           // for(m=0;m<24;m++)
           // if (0!=U[m]) printf("U[m]: %f \n",U[m]);     

         }
    }
    
    for (k = 0; k < (nnz); k++)
    for (j = 0; j < (nny); j++) 
    {     
        for (i = 0; i < (nnx); i++)   
        {
          p=i+j*nnx+k*nnx*nny;
          if (0==BcTypeX[p]) Fx[p] =( BcValueX[p] - Fx[p]); else Fx[p]=0.0;
          if (0==BcTypeY[p]) Fy[p] =( BcValueY[p] - Fy[p]); else Fy[p]=0.0;
          if (0==BcTypeZ[p]) Fz[p] =( BcValueZ[p] - Fz[p]); else Fz[p]=0.0;
        }
    }
    return(1);
}

/*---------------------------------------------------------
	GetRestoringForceIncrement3d
----------------------------------------------------------*/

int GetRestoringForceIncrement3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double *BcValueX, double *BcValueY, double *BcValueZ)
{
    int i,j,k;
    int nex, ney, nez, nns;
    double E;
    double U[24],F[24];
    double *K;
    int p;
    
    nex=nnx-1;
    ney=nny-1;
    nez=nnz-1;
    
    SetVectorToZero_double((nnx*nny*nnz), Fx);
    SetVectorToZero_double((nnx*nny*nnz), Fy);
    SetVectorToZero_double((nnx*nny*nnz), Fz);
    
    for (k = 0; k < nez; k++) 
    for (j = 0; j < ney; j++) 
    {     
        for (i = 0; i < nex; i++)   
        {
            p=i+nex*j+nex*ney*k;
            K=Material[MatFieldInt[p]].K;
            
            p=i+nnx*j+nnx*nny*(k);
            U[0]=Ux[p];
            U[1]=Uy[p];
            U[2]=Uz[p];
            
            p++;
            U[3]=Ux[p];
            U[4]=Uy[p];
            U[5]=Uz[p];
            
            p+=nnx;
            U[6]=Ux[p];
            U[7]=Uy[p];
            U[8]=Uz[p];
            
            p--;
            U[9]=Ux[p];
            U[10]=Uy[p];
            U[11]=Uz[p];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            U[12]=Ux[p];
            U[13]=Uy[p];
            U[14]=Uz[p];
            
            p++;
            U[15]=Ux[p];
            U[16]=Uy[p];
            U[17]=Uz[p];
            
            p+=nnx;
            U[18]=Ux[p];
            U[19]=Uy[p];
            U[20]=Uz[p];
            
            p--;
            U[21]=Ux[p];
            U[22]=Uy[p];
            U[23]=Uz[p];
            
            MatrixMultMatrix(24,24,1,K,U,F);
            E= MatFieldE[i + j*nex + k*nex*ney];
            MatrixMultScalar(24,1, F, E);
          
            p=i+nnx*j+nnx*nny*(k);
            Fx[p]+=F[0];
            Fy[p]+=F[1];
            Fz[p]+=F[2];
            
            p++;
            Fx[p]+=F[3];
            Fy[p]+=F[4];
            Fz[p]+=F[5];
            
            p+=nnx;
            Fx[p]+=F[6];
            Fy[p]+=F[7];
            Fz[p]+=F[8];
            
            p--;
            Fx[p]+=F[9];
            Fy[p]+=F[10];
            Fz[p]+=F[11];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            Fx[p]+=F[12];
            Fy[p]+=F[13];
            Fz[p]+=F[14];
            
            p++;
            Fx[p]+=F[15];
            Fy[p]+=F[16];
            Fz[p]+=F[17];
            
            p+=nnx;
            Fx[p]+=F[18];
            Fy[p]+=F[19];
            Fz[p]+=F[20];
            
            p--;
            Fx[p]+=F[21];
            Fy[p]+=F[22];
            Fz[p]+=F[23];
                    

         }
    }
    
    for (k = 0; k < (nnz); k++)
    for (j = 0; j < (nny); j++) 
    {     
        for (i = 0; i < (nnx); i++)   
        {
          p=i+j*nnx+k*nnx*nny;
          if (0!=BcTypeX[p]) Fx[p]=0.0;
          if (0!=BcTypeY[p]) Fy[p]=0.0;
          if (0!=BcTypeZ[p]) Fz[p]=0.0;
        }
    }
    return(1);
}

/*---------------------------------------------------------
	CGSolver2and3d
----------------------------------------------------------*/

void CGSolver2and3d(SolveBlock *SolBlock, MGBlock *M)
{
int i, j, iii, length, ModelDim;
double tmp1, tmp2, tmp3, tmp4;
double err1;
        
        ModelDim=PutGetModelDim(1,123); 
        
        if (2==ModelDim)      
             GetRestoringForce(M->nnx, M->nny, SolBlock->Material->K, M->MatFieldE, 
              M->Ux,  M->Uy, M->Fx,  M->Fy, &err1, &err1, 
              M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY);   
        else
             GetRestoringForce3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        	  M->MatFieldInt, M->Ux,  M->Uy, M->Uz, M->Fx,  M->Fy, M->Fz,
              M->BcTypeX, M->BcTypeY, M->BcTypeZ, M->BcValueX, M->BcValueY, M->BcValueZ);
        
                        
        memcpy(M->D, M->F, sizeof(double)*M->totsize);
        
        for(i=0; i<100000; i++)    
        {
            ScalarProduct(M->totsize, M->F, M->F, &tmp1);
            printf("tmp1: %f\n",tmp1);
            
            if (2==ModelDim) 
                GetRestoringForceIncrement(M->nnx, M->nny, SolBlock->Material->K, M->MatFieldE, 
            	  M->Dx,  M->Dy, M->Adx, M->Ady, &err1, &err1, M->BcTypeX,
                  M->BcTypeY, M->BcValueX, M->BcValueY);
            else
                GetRestoringForceIncrement3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
            	  M->MatFieldInt,  M->Dx,  M->Dy, M->Dz, M->Adx, M->Ady, M->Adz,
                  M->BcTypeX, M->BcTypeY, M->BcTypeZ, M->BcValueX, M->BcValueY, M->BcValueZ);
        	
            ScalarProduct(M->totsize, M->D, M->Ad, &tmp2);
            printf("tmp2: %f\n",tmp2);   
            
            tmp3=tmp1/tmp2; //tmp3 entspricht alpha
            printf("alpha: %f\n",tmp3);
            
            for(j=0; j<M->totsize; j++) 
            {
                 (M->U[j])=(M->U[j])+tmp3*(M->D[j]);
                 (M->F[j])=(M->F[j])-tmp3*(M->Ad[j]);
            }
        	  
            ScalarProduct(M->totsize, M->F, M->F, &tmp2);          
            tmp3=tmp2/tmp1; //tmp3 entspricht beta
            
            for(j=0; j<M->totsize; j++) 
            {
                (M->D[j])=(M->F[j])+tmp3*(M->D[j]);
            }   

            SolBlock->i=i; 
            iii=MyControl();
            if (1==iii) return;
        }
   return;    
}


/*---------------------------------------------------------
     MGCGSolver
----------------------------------------------------------*/

void MGCGSolver(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes)
{
int i, j, iii, length;
double tmp1, tmp2, tmp3;
double err1;
MGBlock *PreMesh, *P;
double *tmpX, *tmpY;
int    PreCondType;

       ReadCycle();
       ChangeTimeFlag(8);  
                         
       PreMesh=SolBlock->PreMesh;
       P=&(PreMesh[num_levels-1]);
       PreCondType=SolBlock->PreCondType;
       
       ChangeTimeFlag(9);  
       
       GetRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
                M->MatFieldInt, SolBlock->Material,
                M->Ux,  M->Uy, M->Fx, M->Fy, 
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
        length=(M->nnx)*(M->nny);
        
        // BEGINNING OF PRECONDITIONING
        ChangeTimeFlag(12);
        tmpX=P->BcValueX; P->BcValueX=M->Fx;
        tmpY=P->BcValueY; P->BcValueY=M->Fy;
        
        SetVectorToZero_double((P->nnx*P->nny),  P->Ux);  
        SetVectorToZero_double((P->nnx*P->nny),  P->Uy);  
        
        switch(PreCondType)
        {   
            case 0: MGSolverVariable(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
            case 1: MGSolver(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
            case 2: MGVCycle(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
        }
        P->BcValueX=tmpX; P->BcValueY=tmpY; 

        ChangeTimeFlag(12);  
        memcpy(M->SwapX, P->Ux, sizeof(double)*length);
        memcpy(M->SwapY, P->Uy, sizeof(double)*length);
        // END OF PRECONDITIONING                
                
        memcpy(M->Dx, M->SwapX, sizeof(double)*length);
        memcpy(M->Dy, M->SwapY, sizeof(double)*length);
      
        for(i=0; i<100000; i++)    
        {
            
            ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->SwapX, &tmp1);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->SwapY, &tmp2);
            tmp1=tmp1+tmp2; 
            
            ChangeTimeFlag(9);
            GetRestoringForceIncrement2(
              M->nnx, M->nny,M->MatFieldE, 
              M->MatFieldInt, SolBlock->Material,
        	  M->Dx,  M->Dy, M->Adx, M->Ady, 
              M->BcTypeX,
              M->BcTypeY, M->BcValueX, M->BcValueY);
        	
        	ChangeTimeFlag(12);
            ScalarProduct(((M->nnx)*(M->nny)), M->Dx, M->Adx, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Dy, M->Ady, &tmp3);
            tmp2=(tmp2+tmp3);   
            
            tmp3=tmp1/tmp2; //tmp3 entspricht alpha
            
            for(j=0; j<length; j++) 
            {
                (M->Ux[j])=(M->Ux[j])+tmp3*(M->Dx[j]);
                (M->Uy[j])=(M->Uy[j])+tmp3*(M->Dy[j]);
            
                (M->Fx[j])=(M->Fx[j])-tmp3*(M->Adx[j]);
                (M->Fy[j])=(M->Fy[j])-tmp3*(M->Ady[j]);
            }
          
            // BEGINNING OF PRECONDITIONING
            tmpX=P->BcValueX; P->BcValueX=M->Fx;
            tmpY=P->BcValueY; P->BcValueY=M->Fy;
            
            SetVectorToZero_double((P->nnx*P->nny),  P->Ux);  
            SetVectorToZero_double((P->nnx*P->nny),  P->Uy);  
            switch(PreCondType)
            {   
                case 0: MGSolverVariable(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
                case 1: MGSolver(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
                case 2: MGVCycle(1, num_levels, K_matrix, SolveWinData, SolBlock, P, PreMesh); break;
            }
            P->BcValueX=tmpX; P->BcValueY=tmpY;
            ChangeTimeFlag(12);
            memcpy(M->SwapX, P->Ux, sizeof(double)*length);
            memcpy(M->SwapY, P->Uy, sizeof(double)*length);        
            // END OF PRECONDITIONING      
            
       	    ScalarProduct(((M->nnx)*(M->nny)), M->Fx, M->SwapX, &tmp2);
            ScalarProduct(((M->nnx)*(M->nny)), M->Fy, M->SwapY, &tmp3);
            tmp2=tmp2+tmp3; 
            
            tmp3=tmp2/tmp1; //tmp3 entspricht beta
            
            for(j=0; j<length; j++) 
            {
                (M->Dx[j])=(M->SwapX[j])+tmp3*(M->Dx[j]);
                (M->Dy[j])=(M->SwapY[j])+tmp3*(M->Dy[j]);
            }   

            SolBlock->i=i; 
            iii=MyControl();
            if (1==iii) return;
        }
   return;
}


    
/*---------------------------------------------------------
	MGPreconditioner
----------------------------------------------------------*/

void MGPreconditioner(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes)   
{    
int i, j, ii, iii, c, length;
double tmp1, tmp2, tmp3;
double err1,err2; 
double Omega;
int Repitition_int;
double Repititions;
MGBlock *P, *Q;
int lambda;


    Omega=1.0;    
    Omega=(SolveWinData->Omega);
    Repititions=(SolveWinData->Repititions);
    
  /*             
        for(j=1; j<(num_levels-1);j++)
        {
 */           P=&(Meshes[3]);           
            
  /*          ChangeTimeFlag(9);
            GetRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
                M->MatFieldInt, SolBlock->Material,
                M->Ux,  M->Uy, M->Fx, M->Fy, 
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
  */          
            ChangeTimeFlag(4);  
            TransForceOnCoarseMesh(M->nnx, M->nny, M->Fx, M->Fy,
                               P->nnx, P->nny, P->BcValueX, P->BcValueY);
            
            ChangeTimeFlag(7);  
            SetVectorToZero_double((M->nnx*M->nny),  M->SwapX);  
            SetVectorToZero_double((M->nnx*M->nny),  M->SwapY);                          
            SetVectorToZero_double((P->nnx*P->nny),  P->SwapX);  
            SetVectorToZero_double((P->nnx*P->nny),  P->SwapY);           
            
            c=num_levels-j;
            Repitition_int= c*c*((int)Repititions);
            
            ChangeTimeFlag(10);
            for(ii=0; ii<100; ii++)//((num_levels-j)*(num_levels-j))
            {
               GetLocalDisplacements_GS_real3(P->nnx, P->nny, SolBlock->Material, P->MatFieldE,
               P->MatFieldInt, P->SwapX,  P->SwapY, Omega, 
               P->BcTypeX, P->BcTypeY, P->BcValueX, P->BcValueY); 
            }
            
            ChangeTimeFlag(5);
            InterpolDispsOnFineMesh(P->nnx, P->nny, P->SwapX,  P->SwapY,
                                M->nnx, M->nny, M->SwapX, M->SwapY, M->weights);
            
      /*      ChangeTimeFlag(12);
            lambda=1;            
            AddIncrement(M->nnx, M->nny, M->SwapX,  M->SwapY, M->dUx, M->dUy,
                M->BcTypeX, M->BcTypeY, lambda);
       */     
            ChangeTimeFlag(10);  
            for(ii=0; ii<1; ii++)//((num_levels-j)*(num_levels-j))
            {
            GetLocalDisplacements_GS_real3(M->nnx, M->nny, SolBlock->Material, M->MatFieldE,
            M->MatFieldInt, M->SwapX,  M->SwapY, Omega, M->BcTypeX, M->BcTypeY, M->Fx, M->Fy); 
            }
            ChangeTimeFlag(7);
//        }
return;
}    

/*---------------------------------------------------------
	MGPreconditioner2
----------------------------------------------------------*/

void MGPreconditioner2(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes)   
{    
int i, j, ii, iii, c, length;
double tmp1, tmp2, tmp3;
double err1,err2; 
double Omega;
int Repitition_int;
double Repititions;
MGBlock *P, *Q;
int lambda;


    Omega=1.0;    
    Omega=(SolveWinData->Omega);
    Repititions=(SolveWinData->Repititions);
    
  /*             
        for(j=1; j<(num_levels-1);j++)
        {
 */           P=&(Meshes[4]);           
            
  /*          ChangeTimeFlag(9);
            GetRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
                M->MatFieldInt, SolBlock->Material,
                M->Ux,  M->Uy, M->Fx, M->Fy, 
                M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
  */          
            ChangeTimeFlag(4);  
            TransForceOnCoarseMesh(M->nnx, M->nny, M->Fx, M->Fy,
                               P->nnx, P->nny, P->BcValueX, P->BcValueY);
            
            ChangeTimeFlag(7);  
            SetVectorToZero_double((M->nnx*M->nny),  M->SwapX);  
            SetVectorToZero_double((M->nnx*M->nny),  M->SwapY);                          
            SetVectorToZero_double((P->nnx*P->nny),  P->SwapX);  
            SetVectorToZero_double((P->nnx*P->nny),  P->SwapY);           
            
            c=num_levels-j;
            Repitition_int= c*c*((int)Repititions);
            
            ChangeTimeFlag(10);
            for(ii=0; ii<5000; ii++)//((num_levels-j)*(num_levels-j))
            {
               GetLocalDisplacements_GS_real3(P->nnx, P->nny, SolBlock->Material, P->MatFieldE,
               P->MatFieldInt, P->SwapX,  P->SwapY, Omega, 
               P->BcTypeX, P->BcTypeY, P->BcValueX, P->BcValueY); 
            }
            
            ChangeTimeFlag(5);
            InterpolDispsOnFineMesh(P->nnx, P->nny, P->SwapX,  P->SwapY,
                                M->nnx, M->nny, M->SwapX, M->SwapY, M->weights);
            
      /*      ChangeTimeFlag(12);
            lambda=1;            
            AddIncrement(M->nnx, M->nny, M->SwapX,  M->SwapY, M->dUx, M->dUy,
                M->BcTypeX, M->BcTypeY, lambda);
       */     
            ChangeTimeFlag(10);  
            for(ii=0; ii<15; ii++)//((num_levels-j)*(num_levels-j))
            {
            GetLocalDisplacements_GS_real3(M->nnx, M->nny, SolBlock->Material, M->MatFieldE,
            M->MatFieldInt, M->SwapX,  M->SwapY, Omega, M->BcTypeX, M->BcTypeY, M->Fx, M->Fy); 
            }
            ChangeTimeFlag(7);
//        }
return;
}    


