/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/

#define frand() ((double) rand()/(RAND_MAX+1.0))
#include <FL/Fl.H>
#include <FL/Fl_Gl_Window.H>

class MyMainWindowUI; 
class MyInputGeomWindowUI;
class MyInputMeshWindowUI;
class MyInputMesoWindowUI;
class MyInputLoadWindowUI;
class MyInputSolveWindowUI;
class MyFileBaseWindowUI;
class MyInputAboutWindowUI;


class MyGraphicsWindow : public Fl_Gl_Window {
	  public:
	    
	    GraphicBlock *block;
		virtual void draw();
		virtual void draw_overlay();
		virtual int handle(int Event);

        MyGraphicsWindow(int X,int Y,int W,int H, char*l);
        
        
};


//MyMainWindow.cpp
int MainRoutine(void);

//MyMGMesher.cpp
void MesherMainLoop(void);
void FreeSolveBlock(SolveBlock *SolBlock);

//MyMGSolver.cpp

/* Initializations of data arrays */
int InitVector_int(int nn, int **M);
int InitVector_double(int nn, double **M);
int InitVector_float(int nn, float **M);
int InitVector_char(int nn, unsigned char **M);
int SetVectorToZero_double(int nn,  double *M);
int SetVectorToZero_int(int nn,  int *M);


/* Initializations of structure */
void InitMGData (MGBlock **Vector, int MaxBlocks);
void InitMGBlock(double lx, double ly, int num_elem_x, int num_elem_y, 
     MGBlock *MG);
void FreeMGBlock(MGBlock *MG);
void InitMGBlock2and3d(double lx, double ly, double lz, 
                   int nex, int ney, int nez,
                   MGBlock *MG);
void FreeMGBlock2and3d(MGBlock *MG);                   
void CopyMGBlock(MGBlock *M, MGBlock *MG );
int  InitializeUx(int nnx, int nny, double *Ux);
int  InitializeUy(int nnx, int nny, double *Uy);
int  StartVectorZero(int nnx, int nny, double *Ux);

/* Create Meshes */
int  CalcElementSizes(double lx, double ly, int max_num_nodes, int *num_elem_x, 
    int *num_elem_y, int *num_levels);
int  CreateMeshes(double lx, double ly, int min_num_nodes, int *num_levels, 
    MGBlock **MG);
int  CreateFlexibleMeshes(int nx, int ny, int num_levels, MGBlock **Mesh_in);    


/* Finite Elements */
int GetStiffnessMatrix(double mu, double lz, double a, double b, double **K);
int GetStiffnessMatrixPStrain(double nu, double h, double a, double b, double **K);
int GetStiffnessMatrixQ9(double nu, double h, double a, double b, double **K);
int GetEASStiffnessMatrix(double E, double nu, double h, double **unknown);
void BuildStiffnessMatrices();

/* Multigrid */

int GetLocalDisplacements_GS_real(int nnx, int nny, double *K, double *MatFieldE, 
    double *dUx, double *dUy, double *Ux,  double *Uy,double *MFx, double *MFy,
	double *err1,double *err2, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);

int GetLocalDisplacements_GS_real2(int nnx, int nny, MaterialBlock *Mat,
    double *MatFieldE, int *MatFieldInt, double *Ux,  double *Uy, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);

int GetLocalDisplacements_GS_real3(int nnx, int nny, MaterialBlock *Mat,
    double *MatFieldE, int *MatFieldInt, double *Ux,  double *Uy, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);    

int Smooth_GS_block(int nnx, int nny, MaterialBlock *Mat,
    double *MatFieldE, int *MatFieldInt, double *Ux,  double *Uy, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY); 

int Smooth_SSOR(int nnx, int nny, MaterialBlock *Mat,
    double *MatFieldE, int *MatFieldInt, double *Ux,  double *Uy, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY); 

int Smooth_GS(int nnx, int nny, MaterialBlock *Mat,
    double *MatFieldE, int *MatFieldInt, double *Ux,  double *Uy, double lambda,
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);
 
int GetLocalDisplacements_GS_Q9(int nnx, int nny, double *K, double *MatFieldE, 
    double *dUx, double *dUy, double *Ux,  double *Uy, double *MFx, double *MFy,
    double *err1,double *err2, double lambda, 
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);

int GetRestoringForce(int nnx, int nny, double *K, double *MatFieldE, 
    double *Ux,  double *Uy,double *MFx, double *MFy,
	double *err1,double *err2, 
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);

int GetRestoringForceCorrect(int nnx, int nny, double *K, double *MatFieldE, 
    double *Ux,  double *Uy,double *MFx, double *MFy,
	double *err1,double *err2, 
    int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);
    
int GetRestoringForce2(int nnx, int nny, double *MatFieldE, int *MatFieldInt, 
       MaterialBlock *Mat, double *Ux,  double *Uy, double *MFx, double *MFy,
              int *BcTypeX, int *BcTypeY, double *BcValueX, double *BcValueY);
int GetRestoringForceIncrement2(
              int nnx, int nny, double *MatFieldE, 
              int *MatFieldInt, MaterialBlock *Mat,
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY);
int GetRestoringForce3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double *BcValueX, double *BcValueY, double *BcValueZ);

int GetRestoringForceIncrement(
              int nnx, int nny, double *K, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, 
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY);

int GetResForceOfOneQ9(double *K, double *Ux, double *Uy, int nnx,
                       int nodeX, int nodeY, double *F, int XorY);
                       
int GetFactor(int nnx, int nny, double *Fx,  double *Fy, double *dFx, double *dFy,
    int *BcTypeX, int *BcTypeY, 
    double *lambda);			  

void CGSolver(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
void GSSolver(double *K_matrix, SolveBlock *SolBlock, MGBlock *M, double Omega);
void PCGSolver(double *K_matrix, SolveBlock *SolBlock, MGBlock *M);
void MGSolver(int num_iterations, int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                     SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
void MGSolverVariable(int num_iterations, int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                     SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);                     
void MGSolver2and3d(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
void MGSolver3d(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
void NLSolver(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                    SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
int OneMGCycle(MGBlock *Meshes, int num_levels, double Omega,
               double Repititions, double *K_matrix,
               double *act_error);  
   

int TransForceOnCoarseMesh(int nnxfine, int nnyfine, double *FxFine, double *FyFine,
                 int nnxcoarse, int nnycoarse, double *FxCoarse, double *FyCoarse);
int TransForceOnCoarseMesh3d(int nnxfine, int nnyfine, int nnzfine, 
    double *FxFine, double *FyFine, double *FzFine,
    int nnxcoarse, int nnycoarse, int nnzcoarse,
    double *FxCoarse, double *FyCoarse, double *FzCoarse);
                 
int TransForceOnCoarseMeshQ4onQ9(int nnxfine, int nnyfine, double *FxFine, double *FyFine,
                 int nnxcoarse, int nnycoarse, double *FxCoarse, double *FyCoarse);
                 
int AddIncrement(int nnx, int nny, double *Ux,  double *Uy,
			  double *dUx, double *dUy, int *BcTypeX, int *BcTypeY, double lambda);

int AddIncrement3d(int nnx, int nny, int nnz,
              double *Ux,  double *Uy, double *Uz,
			  double *dUx, double *dUy, double *dUz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double lambda);			  

int FinerScaleNodes(int nnxc, int nnyc, double *C,
                    int nnxf, int nnyf, double *F);
                    
int InterpolDispsOnFineMesh(
     int nnxc, int nnyc, double *dUxC, double *dUyC,
     int nnxf, int nnyf, double *dUxF, double *dUyF, double *W);

int InterpolDispsOnFineMesh3d(
     int nnxc, int nnyc, int nnzc, double *dUxC, double *dUyC, double *dUzC,
     int nnxf, int nnyf, int nnzf, double *dUxF, double *dUyF, double *dUzF, double *W);
     
int InterpolDispsOnFineMeshQ9onQ4(
     int nnxc, int nnyc, double *dUxC, double *dUyC,
     int nnxf, int nnyf, double *dUxF, double *dUyF, double *W);

int MyControl(void);
			  
/* Material Field */
int CreateMatFieldOnMesh(MGBlock *Mesh);

int CircularField(double xc,double yc, double r, double value, 
    MGBlock *FinestMesh);

int TransMatFieldToCoarse(MGBlock *Mesh, int num_levels);

/* G2 graphics */
/*int OpenMyWindow(int xsize, int ysize, int *device_num1, int *device_num2);
int CreateColorPalette(int d1, int d2, int *col_palette);
int CreateColorPalette7(int d1, int d2, int *col_palette);
int PlotDoubleMatrix(int d1, int d2, double winx, double winy, 
    double min, double max, MGBlock *Mesh, double *Matrix, int *colors);

*/
int FindMinMaxVector(int length, double *vector, double *min, double *max);


/* Output */
int PrintDoubleMatrix(MGBlock *Mesh);
int PrintMatrix(int x, int y, double *M);
int PrintMatrixT(int row, int col, double *M);
int PrintMatrixInteger(int m, int n, int *M);
int PrintMatrixIntegerT(int row, int col, int *M);
void SolverMainLoop(void);



// MyGraphicsWindow.cpp
  void ChangeGraphic_Zoomfactor(float v);
  void ChangeGraphic_ExtrudeZ(float f);
  void ChangeGraphic_RotateAngleX(float f);
  void ChangeGraphic_RotateAngleZ(float f);
  void ChangeGraphic_Deflect(float f);
  void ChangeGraphic_Frames(int i);  
  void ChangeGraphic_UpdateFlag(int i);
  void ChangeGraphic_ShowValues(int i); 
  void ChangeGraphic_PostSmoothFlag(int i); 
  void ChangeGraphic_SelectObject(void *ObjectName); 
  void ChangeGraphic_transX(double d);
  void ChangeGraphic_transY(double d);
  void ChangeGraphic_ActPalette(int f);
  void ChangeGraphic_ShowLevel(int f);
  void ChangeGraphic_DetailLevel(int f);
  void ChangeGraphic_RunFlag(int f); 
  void ChangeGraphic_PostSaveStep(int f); 
  void ChangeGraphic_SmoothNum(int f); 
  void ChangeGraphic_IsoLines(int f); 
  void ChangeGraphic_DrawGridFlag(int f); 
  void ChangeGraphic_DrawFrameFlag(int f);
  void ChangeGraphic_DrawBCFlag(int f);
  void ChangeGraphic_DrawPatchFlag(int f); 
  void ChangeGraphic_DrawNodeFlag(int f);
  void ChangeGraphic_NodeSize(double d); 
  void ChangeGraphic_LineThickness(double d); 
  void ChangeGraphic_Extrude(float d); 
  void ChangeGraphic_ResetView();
  void ChangeGraphic_IsoView();
  void ChangeGraphic_ShowroomFlag(int f);
  
  void IdleCallback2(void * data);
  void AddIdleCallback2();
  void RemoveIdleCallback2();
  
  void DrawStructure(void);
  void DrawMesoStructureDeformed(void);
  void CompNormalVector(float *v1, float *v2, float *v3, float *vn);
  void CompIsoVector(float *v1, float *v2, float w1,  float w2,
                   float *v3, float *v4, float w3,  float w4,
                   float wiso, 
                   float *vp1, float *vp2); 
  void DrawText(int x, int y, char *string);
  void DrawArrow(float x, float y, float length, int direct); 
  void DrawTriangle(float x, float y, float length, int direct); 
  void DrawNothing();
  int Optimize2DView(double w, double right, double left, double top, 
                   double bottom, 
                   double la, double lb,
                   int nx, int ny, float *stpx, float *stpy,
                   int *istart, int *iend, int *jstart, int *jend);
  int PickColor(int NumColor, double min, double max, double val); 
  void MySetViewport(int w, int h); 
  void MyGlContext(); 
  void UpdateGraphicWindow(void);
  void InitGraphicBlock(void);
  void PutGetMainWindow(int choice, MyMainWindowUI **inblock);
  void PutGetGraphicBlock(int choice, GraphicBlock **inblock);
  void PutGetGraphicWindow(int choice, MyGraphicsWindow **inblock);
  void DrawScale();
  int CreateColorPalette_32(ColorPalette *c); 
  int CreateColorPalette_bw32(ColorPalette *c);
  int CreateColorPalette_bw256(ColorPalette *c);
  int CreateColorPalette_9(ColorPalette *c);
  int CreateColorPalette7(ColorPalette *c);
  int CreateColorPalette7b(ColorPalette *c); 
  int CreateColorPalette_Pantone8(ColorPalette *c);
  int CreateColorPalette_PantoneSmooth(ColorPalette *c);
  int CreateColorPalette_BlueRed(ColorPalette *c);
  int CreateColorPalette_RedGreenBlue(ColorPalette *c);
  void PrintGraphicBlock( ); 
// MyInputGeomWindow.cpp
  
void OpenInputGeomWin(void);
void PutGetGeomInputWindow(int choice, MyInputGeomWindowUI **inblock);
void GeomWin_callback(); 
void GeomWin_cb_OK();
void StoreGeomWinData(MyInputGeomWindowUI *geomwin,GeomWinStruct *GeomWinData);
void RecoverGeomWinData(MyInputGeomWindowUI *geomwin,GeomWinStruct *GeomWinData);  
void PrintGeomWinData(GeomWinStruct *GeomWinData);

// MyInputAboutWindow.cpp
  
void OpenInputAboutWin(void);
void PutGetAboutInputWindow(int choice, MyInputAboutWindowUI **inblock);
void AboutWin_callback(); 
void AboutWin_cb_OK();


//void window_callback(Fl_Widget*, void*); 
  
// MyInputMesoWindow.cpp
  
void OpenInputMesoWin(void);
void PutGetMesoInputWindow(int choice, MyInputMesoWindowUI **inblock);
void MesoWin_callback(); 
void MesoWin_cb_OK();
void OpenInputMesoWin();
void StoreMesoWinData(MyInputMesoWindowUI *mesowin,MesoWinStruct *MesoWinData);
void RecoverMesoWinData(MyInputMesoWindowUI *mesowin,MesoWinStruct *MesoWinData);
void PrintMesoWinData(MesoWinStruct *MesoWinData); 
void MesoWin_cb_Apply(int c);
void MesoWin_TransferEMaterial(int MaterialNumBegin, int MaterialNumEnd, double E_Material); 
void MesoWin_TransferMuMaterial(int MaterialNumBegin, int MaterialNumEnd, double Mu_Material);
void MesoWin_TransferVisibleMaterial(int MaterialNumBegin, int MaterialNumEnd, int VisibleFlag);  
void MesoWin_cb_InputControl(int c);  
void RecomputeElementSize(); 
void RecomputeSpecimenSize(); 
  
// MyInputMeshWindow.cpp
  
void OpenInputMeshWin(void);
void PutGetMeshInputWindow(int choice, MyInputMeshWindowUI **inblock);
void MeshWin_callback();
void MeshWin_cb_OK(); 
void StoreMeshWinData(MyInputMeshWindowUI *meshwin,MeshWinStruct *MeshWinData);
void RecoverMeshWinData(MyInputMeshWindowUI *meshwin,MeshWinStruct *MeshWinData);
void PrintMeshWinData(MeshWinStruct *MeshWinData);      

// MyInputLoadWindow.cpp
  
void OpenInputLoadWin(void);
void PutGetLoadInputWindow(int choice, MyInputLoadWindowUI **inblock);
void LoadWin_callback(); 
void LoadWin_cb_OK();  
void LoadWin_cb_Add();
void LoadWin_cb_DeleteAll();  
void LoadWin_cb_DeleteOne();
void LoadWin_ShowLoads(MyInputLoadWindowUI *loadwin, LoadWinStruct *LoadWinData); 
void LoadWin_cb_Add3d();
void LoadWin_cb_DeleteAll3d();  
void LoadWin_cb_DeleteOne3d();
void LoadWin_ShowLoads3d(MyInputLoadWindowUI *loadwin, LoadWinStruct *LoadWinData);

  
 // MyInputSolveWindow.cpp
  
void OpenInputSolveWin(void);
void PutGetSolveInputWindow(int choice, MyInputSolveWindowUI **inblock);
void SolveWin_callback();   
void SolveWin_cb_OK();
void StoreSolveWinData(MyInputSolveWindowUI *solvewin,SolveWinStruct *SolveWinData); 
void RecoverSolveWinData(MyInputSolveWindowUI *solvewin,SolveWinStruct *SolveWinData); 
void PrintSolveWinData(SolveWinStruct *SolveWinData);

 // MyFileBaseWindow.cpp
  
void OpenFileBaseWin(void);
void PutGetBaseFileWindow(int choice, MyFileBaseWindowUI **inblock);
void BaseWin_callback();   
void BaseWin_cb_OK();   
void BaseWin_Save_Geometry(char *filename);   
void BaseWin_Open_Geometry(char *filename);
void BaseWin_Save_Loads(char *filename);   
void BaseWin_Open_Loads(char *filename);
void BaseWin_Save_EModul_Int(char *filename);
void BaseWin_Open_EModul_Int(char *filename);
void ReadDataFile(int nex, int ney);

void MyPause(void);
void ReadCycle(void);
void CreateCycle(void);
void WriteCycle(void);

// MyInputAllWindow.cpp
void PutGetAllWinStruct(int choice, AllWinStruct **inblock);
void InitAllWinStruct(AllWinStruct *inblock);  

// MySolveMenu.cpp
void PutGetSolveBlock(int choice, SolveBlock **inblock);
void InitSolveBlock( ); 
void InitBSplineBlocktoZero(BSplineBlock *BSP);

// MyMesoPix.cpp
int MesoPixMainLoop();
int MesoPixMainLoop3d();
void FreeMesoPixTmp(MesoPixTmp Tmp);
void GenerateOneInclusion(double rmax, double rmin, AggBlock *AggData);
void GenerateOneInclusion3d(double rmax, double rmin, AggBlock *AggData);
void InclFunc(AggBlock *AggData,
              double la, double lb, int coorx, int coory, int *valid);
void InclFunc3d(AggBlock *A, double la, double lb, double lc, 
              int coorx, int coory, int coorz, int *valid);              
void BitmapInclEdge(double la, double lb, AggBlock *AggData, int *p,
                    int *px, int *py, int *pA, int *pmap, int *fillflag);
void BitmapInclEdge3d(double la, double lb, double lc, AggBlock *AggData, int *p,
                    int *px, int *py, int *pz, int *pA, int *pmap, int *fillflag);                    
void InclEdgeZConstant3d(AggBlock *AggData,int la, int lb, int lc,
 int *px, int *py, int *pz, int sx, int sy, int sz, int *count_out);
int FindMinMax_int(int length, int *vector, int *minimum, int *maximum);
int PrintMatrix_int(int xmin, int xmax, int ymin, int ymax, int *vector);
void RecFill(int posx, int posy, int xlength, int *pmap);
void RecFill2(int X1, int X2, int L, int *V);
void RecFill2WithNumber(int X1, int X2, int L, int *V, int n1,  int n2);
void CountPixelArea(int xmin, int xmax, int ymin, int ymax, int *pmap, int *A);
void shellsort3(int *array, int num_row, int num_col, int sort_col, int distanz);
void SmoothMatField(int nex, int ney, double *MA, double *MB);
void MesoSmooth();

// MyPostProc
void PostMainLoop(void);
int NodeMatrix2ElemNodes(int nnx, int nny, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4);
int ElemMatrix2ElemNodes(int nnx, int nny, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4);
int ElemMatrixInt2ElemNodes(int nnx, int nny, int *M,
              double *Post1, double *Post2, double *Post3, double *Post4);              
int PostStrainExx(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, 
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStrainExy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, 
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStrainEyy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, 
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStressSxx(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStressSyy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStressSxy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostStressMises(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostEqStrain(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MType,  
              double *Post1, double *Post2, double *Post3, double *Post4);
int PostSmooth(int nnx, int nny, 
              double *Post1, double *Post2, double *Post3, double *Post4);              
int PostHomogen1(void);
int PostSave(void);               
int PostRestoringForce(int nnx, int nny, double *K, double *MatFieldE, 
                       double *Ux,  double *Uy,
			           double *MFx, double *MFy);
int PostRestoringForce2(int nnx, int nny, double *MatFieldE,               
                        int *MatFieldInt, MaterialBlock *Mat,
                       double *Ux,  double *Uy,
			           double *Fx,  double *Fy);
int PostRestoringForce3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz);
double PostSystemEnergy(void); 
double PostErrorEnergy(void) ;        
int PostEqStrainCenter(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MType,  double *EqStrain);
int PostEqStrainNonLocal(int nnx, int nny, double la, double lb, 
              int *MType, double Radius, double *Swap, double *EqStrain);
double PostInternalEnergy(void);
void PostEnergies(void);
int PostHomogenY(void);

                  
// MyBCdata    
void BCmain(int level);
void SplineBCmain();
        
 
//CG

int ScalarProduct(int n, double *v1, double *v2, double *alpha); 
int JacobiPreconditioner(int nnx, int nny, double *K, double *MatFieldE, double *Mpc);
int VectorMult(int n, double *v1, double *v2, double *v3);             
void SaveTwoValues(double v1, double v2);                           
   
                                                     
//BSplines                                                                                                       
void  CGSolverForBSplines(SolveBlock *SolBlock, MGBlock *NormalMesh, MGBlock *SplineMesh, 
      double *K11,double *K12,double *K13,
      double *K21,double *K22,double *K23,
      double *K31,double *K32,double *K33);
void BSplineMain(); 
void TransposeMatrix(int row, int col,double *M,double *MT); 
void MatrixMultMatrix(int rowA,int colA,int colB,double *A,double *B,double *C);
void AddMatrix(int row, int col, double *M, double *Minc);  
void BuildStiffnessForBSplines(double la, double lb,
   double *K11, double *K12, double *K13,
   double *K21, double *K22, double *K23,
   double *K31, double *K32, double *K33);
void MatrixMultScalar(int row,int col, double *M, double S);
void BTMatrixForBSplines(
               double Nxa, double Nxb, double Nxc,
               double Nya, double Nyb, double Nyc,
               double Nxad, double Nxbd, double Nxcd,
               double Nyad, double Nybd, double Nycd,
               double *BT);  
int GetRestoringForceBSplines(
              int nnx, int nny, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *Fx, double *Fy,
			  int *BcTypeX, int *BcTypeY,
			  double *BcValueX, double *BcValueY,
              double *K11, double *K12, double *K13,
              double *K21, double *K22, double *K23,
              double *K31, double *K32, double *K33);  
              

int GetRestoringForceIncrementBSplines(
              int nnx, int nny, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *Fx, double *Fy,
			  int *BcTypeX, int *BcTypeY,
			  double *BcValueX, double *BcValueY,
              double *K11, double *K12, double *K13,
              double *K21, double *K22, double *K23,
              double *K31, double *K32, double *K33);  
int TransformSplinesOnMesh(int nnx, int nny, double *NUx,  double *NUy,
                                             double *SUx, double *SUy);  
void InitPolynomStruct(PolynomStruct **Vector, int numpoly);
void InitQuadraticBSplines();
void SetQuadraticBSplines(SomePolynoms *Polynomials, double la);    

void BSplineMain2();
void GetBTatXY(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y);
void BuildStiffnessForBSplines2(BSplineBlock *BSP);

int TransformSplinesOnMesh2(int nnx, int nny, double *NUx,  double *NUy,
                                             double *SUx, double *SUy);
void InitSomePolynoms(SomePolynoms **Vector, int numpoly);
void BuildMaterialMatrix(double mu, double *C);
double PolyVal(PolynomStruct P,double x);
void AllocateMatrices(BSplineBlock *BSP);
void GetNatXY(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y);
int TransformSplinesOnMesh2(int direction,   int nnx, int nny, 
                            double la, double lb, double *SUx, double *SUy,
                            BSplineBlock *BSP, 
                            double *Post1,double *Post2,double *Post3,double *Post4);                              

int TransformSplinesOnMesh3(int type,   int nnx, int nny, 
                            double la, double lb, double *SUx, double *SUy,
                            double *MatFieldE,
                            BSplineBlock *BSP, 
                            double *Post1,double *Post2,double *Post3,double *Post4);
void ChangeGraphic_RotateFlag(int f); 
void ChangeGraphic_BmpDataFlag(); 
void ChangeGraphic_BmpScreenFlag();
void ChangeGraphic_BmpDataSeqFlag(int i);
void ChangeGraphic_BmpScreenSeqFlag(int i);
void WriteBmp(int nex, int ney, double *Post1, double *Post2, double *Post3,double *Post4,
              double minimum, double maximum, ColorPalette cp);
void WriteBmp2(int nex, int ney);
void ReadBmp();

/*--------neue Deklarationen im Zusammenhang mit order k B-Splines------------*/

//MyInputLoadWindow
void SortLoads(LoadWinStruct *LoadWinData); 

//Matrixoperationen
int GaussSeidelGLSloeser(double *A, double *x, double *b, int GLsize);
//int PrintMatrixToFile(int row, int col, double *M);
int PrintMatrixToFile(int row, int col, double *M, char name[20]);
void AddMatrix(int row, int col, double *M, double *Minc);
void MatrixMultMatrix(int rowA,int colA,int colB,double *A,double *B,double *C);
void MatrixMultScalar(int row,int col, double *M, double S);
void TransposeMatrix(int row, int col,double *M,double *MT);


//Polynomfunktionen, alle in MyPolynomRoutines.c
double PolyVal(PolynomStruct P,double x);
int PrintSplinePolynomStruct( SomeElements *SpElems, int e, int p);
void AddPolynoms( PolynomStruct *P1, PolynomStruct *P2);
void CopyPolynoms( PolynomStruct *P1, PolynomStruct *P2);
void DeletePolynomFunction(PolynomStruct *P);
void DerivePolynomStruct(PolynomStruct *P);
void InitPolynomFunction(PolynomStruct *P, int numcoeff);
void InitPolynomStruct(PolynomStruct **Vector, int numpoly);
void MultPolynoms( PolynomStruct *P1, PolynomStruct *P2);        
void PrintPolynomStruct(PolynomStruct *P);

//GaussQuadratur, alle in MyGaussQuadRoutines.c
int BuildGaussValues(GaussValues *GBV, int numXi);
int GetSamplePoints( double **Xv, double **Gv, int numXi);
int GetSamplePointsToBSP(BSplineBlock *BSP,int numXi);
int GetSamplePointsToBSP_old(BSplineBlock *BSP,int numXi);
void InitSamplePoints(int n);
void PutGetGaussValues(int choice, GaussValues **inblock);

//B-SplineBasis
int InitOrderkBSplines();
void AllocBSplineMem(SomeElements *SpElems, int numElements, int numpoly, int numcoeff);
void FreeBSplineMem(SomeElements *SpElems);
void SetOrderkBSplines(BSplineBlock *BSP);
void AllocateMatrices_K(BSplineBlock *BSP);
void BTCBandAddKreplace(BSplineBlock *BSP, double *K, double *KM, double G);
void BuildMaterialMatrix(double mu, double *C);
void BuildStiffnessForBSplines3(BSplineBlock *BSP);
void GetBatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y);
void GetBTatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y);
void GetNatXY_K(int TypeX, int TypeY, BSplineBlock *BSP, double x, double y);
void ReflectHorizontal( double *Kold, double *Knew, int numpoly);
void ReflectVertical( double *Kold, double *Knew, int numpoly);

//B-SplineRest
void BSplineMain3();
void SplineBCmain_K();
void PrintLoadlist(OneLoad *LoadList, int num_loads);
int PostSaveProfile(void);

//MyBSplineCGSolver
void CGSolverForBSplines_K(SolveBlock *SolBlock, MGBlock *NormalMesh, MGBlock *SplineMesh,
                           int YMsF, int numXi);
void GetRestoringForceBSplines_K(MGBlock *NormalMesh, MGBlock *SplineMesh, BSplineBlock *BSP,
                               double *U, double *Ux, double *Uy,
                               double *F, double *Fx, double *Fy,
                               int YMsF, int numXi, double *Ftemp);
void GetRestoringForceIncrementBSplines_K(MGBlock *NormalMesh, MGBlock *SplineMesh, BSplineBlock *BSP,
                                        double *U, double *Ux, double *Uy,
                                        double *F, double *Fx, double *Fy,
                                        int YMsF, int numXi, double *Ftemp);
void GetBSPForceVector(MGBlock *NormalMesh, MGBlock *SplineMesh, BSplineBlock *BSP,
                       double *U, double *Ux, double *Uy,
                       double *F, double *Fx, double *Fy,
                       int YMsF, int numXi, double *Ftemp);

//Post Processing
void TransformSplinesOnMesh2_K(int direction,   int nex, int ney,
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             BSplineBlock *BSP,
                             double *Post1,double *Post2,double *Post3,double *Post4);
void TransformSplinesOnMesh3_K(int type,   int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *MatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4);

/*------Ende neue Deklarationen im Zusammenhang mit order k B-Splines----------*/

/*------neue Deklarationen im Zusammenhang mit multi-phase B-Splines-----------*/
void createKnMatFieldE(MGBlock *MG);
void BuildStiffnessForBSplines3a(BSplineBlock *BSP);
double GetEatIntPoints(MGBlock *NormalMesh, BSplineBlock *BSP, int i, int j, int nex, int xi, int xj);
double GetEatIntPoints_a(MGBlock *NormalMesh, BSplineBlock *BSP, double E1, double E2, double E3, double E4, int xi, int xj);
void TransformSplinesOnMesh3a(int type,   int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *KnMatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4);
void cb_PrintSectionvalues(void);
void cb_PrintSectionvaluesFiner(void);
void UpdateSectionSliders(void);                       
                             
void cb_DrawFinerSplines(void);

void TransformSplinesOnMesh4(int direction, int Factor,  int nex, int ney,
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             BSplineBlock *BSP,
                             double *Post1,double *Post2,double *Post3,double *Post4);
                             
void TransformSplinesOnMesh5(int type, int Factor,  int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *MatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4);
                             
void TransformSplinesOnMesh5a(int type, int Factor,  int nex, int ney, 
                             double la, double lb, double *Ux, double *Uy, int Spnnx,
                             double *KnMatFieldE, BSplineBlock *BSP, 
                             double *Post1,double *Post2,double *Post3,double *Post4);                                                          
/*------Ende neue Deklarationen im Zusammenhang mit multi-phase B-Splines------*/


//MyMaterial.c
void InitMaterialBlock(int num, MaterialBlock **M);
void FreeMaterialBlock(int num, MaterialBlock *M);
void MaterialInfo();
void CopyMaterialBlock(int num, MaterialBlock *Mnew, MaterialBlock *Mold);


// 3D 3D 3D
int PutGetModelDim(int choice, int value) ;
void MesherMainLoop3d(void);
void InitMGBlock3d(double lx, double ly, double lz, int nex, int ney, int nez, MGBlock *MG);
int  CreateMeshes3D(double lx, double ly, double lz, int min_num_nodes, int *num_levels, MGBlock **Mesh_in);
int CalcElementSizes3d(double lx, double ly, double lz, int min_num_nodes, 
                       int *num_elem_x, int *num_elem_y, int *num_elem_z, int *num_levels);
void DrawMesoStructure3d(); 
void DrawOneCube3d(int i, int j, int k, double la, double lb, double lc,  
                   double Post1,  double Post2, double Post3,  double Post4,
                   double Post5,  double Post6, double Post7,  double Post8,
                   double minimum, double maximum, GraphicBlock *graphblock,
                   int NumColor, float *r, float *g, float *b);
void BCmain3d(int level);                   
void CollectNodesFace(int face, double la, double lb, double lc,
                       int nnx, int nny, int nnz, int *NumNodes, 
                       int *Nodes, double *Area);             
void BuildMaterialMatrix3d(double mu, double *C);            

void GetBatXYZ3d(double a, double b, double c, 
                 double x, double y, double z, double *BMatrix);  
void GetBatNode3d(double a, double b, double c, 
                  int node, double *BMatrix);
void GetStiffnessMatrix3d(double mu, double a, double b, double c, double **KMatrix);
int ElemMatrix2ElemNodes3d(int nnx, int nny, int nnz, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8);
int ElemMatrixInt2ElemNodes3d(int nnx, int nny, int nnz, int *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8);  
int NodeMatrix2ElemNodes3d(int nnx, int nny, int nnz, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8); 
int PostSmooth3d(int nnx, int nny, int nnz, 
                 double *Post1, double *Post2, double *Post3, double *Post4,
                 double *Post5, double *Post6, double *Post7, double *Post8);
void PostMainLoop3d(void);  
void TransformDetail2and3d(void);  
void CGSolver3d(SolveBlock *SolBlock, MGBlock *M);
int GetRestoringForceIncrement3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double *BcValueX, double *BcValueY, double *BcValueZ);
int PostStrain3d(int component, int nnx, int nny, int nnz, 
                    double la, double lb, double lc, 
              double *Ux, double *Uy, double *Uz,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8);
int PostStress3d(int component, int nnx, int nny, int nnz, 
              int *MatFieldInt, double *MatFieldE,MaterialBlock *Material,
                    double la, double lb, double lc, 
              double *Ux, double *Uy, double *Uz,
              double *Post1, double *Post2, double *Post3, double *Post4,
              double *Post5, double *Post6, double *Post7, double *Post8);
void IdleCallback(void * data);
void AddIdleCallback();
void RemoveIdleCallback();   
void CGSolver2and3d(SolveBlock *SolBlock, MGBlock *M);     
void GetResForceOfOneElement3d(double E,double *K, double *Ux, double *Uy, double *Uz,
                        int nnx, int nny, int nnz, int node, double *F, int XorYorZ);   

int GetLocalDisplacements_GS_3d(
              int nnx, int nny, int nnz,
              MaterialBlock *Mat,
              double *MatFieldE, 
              int *MatFieldInt,
              double *Ux,  double *Uy, double *Uz,
			  double lambda,
              int *BcTypeX, int *BcTypeY, int *BcTypeZ,
              double *BcValueX, double *BcValueY, double *BcValueZ);     
void CreateColorIndex2and3d(void); 
void CreateColorIndexB(void); 
void CreateColorIndexC(void);
void UpdateList(int *Map, int *List, int *ListSize);
void UpdateMatList(int k, int *Map, int *List, int *ListSize);
void ReSortColorIndex2and3d();

void TEST1();
void TEST2();
void StartGameOfLife();

// Time-Control
void InitTimeData(TimeBlock *TimeData);
void UpdateTimeData(int flag, TimeBlock *TimeData);
void ChangeTimeFlag(int flag);
void PrintTimeData(void);
void PrintCheckTimeData(void);

// Error-Control              
void ChangeErrorValue(int type, double value);
void PrintErrorData(void);
int ComputeEuclideanNorm(void);
int ComputeRelEnergyError(void);
double GetSystemEnergy(void);
double GetErrorEnergy(void);
int ComputeRelEuclideanError(void);
double GetSystemEuclidean(void);
double GetErroreuclidean(void);
int CheckErrorTol(int type, double value);

 void MGVCycle(int num_iterations, int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);
void MGSubVCycle(int act_level, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *Meshes) ;      
void MGPreconditioner(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes)   ;
void MGPreconditioner2(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes) ;
void MGCGSolver(int num_levels, double *K_matrix, SolveWinStruct *SolveWinData, 
                       SolveBlock *SolBlock, MGBlock *M, MGBlock *Meshes);                  
void InitMGBlock2and3dPreCond(MGBlock *MG, MGBlock *PG);
void FreeMGBlock2and3dPreCond(MGBlock *MG);
int  InitPreMeshes();
void GenerateOneFiber3d(double rmax, double rmin, AggBlock *AggData);

void BaseWin_Open_Results(char *savefile) ;
void BaseWin_Save_Results(char *savefile) ;
void MapSxxInclusion(int type);   
void createKnMatFieldCircle();
void CircleFunction(double xpos, double ypos, double *EModul);

double PostAverages(void);
int SystemCheck(void);    
int SystemCheck2(void); 
void PerformMC(void);                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
// MyMGMesoGen.cpp
/*
int MesoGenMainLoop();
int CreateInclusionMatrix1(int num_incl,  double *incl_matrix);
int CreateInclusionMatrix2(int num_incl,int insert_loop, double b_minx, double b_maxx, double b_miny, double b_maxy,double *incl_matrix);
int getBoxesize( int num_incl,double *incl_matrix);
int setbox(int insert_incl,int num_incl, double *incl_matrix,double *box_matrix);
int cutBox(int i,int j,double factor,int num_incl,double *incl_matrix,double *box_matrix);
int insert(double factor,double b_minx, double b_maxx, double b_miny, double b_maxy,int num_incl,double
*incl_matrix,double *box_matrix);
int Build2dMatrix(int num_incl, double *incl_matrix,int winsize_x,int winsize_y,double b_minx, double b_maxx,
double b_miny, double b_maxy, int *plot_matrix, int *ptx, int *pty);
int Print2dMatrix(int *plot_matrix, int ptx, int pty);
void shellsort2(double *array, int num_row, int num_col, int sort_col, int distanz);
int cutEdges(int m,double b_minx, double b_maxx,double b_miny, double b_maxy,int num_incl,
		double *incl_matrix,double *box_matrix);
double incl_func(double x,double y,double cx,double cy,double sinphi,double cosphi,double a,double b,double pot);
*/
