/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/*---------------------------------------------------------
	TransForceOnCoarseMeshQ4onQ9
----------------------------------------------------------*/

int TransForceOnCoarseMeshQ4onQ9(int nnxfine, int nnyfine, double *FxFine, double *FyFine,
                 int nnxcoarse, int nnycoarse, double *FxCoarse, double *FyCoarse)
{
	int i,j,m,n, step;
	double step_db;
	int PosXFine, PosYFine;
	double x, y;
	double W1,W2,W3,W4,W5,W6,W7,W8,W9;
	int ma,mb,na,nb;
    int nexcoarse, neycoarse;
    int p, q;
    
	
    nexcoarse=nnxcoarse-1;
    neycoarse=nnycoarse-1;
    step=(nnxfine-1)/(nnxcoarse-1);
    step_db=(double) step;

    for(j=0; j<nnycoarse; j++)
	{
		for(i=0; i<nnxcoarse; i++)
		{	
			p=i+j*nnxcoarse;
            FxCoarse[p]=0.0;
			FyCoarse[p]=0.0;
		}
	
	}

    
    //   7-8-9
    //   4-5-6
    //   1-2-3

    
	for(j=1; j<(nnycoarse-1); j+=2)
	{
		for(i=1; i<(nnxcoarse-1); i+=2)
		{	
			p=i+j*nnxcoarse;
            			
            /* An welcher Stelle im feinen Grid ?*/	
			PosXFine=step*i;
			PosYFine=step*j;
			
            ma=-(step);
			mb=(step-1);
			na=ma;
			nb=mb;
			
            if ((nexcoarse-1)==i) {mb=step;}
            if ((neycoarse-1)==j) {nb=step;}
            
            for(n=na; n<=nb; n++)
			{
				for(m=ma; m<=mb; m++)
				{	
                    q= (PosXFine+m) + (PosYFine+n) * nnxfine;
                    x=((double) m / step_db);
					y=((double) n / step_db);
					
					W1 =  0.25 * (1-x)*x  * (1-y)*y;
					W2 =  -0.5 * (1-x*x)  * (1-y)*y;
					W3 = -0.25 * (1+x)*x  * (1-y)*y;
					W4 =  -0.5 * (1-x)*x  * (1-y*y);
					W5 =         (1-x*x)  * (1-y*y);
					W6 =   0.5 * (1+x)*x  * (1-y*y);
					W7 = -0.25 * (1-x)*x  * (1+y)*y;
					W8 =   0.5 * (1-x*x)  * (1+y)*y;
					W9 =  0.25 * (1+x)*x  * (1+y)*y;
				
					p=(i-1)+(j-1)*nnxcoarse;      
					FxCoarse[p]+=FxFine[q]*W1 ;
					FyCoarse[p]+=FyFine[q]*W1 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W2 ;
					FyCoarse[p]+=FyFine[q]*W2 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W3 ;
					FyCoarse[p]+=FyFine[q]*W3 ;
					
					p=(i-1)+j*nnxcoarse;
					FxCoarse[p]+=FxFine[q]*W4 ;
					FyCoarse[p]+=FyFine[q]*W4 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W5 ;
					FyCoarse[p]+=FyFine[q]*W5 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W6 ;
					FyCoarse[p]+=FyFine[q]*W6 ;
					
					p=(i-1)+(j+1)*nnxcoarse;
					FxCoarse[p]+=FxFine[q]*W7 ;
					FyCoarse[p]+=FyFine[q]*W7 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W8 ;
					FyCoarse[p]+=FyFine[q]*W8 ;
					
					p++;
					FxCoarse[p]+=FxFine[q]*W9 ;
					FyCoarse[p]+=FyFine[q]*W9 ;
					
				}
			}
		}
	
	}	


return(1);
}


/*---------------------------------------------------------
	InterpolDispsOnFineMeshQ9onQ4
----------------------------------------------------------*/

int InterpolDispsOnFineMeshQ9onQ4(
     int nnxcoarse, int nnycoarse, double *dUxC, double *dUyC,
     int nnxfine, int nnyfine, double *dUxF, double *dUyF, double *W)
{
	int i,j,m,n, step;
	double step_db;
	int PosXFine, PosYFine;
	double x, y;
	double W1,W2,W3,W4,W5,W6,W7,W8,W9;
	int ma,mb,na,nb;
    int nexcoarse, neycoarse;
    int p, q;
    
	
    nexcoarse=nnxcoarse-1;
    neycoarse=nnycoarse-1;
    step=(nnxfine-1)/(nnxcoarse-1);
    step_db=(double) step;

    for(j=0; j<nnyfine; j++)
	{
		for(i=0; i<nnxfine; i++)
		{	
			p=i+j*nnxcoarse;
            dUxF[p]=0.0;
			dUyF[p]=0.0;
		}
	
	}

    
    //   7-8-9
    //   4-5-6
    //   1-2-3

    
	for(j=1; j<(nnycoarse-1); j+=2)
	{
		for(i=1; i<(nnxcoarse-1); i+=2)
		{	
			p=i+j*nnxcoarse;
            			
            /* An welcher Stelle im feinen Grid ?*/	
			PosXFine=step*i;
			PosYFine=step*j;
			
            ma=-(step);
			mb=(step-1);
			na=ma;
			nb=mb;
			
            if ((nexcoarse-1)==i) {mb=step;}
            if ((neycoarse-1)==j) {nb=step;}
            
            for(n=na; n<=nb; n++)
			{
				for(m=ma; m<=mb; m++)
				{	
                    q= (PosXFine+m) + (PosYFine+n) * nnxfine;
                    x=((double) m / step_db);
					y=((double) n / step_db);
					
					W1 =  0.25 * (1-x)*x  * (1-y)*y;
					W2 =  -0.5 * (1-x*x)  * (1-y)*y;
					W3 = -0.25 * (1+x)*x  * (1-y)*y;
					W4 =  -0.5 * (1-x)*x  * (1-y*y);
					W5 =         (1-x*x)  * (1-y*y);
					W6 =   0.5 * (1+x)*x  * (1-y*y);
					W7 = -0.25 * (1-x)*x  * (1+y)*y;
					W8 =   0.5 * (1-x*x)  * (1+y)*y;
					W9 =  0.25 * (1+x)*x  * (1+y)*y;
				
					p=(i-1)+(j-1)*nnxcoarse;      
					dUxF[q]=
					W1*dUxC[(i-1)+(j-1)*nnxcoarse]+
					W2*dUxC[(i  )+(j-1)*nnxcoarse]+
					W3*dUxC[(i+1)+(j-1)*nnxcoarse]+
					W4*dUxC[(i-1)+(j  )*nnxcoarse]+
					W5*dUxC[(i  )+(j  )*nnxcoarse]+
					W6*dUxC[(i+1)+(j  )*nnxcoarse]+
					W7*dUxC[(i-1)+(j+1)*nnxcoarse]+
					W8*dUxC[(i  )+(j+1)*nnxcoarse]+
					W9*dUxC[(i+1)+(j+1)*nnxcoarse];
					
					dUyF[q]=
					W1*dUyC[(i-1)+(j-1)*nnxcoarse]+
					W2*dUyC[(i  )+(j-1)*nnxcoarse]+
					W3*dUyC[(i+1)+(j-1)*nnxcoarse]+
					W4*dUyC[(i-1)+(j  )*nnxcoarse]+
					W5*dUyC[(i  )+(j  )*nnxcoarse]+
					W6*dUyC[(i+1)+(j  )*nnxcoarse]+
					W7*dUyC[(i-1)+(j+1)*nnxcoarse]+
					W8*dUyC[(i  )+(j+1)*nnxcoarse]+
					W9*dUyC[(i+1)+(j+1)*nnxcoarse];
					
				}
			}
		}
	
	}	

	
return(1);
}

/*------------------------------------------------------------------------------
	GetResForceOfOneQ9
--------------------------------------------------------------------------------*/ 

int GetResForceOfOneQ9(double *K, double *Ux, double *Uy, int nnx,
                       int nodeX, int nodeY, double *F, int XorY)
{
    int p,q, nnx2;
    double *M;
    
    p=nodeX+nodeY*nnx;  
    
    /*
    NodeY  
    
     2   6-7-8
     1   3-4-5
     0   0-1-2
        
         0 1 2 NodeX
     */
    
    q=((nodeX+nodeY*3)*2)*18; 
    if (1==XorY) q+=18;
    M=&(K[q]);
    
    /*
    dof
    (12,13)-(14,15)-(16,17)
    (6,7)  -(8,9)  -(10,11)
    (0,1)  -(2,3)  -(4,5)
    */
    
    nnx2=nnx*2;
    
    F[0] += M[0]*Ux[0]      +M[1] *Uy[0]
           +M[2]*Ux[1]      +M[3] *Uy[1]
           +M[4]*Ux[2]      +M[5] *Uy[2]
           
           +M[6] *Ux[nnx]   +M[7] *Uy[nnx] 
           +M[8] *Ux[1+nnx] +M[9] *Uy[1+nnx] 
           +M[10]*Ux[2+nnx] +M[11]*Uy[2+nnx]
    
           +M[12]*Ux[nnx2]  +M[13]*Uy[0+nnx2] 
           +M[14]*Ux[1+nnx2]+M[15]*Uy[1+nnx2]
           +M[16]*Ux[2+nnx2]+M[17]*Uy[2+nnx2];
    
}


/*------------------------------------------------------------------------------
	GetLocalDisplacements_GS_Q9
--------------------------------------------------------------------------------*/ 
/* 
Nine Node Element:
4-7-3
8-9-6
1-5-2

Degrees of Freedom: [u1,v1,u2,v2,u3,v3,...u9,v9]
*/


int GetLocalDisplacements_GS_Q9(
              int nnx, int nny, double *K, double *MatFieldE, 
              double *dUx, double *dUy, 
			  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, double lambda,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    int i,j,ie,je;
    int p,q;
    int nex, ney;
    double E1, E2, E3, E4;
    double Fx, Fy, FxS, FyS;
    
    (*err1) = 0;
    (*err2) = 0;
    nex=(nnx-1)/2;
    ney=(nny-1)/2;
    
    /* 1. Middle Node 
    O-O-O
    O-X-O
    O-O-O
    */ 
       
    for (j = 0; j < (nny-2); j+=2) 
    {     
        for (i = 0; i < (nnx-2); i+=2)   
        {
	      	 p=i/2 + j/2*nex;
             E1= MatFieldE[p]; 
             
             p = i + j*nnx; 
             q = p+1+nnx; 
         
             if (0==BcTypeX[q])
             {
                 Fx=0.0;
                 GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 1, &Fx, 0);
              
                 Fx *= E1;
                 Fx -= BcValueX[q]; 
                 MFx[q]=Fx;
                 dUx[q]=(-Fx/(E1*K[152])); // 152=8*18+8
		         Ux[q]+=lambda*dUx[q]; 
             } 
   
   
             if (0==BcTypeY[q])
             {
                 Fy=0.0;
                 GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 1, &Fy, 1);
              
                 Fy *= E1;         
		         Fy -= BcValueY[q]; 
                 MFy[q]=Fy;
	             dUy[q]=(-Fy/(E1*K[171])); // 171=9*18+9
	             Uy[q]+=lambda*dUy[q];
	         }
         }
    }
    
   //----------------------------------------------------------------- 
   /* Center node of 1x2 Element Patches
    O-O-O-O-O
    O-0-X-O-O
    O-O-O-O-O
    */ 
   
    for (j = 0; j < (nny-2); j+=2) 
    {     
        for (i = 0; i < nnx; i+=2)   
        {
             p=i/2 + j/2*nex-1;
             if (i!=0) E1= MatFieldE[p]; else E1=0.0;
             
             p++;
             if (i!=(nnx-1)) E2= MatFieldE[p]; else E2=0.0;
         
             p = i + j*nnx; 
             q = p+nnx; 
         
             if (0==BcTypeX[q])
             {
                 FxS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2; // lower left node of element
                     q = p + 2 + nnx;   // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 1, &Fx, 0);
                     Fx *= E1;
                     FxS+=Fx;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p + nnx;     // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 1, &Fx, 0);
                     Fx *= E2;
                     FxS+=Fx;
                 } 
                 
                 Fx=FxS;                
                 Fx -= BcValueX[q]; 
                 MFx[q]=Fx;
                 dUx[q]=(-Fx/(E1*K[190]+E2*K[114])); // 10*18+10=190 6*18+6=114
		         Ux[q]+=lambda*dUx[q]; 
             } 
             
             p = i + j*nnx; 
             q = p+nnx; 
         
             if (0==BcTypeY[q])
             {
                 FyS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2; // lower left node of element
                     q = p + 2 + nnx;   // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 1, &Fy, 1);
                     Fy *= E1;
                     FyS+=Fy;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p + nnx;     // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 1, &Fy, 1);
                     Fy *= E2;
                     FyS+=Fy;
                 } 
                 
                 Fy=FyS;                
                 Fy -= BcValueY[q]; 
                 MFy[q]=Fy;
                 dUy[q]=(-Fy/(E1*K[209]+E2*K[133])); // 11*18+11=209 7*18+7=133
		         Uy[q]+=lambda*dUy[q]; 
             }      
         }
    }
    
     //----------------------------------------------------------------- 
   /* Center node of 2x1 Element Patches
    O-O-O
    0-O-O
    0-X-0
    O-0-0
    O-O-O
    */ 
   
    for (j = 0; j < nny; j+=2) 
    {     
        for (i = 0; i < (nnx-2); i+=2)   
        {
             p=i/2 + j/2*nex-nex;
             if (j!=0) E1= MatFieldE[p]; else E1=0.0;
             
             p+=nex;
             if (j!=(nny-1)) E2= MatFieldE[p]; else E2=0.0;
         
             p = i + j*nnx; 
             q = p+1; 
         
             if (0==BcTypeX[q])
             {
                 FxS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2*nnx; // lower left node of element
                     q = p + 1 + 2*nnx;   // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 2, &Fx, 0);
                     Fx *= E1;
                     FxS+=Fx;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p + 1;     // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 0, &Fx, 0);
                     Fx *= E2;
                     FxS+=Fx;
                 } 
                 
                 Fx=FxS;                
                 Fx -= BcValueX[q]; 
                 MFx[q]=Fx;
                 dUx[q]=(-Fx/(E1*K[266]+E2*K[38])); // 14*18+14=266 2*18+2=38
		         Ux[q]+=lambda*dUx[q]; 
             } 
             
             p = i + j*nnx; 
             q = p+1; 
         
             if (0==BcTypeY[q])
             {
                 FyS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2*nnx; // lower left node of element
                     q = p + 1 + 2*nnx;   // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 2, &Fy, 1);
                     Fy *= E1;
                     FyS+=Fy;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p + 1;     // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 1, 0, &Fy, 1);
                     Fy *= E2;
                     FyS+=Fy;
                 } 
                 
                 Fy=FyS;                
                 Fy -= BcValueY[q]; 
                 MFy[q]=Fy;
                 dUy[q]=(-Fy/(E1*K[285]+E2*K[57])); // 15*18+15=285 3*18+3=57
		         Uy[q]+=lambda*dUy[q]; 
             }       
         }
     }
     //----------------------------------------------------------------- 
   /* Center node of 2x2 Element Patch
    O-O-O-0-0
    0-O-O-0-0
    0-0-X-0-0
    O-0-0-0-0
    O-O-O-0-0
    
    E4-E3
    E1-E2
    */ 
   
    for (j = 0; j < nny; j+=2) 
    {     
        for (i = 0; i < nnx; i+=2)   
        {
             p=i/2 + j/2*nex-nex-1;
             if (j!=0 && i!=0) E1= MatFieldE[p]; else E1=0.0;
             
             p++;
             if (i!=(nnx-1) && j!=0) E2= MatFieldE[p]; else E2=0.0;
         
             p+=nex;
             if (i!=(nnx-1) && j!=(nny-1)) E3= MatFieldE[p]; else E3=0.0;
         
             p--;
             if (i!=0 && j!=(nny-1)) E4= MatFieldE[p]; else E4=0.0;
         
             p = i + j*nnx; 
             q = p; 
         
             if (0==BcTypeX[q])
             {
                 FxS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2*nnx -2; // lower left node of element
                     q = p + 2 + 2*nnx;   // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 2, &Fx, 0);
                     Fx *= E1;
                     FxS+=Fx;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx-2*nnx; // lower left node of element
                     q = p + 2*nnx;     // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 2, &Fx, 0);
                     Fx *= E2;
                     FxS+=Fx;
                 } 
                 
                 if (E3!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p;     // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 0, &Fx, 0);
                     Fx *= E3;
                     FxS+=Fx;
                 }
                 
                 if (E4!=0.0)
                 {
                     p = i + j * nnx - 2; // lower left node of element
                     q = p + 2;     // node of force
                     Fx=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 0, &Fx, 0);
                     Fx *= E4;
                     FxS+=Fx;
                 }
                 
                 Fx=FxS;                
                 Fx -= BcValueX[q]; 
                 MFx[q]=Fx;
                 dUx[q]=(-Fx/(E1*K[304]+E2*K[228]+E3*K[0]+E4*K[76]));
                   // 16*18+16=266 12*18+12=228 0+18*0=0 4+18*4=76
                                                  
		         Ux[q]+=lambda*dUx[q]; 
             } 
             
             
             if (0==BcTypeY[q])
             {
                 FyS=0.0;
                 
                 if (E1!=0.0)
                 {
                     p = i + j*nnx - 2*nnx -2; // lower left node of element
                     q = p + 2 + 2*nnx;   // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 2, &Fy, 1);
                     Fy *= E1;
                     FyS+=Fy;
                 } 
                 
                 if (E2!=0.0)
                 {
                     p = i + j * nnx-2*nnx; // lower left node of element
                     q = p + 2*nnx;     // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 2, &Fy, 1);
                     Fy *= E2;
                     FyS+=Fy;
                 } 
                 
                 if (E3!=0.0)
                 {
                     p = i + j * nnx; // lower left node of element
                     q = p;     // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 0, 0, &Fy, 1);
                     Fy *= E3;
                     FyS+=Fy;
                 }
                 
                 if (E4!=0.0)
                 {
                     p = i + j * nnx - 2; // lower left node of element
                     q = p + 2;     // node of force
                     Fy=0.0;
                     GetResForceOfOneQ9(K, &(Ux[p]), &(Uy[p]), nnx, 2, 0, &Fy, 1);
                     Fy *= E4;
                     FyS+=Fy;
                 }
                 
                 Fy=FyS;                
                 Fy -= BcValueY[q]; 
                 MFy[q]=Fy;
                 dUy[q]=(-Fy/(E1*K[323]+E2*K[247]+E3*K[19]+E4*K[95])); 
                     // 17+18*17=323 13*18+13=247 1+18*1=19 5+18*5=95
		         Uy[q]+=lambda*dUy[q]; 
             }
        }
        }
    
    
    
    return(1);
}


