/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/*---------------------------------------------------------
	ComputeEuclideanNorm
----------------------------------------------------------*/
int ComputeEuclideanNorm()
{
    
    MGBlock *Meshes, *M;
    int     num_levels;
    SolveBlock *SolBlock;
    int ModelDim;
    int i,j,k;
    double error;
    int returnvalue;
    
    returnvalue=0;
    
    PutGetSolveBlock(1,&SolBlock); 
    
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    ModelDim=PutGetModelDim(1,123);
    
    if (2==ModelDim)
        GetRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
        M->MatFieldInt, SolBlock->Material,
        M->Ux,  M->Uy, M->Adx, M->Ady, 
        M->BcTypeX, M->BcTypeY, M->BcValueX, M->BcValueY); 
    else    
        GetRestoringForce3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        M->MatFieldInt, M->Ux,  M->Uy, M->Uz, M->Adx,  M->Ady, M->Adz,
        M->BcTypeX, M->BcTypeY, M->BcTypeZ, M->BcValueX, M->BcValueY, M->BcValueZ);
    
     if (2==ModelDim)
        ScalarProduct((ModelDim*(M->nnx)*(M->nny)), M->Ad, M->Ad, &error);
     else
        ScalarProduct((ModelDim*(M->nnx)*(M->nny)*(M->nnz)), M->Ad, M->Ad, &error);
    
    error=sqrt(error);
    
    ChangeErrorValue(0,error);
    
    returnvalue=CheckErrorTol(0, error); 
    
return returnvalue;
}

/*------------------------------------------------------------------------------
     GetErrorEuclidean
------------------------------------------------------------------------------*/
double GetErrorEuclidean(void)
{
    int     i,j,k;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    int ModelDim;
    int nxy,nxyz;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    ModelDim=PutGetModelDim(1,123);
    
    if (2==ModelDim)
        PostRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
        M->MatFieldInt, SolBlock->Material,
        M->Ux,  M->Uy, M->Adx, M->Ady); 
    else    
        PostRestoringForce3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        M->MatFieldInt, M->Ux,  M->Uy, M->Uz, M->Adx,  M->Ady, M->Adz);
        
    Energy=0.0;
   
    nxy=M->nnx*M->nny;
    nxyz=nxy*M->nnz;
    if (2==ModelDim)
    {
        for(i=0; i<nxy; i++)
        {
            if (0==(M->BcTypeX[i])) Energy+=(((M->Adx[i])-(M->BcValueX[i]))*((M->Adx[i])-(M->BcValueX[i])));
            if (0==(M->BcTypeY[i])) Energy+=(((M->Ady[i])-(M->BcValueY[i]))*((M->Ady[i])-(M->BcValueY[i])));
        }
    }
    else 
    {
        for(i=0; i<nxyz; i++)
        {
            if (0==(M->BcTypeX[i])) Energy+=fabs( ((M->Adx[i])-(M->BcValueX[i])) * ((M->Adx[i])-(M->BcValueX[i])) );
            if (0==(M->BcTypeY[i])) Energy+=fabs( ((M->Ady[i])-(M->BcValueY[i])) * ((M->Ady[i])-(M->BcValueY[i])) );
            if (0==(M->BcTypeZ[i])) Energy+=fabs( ((M->Adz[i])-(M->BcValueZ[i])) * ((M->Adz[i])-(M->BcValueZ[i])) );
        }
    }
    return(sqrt(Energy));
}

/*------------------------------------------------------------------------------
     GetSystemEuclidean
------------------------------------------------------------------------------*/
double GetSystemEuclidean(void)
{
    int     i,j,k;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    int ModelDim;
    int nxy,nxyz;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    ModelDim=PutGetModelDim(1,123);
    
    Energy=0.0;
   
    nxy=M->nnx*M->nny;
    nxyz=nxy*M->nnz;
    if (2==ModelDim)
    {
        for(i=0; i<nxy; i++)
        {
            if (0==(M->BcTypeX[i]) && 0.0!=(M->BcValueX[i])) Energy+=((M->BcValueX[i])*(M->BcValueX[i]));
                else if (1==(M->BcTypeX[i]))                 Energy+=((M->Adx[i])*(M->Adx[i]));
            if (0==(M->BcTypeY[i]) && 0.0!=(M->BcValueY[i])) Energy+=((M->BcValueY[i])*(M->BcValueY[i]));
                else if (1==(M->BcTypeY[i]))                 Energy+=((M->Ady[i])*(M->Ady[i]));
        }
    }
    else 
    {
        for(i=0; i<nxyz; i++)
        {  
            if (0==(M->BcTypeX[i]) && 0.0!=(M->BcValueX[i])) Energy+=((M->BcValueX[i])*(M->BcValueX[i]));
                else if (1==(M->BcTypeX[i]))                 Energy+=((M->Adx[i])*(M->Adx[i]));
            if (0==(M->BcTypeY[i]) && 0.0!=(M->BcValueY[i])) Energy+=((M->BcValueY[i])*(M->BcValueY[i]));
                else if (1==(M->BcTypeY[i]))                 Energy+=((M->Ady[i])*(M->Ady[i])); 
            if (0==(M->BcTypeZ[i]) && 0.0!=(M->BcValueZ[i])) Energy+=((M->BcValueZ[i])*(M->BcValueZ[i]));
                else if (1==(M->BcTypeZ[i]))                 Energy+=((M->Adz[i])*(M->Adz[i]));
        }
    }
    return(sqrt(Energy));
}

/*---------------------------------------------------------
	ComputeRelEuclideanError
----------------------------------------------------------*/
int ComputeRelEuclideanError()
{

    double error;
    double SystemEnergy;
    double ErrorEnergy;
    int    returnvalue;
    
    // Order is important !
    // First Error Energy and then System Energy
    ErrorEnergy=GetErrorEuclidean();
    SystemEnergy=GetSystemEuclidean();
    if (0.0==SystemEnergy) SystemEnergy=1.0;
    
    error=ErrorEnergy/SystemEnergy;
    //print("Error Energy: %g SystemEnergy:%g", 
    ChangeErrorValue(1,error);
    returnvalue=CheckErrorTol(1, error); 
    
return returnvalue;
}



/*------------------------------------------------------------------------------
     GetErrorEnergy
------------------------------------------------------------------------------*/
double GetErrorEnergy(void)
{
    int     i,j,k;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    int ModelDim;
    int nxy,nxyz;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    ModelDim=PutGetModelDim(1,123);
    
    if (2==ModelDim)
        PostRestoringForce2(M->nnx, M->nny, M->MatFieldE, 
        M->MatFieldInt, SolBlock->Material,
        M->Ux,  M->Uy, M->Adx, M->Ady); 
    else    
        PostRestoringForce3d(M->nnx, M->nny, M->nnz, SolBlock->Material, M->MatFieldE, 
        M->MatFieldInt, M->Ux,  M->Uy, M->Uz, M->Adx,  M->Ady, M->Adz);
        
    Energy=0.0;
   
    nxy=M->nnx*M->nny;
    nxyz=nxy*M->nnz;
    if (2==ModelDim)
    {
        for(i=0; i<nxy; i++)
        {
            if (0==(M->BcTypeX[i])) Energy+=fabs( (M->Ux[i]) * ((M->Adx[i])-(M->BcValueX[i])) );
            if (0==(M->BcTypeY[i])) Energy+=fabs( (M->Uy[i]) * ((M->Ady[i])-(M->BcValueY[i])) );
        }
    }
    else 
    {
        for(i=0; i<nxyz; i++)
        {
            if (0==(M->BcTypeX[i])) Energy+=fabs( (M->Ux[i]) * ((M->Adx[i])-(M->BcValueX[i])) );
            if (0==(M->BcTypeY[i])) Energy+=fabs( (M->Uy[i]) * ((M->Ady[i])-(M->BcValueY[i])) );
            if (0==(M->BcTypeZ[i])) Energy+=fabs( (M->Uz[i]) * ((M->Adz[i])-(M->BcValueZ[i])) );
        }
    }
    return(Energy);
}

/*------------------------------------------------------------------------------
     GetSystemEnergy
------------------------------------------------------------------------------*/
double GetSystemEnergy(void)
{
    int     i,j,k;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    int ModelDim;
    int nxy,nxyz;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    ModelDim=PutGetModelDim(1,123);
    
    Energy=0.0;
   
    nxy=M->nnx*M->nny;
    nxyz=nxy*M->nnz;
    if (2==ModelDim)
    {
        for(i=0; i<nxy; i++)
        {
            if (0==(M->BcTypeX[i]) && 0.0!=(M->BcValueX[i])) Energy+=((M->Ux[i])*(M->BcValueX[i]));
                else if (1==(M->BcTypeX[i]))                 Energy+=((M->Ux[i])*(M->Adx[i]));
            if (0==(M->BcTypeY[i]) && 0.0!=(M->BcValueY[i])) Energy+=((M->Uy[i])*(M->BcValueY[i]));
                else if (1==(M->BcTypeY[i]))                 Energy+=((M->Uy[i])*(M->Ady[i]));
        }
    }
    else 
    {
        for(i=0; i<nxyz; i++)
        {  
            if (0==(M->BcTypeX[i]) && 0.0!=(M->BcValueX[i])) Energy+=((M->Ux[i])*(M->BcValueX[i]));
                else if (1==(M->BcTypeX[i]))                 Energy+=((M->Ux[i])*(M->Adx[i]));
            if (0==(M->BcTypeY[i]) && 0.0!=(M->BcValueY[i])) Energy+=((M->Uy[i])*(M->BcValueY[i]));
                else if (1==(M->BcTypeY[i]))                 Energy+=((M->Uy[i])*(M->Ady[i])); 
            if (0==(M->BcTypeZ[i]) && 0.0!=(M->BcValueZ[i])) Energy+=((M->Uz[i])*(M->BcValueZ[i]));
                else if (1==(M->BcTypeZ[i]))                 Energy+=((M->Uz[i])*(M->Adz[i]));
        }
    }
    return(Energy);
}

/*---------------------------------------------------------
	ComputeRelEnergyError
----------------------------------------------------------*/
int ComputeRelEnergyError()
{

    double error;
    double SystemEnergy;
    double ErrorEnergy;
    int    returnvalue;
    
    // Order is important !
    // First Error Energy and then System Energy
    ErrorEnergy=GetErrorEnergy();
    SystemEnergy=GetSystemEnergy();
    if (0.0==SystemEnergy) SystemEnergy=1.0;
    
    error=ErrorEnergy/SystemEnergy;
    //print("Error Energy: %g SystemEnergy:%g", 
    ChangeErrorValue(2,error);
    returnvalue=CheckErrorTol(2, error); 
    
return returnvalue;
}

/*---------------------------------------------------------
	ChangeErrorValue
----------------------------------------------------------*/
void ChangeErrorValue(int type, double value)
{
    SolveBlock *SolBlock;
    PutGetSolveBlock(1,&SolBlock); 
    SolBlock->ErrorData.ErrValue[type]=value;

return;
}

/*---------------------------------------------------------
	CheckErrorTol
----------------------------------------------------------*/
int CheckErrorTol(int type, double value)
{
    SolveBlock *SolBlock;
    int returnvalue;
    
    returnvalue=0;
    PutGetSolveBlock(1,&SolBlock); 
    if (value<SolBlock->ErrorData.ErrTol[type]) returnvalue=1; 
    //printf("\n%g<%g\n",value,SolBlock->ErrorData.ErrTol[type]);

return returnvalue;
}

/*---------------------------------------------------------
	PrintErrorData
----------------------------------------------------------*/
void PrintErrorData(void)
{
    double *Value;
    SolveBlock *SolBlock;
    FILE   *datei;
    char str[36]="Logfile.dat";
    
    PutGetSolveBlock(1,&SolBlock);  
    Value=SolBlock->ErrorData.ErrValue;
    printf("\n%i: Euclid Err.:%g Rel. Euclid Err.:%g Rel. Energy Err.:%g ",SolBlock->i, Value[0], Value[1], Value[2]);   
    
 // if ((datei=fopen(str, "awb"))==NULL) return;
 // fprintf(datei,"\nEuclidean norm:%g :Rel. Energy error:%g ",Value[0], Value[2]); 
 // fclose(datei);
return;
}
