/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*------------------------------------------------------------------------------
	BuildGaussValues   !neu -n16-!
------------------------------------------------------------------------------*/

int BuildGaussValues(GaussValues *GBV, int numXi)
{
    int i,j,w,z1,z2;
    double xpo,xne,xn,xnn;
    int existnumXi;
    double leftlimit,rightlimit;
    double **newGv, **newXv;
    PolynomStruct *newPoly;
    PolynomStruct PS1,PS2,PS3,PS4;
    int maxIter=5000;               //values for bisection iteration
    double precision=1e-14;
    
    existnumXi=GBV->numXi;
    
    //create new mainregister
    
    InitPolynomStruct(&(newPoly), numXi+1);    
    newXv =(double**) malloc((numXi+1)*sizeof(double*));
    newGv =(double**) malloc((numXi+1)*sizeof(double*));    
    
    //move subvectors
    
    for(i=0; i<=existnumXi; i++)
    {
     newPoly[i]=GBV->LegPoly.Polynom[i];
     newXv[i]=GBV->Xv[i];
     newGv[i]=GBV->Gv[i];     
    }
    
    //delete old mainregister
    
    free(GBV->LegPoly.Polynom);
    free(GBV->Xv);    
    free(GBV->Gv);
    
    //link new mainregister
    
    GBV->LegPoly.Polynom=newPoly;
    GBV->Xv=newXv;
    GBV->Gv=newGv;
    
    //init new subvectors
    
    for(i=existnumXi+1; i<=numXi; i++)
    {
     InitPolynomFunction(&(GBV->LegPoly.Polynom[i]),1);
     InitVector_double(i+1,&(GBV->Xv[i]));
     InitVector_double(i+1,&(GBV->Gv[i]));    
    }
    
    //get Legendre-polynoms
    
    InitPolynomFunction( &PS3,1);
    InitPolynomFunction( &PS4,2);
    
    for(i=existnumXi+1; i<=numXi; i++)
    {
     InitPolynomFunction( &PS1, 1);
     InitPolynomFunction( &PS2, 1);
     
     CopyPolynoms(&PS1,&(GBV->LegPoly.Polynom[i-2]));
     CopyPolynoms(&PS2,&(GBV->LegPoly.Polynom[i-1]));
     
     PS3.coefficient[0]=1.0/(double)i -1.0;
     PS4.coefficient[1]=2.0 -1.0/(double)i;
     
     MultPolynoms(&PS1,&PS3);
     MultPolynoms(&PS2,&PS4);
     
     AddPolynoms(&PS1,&PS2);
     CopyPolynoms(&(GBV->LegPoly.Polynom[i]),&PS1);
     
     DeletePolynomFunction( &PS1);
     DeletePolynomFunction( &PS2);
    }
    DeletePolynomFunction( &PS3);
    DeletePolynomFunction( &PS4);
    
    //get samplingpoints (bisection-method)
    
    for(i=existnumXi+1; i<=numXi; i++)
    {
     w=(int)floor((double)i/2.0);
     
     InitPolynomFunction(&PS1,1);
     InitPolynomFunction(&PS2,1);
     InitPolynomFunction(&PS3,1);
     CopyPolynoms(&PS1,&(GBV->LegPoly.Polynom[i]));
     CopyPolynoms(&PS2,&(GBV->LegPoly.Polynom[i]));
     CopyPolynoms(&PS3,&(GBV->LegPoly.Polynom[i-1]));
     DerivePolynomStruct(&PS2);     
     
     for(j=1; j<=w; j++)
     {
      z1=0; z2=0; xnn=1.0;
      leftlimit=-cos(((double)j-0.5)*M_PI/((double)i+0.5));
      rightlimit=-cos((double)j*M_PI/((double)i+0.5));
      
      if((PolyVal(PS1,leftlimit)>0) && (PolyVal(PS1,rightlimit)<0))
        {z1=1; xpo=leftlimit; xne=rightlimit;}
      
      if((PolyVal(PS1,leftlimit)<0) && (PolyVal(PS1,rightlimit)>0))
        {z1=2; xpo=rightlimit; xne=leftlimit;}      
      
      if (z1 == 0)
      {
       z1=4; 
       printf("\n\nwrong given intervall "); 
       //printf("i=%d  j=%d  leftlimit=%20.18f  rightlimit=%20.18f",i,j,leftlimit,rightlimit);
      }      
      
      while ( (fabs(xpo-xne)) > precision && z2 < maxIter && z1 != 4 && xnn == 1.0)
      {
       xn=(xpo+xne)/2.0;
       xnn=0.0;
       if (PolyVal(PS1,xn)<0) {xnn=1.0; xne=xn;}
       if (PolyVal(PS1,xn)>0) {xnn=1.0; xpo=xn;}
       z2+=1; 
      }
            
      if (xnn == 0.0)
      {/*z2=-z2;*/ printf("\n\nNullstelle exakt gefunden");}
      
      if (z2 == maxIter || z2 == -maxIter)
      {z1=4; printf("\n\ncannot find sampling point with less than %d iterations",maxIter);}
      
      if (z1 == 4)
      {
       numXi=i-1;      
       printf("\n\n!!! samplepoint iteration interrupted !!!");
       //printf("\n\nmaximal number of iterations = %d",maxIter);
       //printf("\niteration precision = %1.0e",precision);      
       printf("\nmaximal available number of sampling points = %d",numXi);
       break;
      } 
                 
      GBV->Xv[i][j]=xn;
      GBV->Xv[i][i+1-j]=-xn;
      
      GBV->Gv[i][j]=2/((PolyVal(PS3,xn))*(PolyVal(PS2,xn))*(double)i);
      GBV->Gv[i][i+1-j]=GBV->Gv[i][j];
      if (j+2 == i+1-j) 
      {GBV->Gv[i][j+1]=2.0/((PolyVal(PS3,0))*(PolyVal(PS2,0))*(double)i);}
     }
     DeletePolynomFunction( &PS1);
     DeletePolynomFunction( &PS2);
     DeletePolynomFunction( &PS3); 
     if (z1 == 4) break;   
    }
    
    GBV->numXi=numXi;
    return numXi;
}


/*------------------------------------------------------------------------------
	GetSamplePoints   !neu -n17-!
------------------------------------------------------------------------------*/

int GetSamplePoints( double **Xvn, double **Gvn, int numXi)
{
    int i,existnumXi;    
    GaussValues *GBV;
    
    if (*Xvn != NULL) free(*Xvn);
    if (*Gvn != NULL) free(*Gvn);
    
    PutGetGaussValues (1,&GBV);    
    existnumXi=GBV->numXi;    
    
    if (numXi > existnumXi) numXi=BuildGaussValues(GBV,numXi);

    InitVector_double(numXi,Xvn);
    InitVector_double(numXi,Gvn);    
    
    for(i=0; i<numXi; i++)
    {
     (*Xvn)[i]=GBV->Xv[numXi][i+1];
     (*Gvn)[i]=GBV->Gv[numXi][i+1];
    }
    
return numXi;    
}


/*------------------------------------------------------------------------------
	GetSamplePointsToBSP   !neu -n13-!
------------------------------------------------------------------------------*/

int GetSamplePointsToBSP(BSplineBlock *BSP,int numXi)
{
    if (BSP->Xi != NULL) free(BSP->Xi);
    if (BSP->Gi != NULL) free(BSP->Gi);    
    
    InitVector_double(1,&(BSP->Xi));
    InitVector_double(1,&(BSP->Gi));
        
    numXi=GetSamplePoints( &(BSP->Xi), &(BSP->Gi), numXi);
    
    BSP->numXi = numXi;
        
    //PrintMatrixT(1,numXi,BSP->Xi);
    //PrintMatrixT(1,numXi,BSP->Gi);
            
    return numXi;
}

/*------------------------------------------------------------------------------
	InitSamplePoints   !neu -n15-!
------------------------------------------------------------------------------*/

void InitSamplePoints(int n)
{
    int i;
    static GaussValues Gaussblock ;
    static GaussValues *loc_ptr;

    loc_ptr=&Gaussblock;    
    PutGetGaussValues(0,&loc_ptr);
    
    //alloc memory
    
    InitPolynomStruct(&(Gaussblock.LegPoly.Polynom), 2);
    Gaussblock.Xv =(double**) malloc((2)*sizeof(double*));
    Gaussblock.Gv =(double**) malloc((2)*sizeof(double*));
    
    for(i=0; i<=1; i++)
    {    
     InitPolynomFunction(&(Gaussblock.LegPoly.Polynom[i]),i+1);
     InitVector_double(i+1,&(Gaussblock.Xv[i]));
     InitVector_double(i+1,&(Gaussblock.Gv[i]));
    }    
        
    //first Legendre-Polynom, sample point and weights values
        
    Gaussblock.LegPoly.Polynom[0].coefficient[0]=1.0;
    Gaussblock.LegPoly.Polynom[1].coefficient[1]=1.0;    
    Gaussblock.Gv[1][1]=2.0;
        
    Gaussblock.numXi=1;
    
    //expand Gauss-Basis to n    
    
    BuildGaussValues(loc_ptr,n);

    return;    
}


/*------------------------------------------------------------------------------
     PutGetGaussValues   !neu -n14-!
------------------------------------------------------------------------------*/
void PutGetGaussValues(int choice, GaussValues **inblock)
{
     static GaussValues *localblock;
     if (0==choice) {localblock=(*inblock);}
     if (1==choice) {(*inblock)=localblock;}
}


/*------------------------------------------------------------------------------
	GetSamplePointsToBSP_old   !neu -n18-!
------------------------------------------------------------------------------*/

int GetSamplePointsToBSP_old(BSplineBlock *BSP,int numXi)
{
     int i;
     static double *Xv[12],*Gv[12];
     
     for(i=0; i<12; i++)
     {
      Xv[i]=NULL; Gv[i]=NULL;
     }
     
     static double   Xi3[3]={-0.77459666924148337704,  0.0,
                              0.77459666924148337704};
                     Xv[3]=Xi3;         
                            
     static double   Gi3[3]={ 0.55555555555555555556,  0.8888888888888888889,
                              0.55555555555555555556};
                     Gv[3]=Gi3;
                     
     static double   Xi5[5]={-0.90617984593866399280, -0.53846931010568309104,
                              0.0,                     0.53846931010568309104,
                              0.90617984593866399280};
                     Xv[5]=Xi5;           
     
     static double   Gi5[5]={ 0.23692688505618908751,  0.47862867049936646806,
                              0.56888888888888888892,  0.47862867049936646806,
                              0.23692688505618908751};
                     Gv[5]=Gi5;                      
                     
     static double   Xi7[7]={-0.94910791234275852453, -0.74153118559939443986,
                             -0.40584515137739716691,  0.0,
                              0.40584515137739716691,  0.74153118559939443986,
                              0.94910791234275852453};
                     Xv[7]=Xi7;           
     
     static double   Gi7[7]={ 0.12948496616886969324,  0.27970539148927666803,
                              0.38183005050511894471,  0.41795918367346938810,
                              0.38183005050511894471,  0.27970539148927666803,
                              0.12948496616886969324};
                     Gv[7]=Gi7;                     


     static double   Xi9[9]={-0.96816023950762608984, -0.83603110732663579430,
                             -0.61337143270059039731, -0.32425342340380892904,
                              0.0,                     0.32425342340380892904,
                              0.61337143270059039731,  0.83603110732663579430,
                              0.96816023950762608984};
                     Xv[9]=Xi9;           
     
     static double   Gi9[9]={ 0.08127438836157441199,  0.18064816069485740397,
                              0.26061069640293546255,  0.31234707704000283966,
                              0.33023935500125976371,  0.31234707704000283966,
                              0.26061069640293546255,  0.18064816069485740397,
                              0.08127438836157441199};
                     Gv[9]=Gi9;
                             
     
     static double Xi10[10]={-0.97390652851717172008, -0.86506336668898451073,
                             -0.67940956829902440623, -0.43339539412924719080,
                             -0.14887433898163121088,  0.14887433898163121088, 
                              0.43339539412924719080,  0.67940956829902440623,
                              0.86506336668898451073,  0.97390652851717172008};
                   Xv[10]=Xi10;           
     
     static double Gi10[10]={ 0.06667134430868813768,  0.14945134915058059284,
                              0.21908636251598204456,  0.26926671930999635443,
                              0.29552422471475287048,  0.29552422471475287048,
                              0.26926671930999635443,  0.21908636251598204456,
                              0.14945134915058059284,  0.06667134430868813768};
                   Gv[10]=Gi10;


     static double Xi11[11]={-0.97822865814605699280, -0.88706259976809529908,
                             -0.73015200557404932409, -0.51909612920681181593,
                             -0.26954315595234497233,  0.0,
                              0.26954315595234497233,  0.51909612920681181593,
                              0.73015200557404932409,  0.88706259976809529908,
                              0.97822865814605699280};
                   Xv[11]=Xi11;           
     
     static double Gi11[11]={ 0.05566856711617366654,  0.12558036946490462439,
                              0.18629021092773425205,  0.23319376459199047866,
                              0.26280454451024666423,  0.27292508677790062822,
                              0.26280454451024666423,  0.23319376459199047866,
                              0.18629021092773425205,  0.12558036946490462439,
                              0.05566856711617366654};
                   Gv[11]=Gi11;

     
     if (numXi > 11) numXi=11;
     
     while ( Xv[numXi] == NULL ) numXi += 1;

     BSP->Xi = Xv[numXi]; 
     BSP->Gi = Gv[numXi];
     BSP->numXi = numXi;
     
     //PrintMatrixT(1,numXi,BSP->Xi);
     //PrintMatrixT(1,numXi,BSP->Gi);
            
return numXi;
}
