/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyMainWindowUI.h"
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>

/*------------------------------------------------------------------------------
     PostEnergies
------------------------------------------------------------------------------*/

void PostEnergies(void)
{
    double ExtEnergy;
    double IntEnergy;
    
    
    ExtEnergy=PostSystemEnergy();
    IntEnergy=PostInternalEnergy();
    
    printf("\nInt. Energy: %18.12g   Ext. Energy: %18.12g\n", IntEnergy, ExtEnergy);
    
    return;
}

/*------------------------------------------------------------------------------
     PostInternalEnergyWrong
------------------------------------------------------------------------------*/

double PostInternalEnergyWrong(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    PostRestoringForce2(M->nnx, M->nny, M->MatFieldE,               
                        M->MatFieldInt, SolBlock->Material,
                        M->Ux,  M->Uy,
			            M->SwapX, M->SwapY);      
    
                
    //PrintMatrix(M->nnx, M->nny, M->Ux); 
    //PrintMatrix(M->nnx, M->nny, M->Uy);
    //PrintMatrix(M->nnx, M->nny, M->SwapX);
    //PrintMatrix(M->nnx, M->nny, M->SwapY);                                   
    //PostRestoringForce(M->nnx, M->nny, SolBlock->K_matrix, M->MatFieldE, 
    //                  M->Ux, M->Uy, M->SwapX, M->SwapY);
    Energy=0.0;
    for(i=0; i<M->nex; i++)
    {
        for(j=0; j<M->ney; j++)
        {
            Energy+=
            M->Ux[i+j*M->nnx]      *M->SwapX[i+j*M->nnx]+
            M->Ux[i+1+j*M->nnx]    *M->SwapX[i+1+j*M->nnx]+
            M->Ux[i+(j+1)*M->nnx]  *M->SwapX[i+(j+1)*M->nnx]+
            M->Ux[i+1+(j+1)*M->nnx]*M->SwapX[i+1+(j+1)*M->nnx]+
            M->Uy[i+j*M->nnx]      *M->SwapY[i+j*M->nnx]+
            M->Uy[i+1+j*M->nnx]    *M->SwapY[i+1+j*M->nnx]+
            M->Uy[i+(j+1)*M->nnx]  *M->SwapY[i+(j+1)*M->nnx]+
            M->Uy[i+1+(j+1)*M->nnx]*M->SwapY[i+1+(j+1)*M->nnx];
        }
    }
    //printf("\nEnergy: %g\n", Energy);
    
    return(Energy);
}


/*------------------------------------------------------------------------------
     PostInternalEnergy
------------------------------------------------------------------------------*/

double PostInternalEnergy(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    
    double u1,u2,u3,u4,
           v1,v2,v3,v4;
    double E,nu;
    double la,lb;
    int    p;
    double *MatFieldE;
    int    *MatFieldInt;
    MaterialBlock *Mat;
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);    
                                                

    Energy=0.0;
    la=M->la;
    lb=M->lb;
    MatFieldInt=M->MatFieldInt;
    MatFieldE=M->MatFieldE;
    Mat=SolBlock->Material;
    
    for(i=0; i<M->nex; i++)
    {
        for(j=0; j<M->ney; j++)
        {
            u1=M->Ux[i+1+(j+1)*M->nnx];
            u2=M->Ux[i+(j+1)*M->nnx];
            u3=M->Ux[i+(j)*M->nnx];
            u4=M->Ux[i+1+(j)*M->nnx];
            
            v1=M->Uy[i+1+(j+1)*M->nnx];
            v2=M->Uy[i+(j+1)*M->nnx];
            v3=M->Uy[i+(j)*M->nnx];
            v4=M->Uy[i+1+(j)*M->nnx];
            
            p=i+j*M->nex;
            E= MatFieldE[p]; 
            nu=Mat[MatFieldInt[p]].Mu;
            
            Energy+=(double) (E * (2 * nu * v2 * v3 - 4 * nu * u1 * u4 - 4 * nu 
            * u2 * u3 - 2 * nu * u2 * u4 + 2 * nu * u2 * u1 - 2 * nu * v2 * v4 
            + 2 * nu * v1 * v4 - 2 * nu * v1 * v3 - 4 * nu * v1 * v2 + 3 * nu 
            * v4 * u4 - 9 * nu * v4 * u3 - 3 * nu * v4 * u2 + 9 * nu * v4 * u1 
            + 9 * nu * v3 * u4 - 3 * nu * v3 * u3 - 9 * nu * v3 * u2 + 3 * nu 
            * v3 * u1 + 9 * nu * v2 * u3 + 3 * nu * v2 * u2 - 9 * nu * v2 * u1 
            - 9 * nu * v1 * u4 + 3 * nu * v1 * u3 + 9 * nu * v1 * u2 - 3 * nu 
            * v1 * u1 + 2 * nu * u4 * u3 - 2 * nu * u1 * u3 - 4 * nu * v3 * v4 
            + 2 * nu * u2 * u2 - 6 * v3 * v3 + 6 * v2 * v4 + 6 * v2 * v3 - 6 
            * v2 * v2 + 6 * v1 * v4 + 6 * v1 * v3 - 6 * v1 * v1 + 2 * nu * v2 
            * v2 - 6 * v4 * v4 + 6 * u2 * u4 + 6 * u2 * u1 - 6 * u2 * u2 + 2 
            * nu * v4 * v4 + 2 * nu * v3 * v3 - 6 * u3 * u3 + 6 * u4 * u3 - 6 
            * u4 * u4 + 6 * u1 * u3 - 6 * u1 * u1 + 2 * nu * u1 * u1 + 2 * nu 
            * u3 * u3 + 2 * nu * u4 * u4 + 3 * v1 * u3 - 3 * v1 * u2 + 3 * v1 
            * u4 - 3 * v3 * u3 + 3 * v3 * u2 - 3 * v3 * u4 - 3 * v2 * u3 + 3 
            * v2 * u2 - 3 * v2 * u4 - 3 * u1 * v4 - 3 * u1 * v1 + 3 * u1 * v3 
            + 3 * u1 * v2 + 3 * v4 * u3 - 3 * u2 * v4 + 3 * u4 * v4 + 2 * nu 
            * v1 * v1 - 3 * nu * v2 * u4) / (-1 + nu * nu)) / 0.12e2;
           
             
        }
    }
    //printf("\nEnergy: %g\n", Energy);
    
    return(Energy);
}

/*------------------------------------------------------------------------------
     PostAverages
------------------------------------------------------------------------------*/

double PostAverages(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    
    double u1,u2,u3,u4,
           v1,v2,v3,v4;
    double E,nu;
    double la,lb;
    int    p;
    double *MatFieldE;
    int    *MatFieldInt;
    MaterialBlock *Mat;
    
    double eps1,eps2,eps3;
    double sig1,sig2,sig3;
    double epssig1,epssig2,epssig3;
    double factor;
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);    
                                                

    la=M->la;
    lb=M->lb;
    MatFieldInt=M->MatFieldInt;
    MatFieldE=M->MatFieldE;
    Mat=SolBlock->Material;
    
    eps1=0.0;eps2=0.0;eps3=0.0;
    sig1=0.0;sig2=0.0;sig3=0.0;
    
    for(i=0; i<M->nex; i++)
    {
        for(j=0; j<M->ney; j++)
        {
            u1=M->Ux[i+1+(j+1)*M->nnx]*(2/la);
            u2=M->Ux[i+(j+1)*M->nnx]*(2/la);
            u3=M->Ux[i+(j)*M->nnx]*(2/la);
            u4=M->Ux[i+1+(j)*M->nnx]*(2/la);
            
            v1=M->Uy[i+1+(j+1)*M->nnx]*(2/lb);
            v2=M->Uy[i+(j+1)*M->nnx]*(2/lb);
            v3=M->Uy[i+(j)*M->nnx]*(2/lb);
            v4=M->Uy[i+1+(j)*M->nnx]*(2/lb);
            
            p=i+j*M->nex;
            E= MatFieldE[p]; 
            nu=Mat[MatFieldInt[p]].Mu;
            
            eps1+=u1-u2-u3+u4;
            eps2+=v1+v2-v3-v4;
            eps3+=u1-v2-v3+v1-u4+u2+v4-u3;
            
            sig1+=0.4e1 * E / (double) (1 - nu * nu) * 
            (u1 / 0.4e1 - u2 / 0.4e1 - u3 / 0.4e1 + u4 / 0.4e1) + 0.4e1 *
            E / (double) (1 - nu * nu) * (double) nu * 
            (v1 / 0.4e1 + v2 / 0.4e1 - v3 / 0.4e1 - v4 / 0.4e1);
            
            sig2+=0.4e1 * E / (double) (1 - nu * nu) * 
            (double) nu * (u1 / 0.4e1 - u2 / 0.4e1 - u3 / 0.4e1 + u4 / 0.4e1) 
            + 0.4e1 * E / (double) (1 - nu * nu) * (v1 / 0.4e1 + v2 / 0.4e1 
            - v3 / 0.4e1 - v4 / 0.4e1);
            
            sig3+=0.4e1 * E / (double) (1 - nu * nu) * 
            (0.1e1 / 0.2e1 - (double) nu / 0.2e1) * 
            (u1 / 0.4e1 - v2 / 0.4e1 - v3 / 0.4e1 + v1 / 0.4e1 - u4 / 
            0.4e1 + u2 / 0.4e1 + v4 / 0.4e1 - u3 / 0.4e1);    
            
            /*    
            epssig1+=0.4e1 / 0.3e1 * E / (double) (1 - nu * nu) * 
            pow(-u2 / 0.4e1 + u1 / 0.4e1 - u4 / 0.4e1 + u3 / 0.4e1, 0.2e1)
            + 0.4e1 * (u1 / 0.4e1 - u2 / 0.4e1 - u3 / 0.4e1 + u4 / 0.4e1) 
            * (E / (double) (1 - nu * nu) * (u1 / 0.4e1 - u2 / 0.4e1 - u3 
            / 0.4e1 + u4 / 0.4e1) + E / (double) (1 - nu * nu) * (double) 
            nu * (v1 / 0.4e1 + v2 / 0.4e1 - v3 / 0.4e1 - v4 / 0.4e1));
            
            epssig2+=0.4e1 / 0.3e1 * E / (double) (1 - nu * nu) * 
            pow(v1 / 0.4e1 - v2 / 0.4e1 + v3 / 0.4e1 - v4 / 0.4e1, 0.2e1) 
            + 0.4e1 * (v1 / 0.4e1 + v2 / 0.4e1 - v3 / 0.4e1 - v4 / 0.4e1) 
            * (E / (double) (1 - nu * nu) * (double) nu * (u1 / 0.4e1 - u2 
            / 0.4e1 - u3 / 0.4e1 + u4 / 0.4e1) + E / (double) (1 - nu * nu) 
            * (v1 / 0.4e1 + v2 / 0.4e1 - v3 / 0.4e1 - v4 / 0.4e1));
            
            epssig3+=0.4e1 / 0.3e1 * E / (double) (1 - nu * nu) * pow(v1 / 
            0.4e1 - v2 / 0.4e1 + v3 / 0.4e1 - v4 / 0.4e1, 0.2e1) * (0.1e1 / 
            0.2e1 - (double) nu / 0.2e1) + 0.4e1 / 0.3e1 * E / (double) (1 
            - nu * nu) * pow(-u2 / 0.4e1 + u1 / 0.4e1 - u4 / 0.4e1 + u3 / 
            0.4e1, 0.2e1) * (0.1e1 / 0.2e1 - (double) nu / 0.2e1) + 0.4e1 
            * pow(u1 / 0.4e1 - v2 / 0.4e1 - v3 / 0.4e1 + v1 / 0.4e1 - u4 / 
            0.4e1 + u2 / 0.4e1 + v4 / 0.4e1 - u3 / 0.4e1, 0.2e1) * E / 
            (double) (1 - nu * nu) * (0.1e1 / 0.2e1 - (double) nu / 0.2e1);
            */
        }
    }
    
    factor=(1/2.0)*(1/2.0)/((M->nex*M->ney));
    
    eps1*=factor;
    eps2*=factor;
    eps3*=factor;
    
    sig1*=factor;
    sig2*=factor;
    sig3*=factor;
    
    epssig1=(eps1*sig1+eps2*sig2+eps3*sig3)*la*lb*M->nex*M->ney;
    //epssig2*=(la/2.0)*(lb/2.0);
    //epssig3*=(la/2.0)*(lb/2.0);
    
    
    printf("\ne1: %18.12g e2: %18.12g e3: %18.12g",   eps1,eps2,eps3);
    printf("\ns1: %18.12g s2: %18.12g s3: %18.12g\n", sig1,sig2,sig3);
    printf("Energy of Averages: %18.12g", epssig1);
    
    
    //printf("\nEnergy: %g\n", Energy);
    
    return(Energy);
}

/*------------------------------------------------------------------------------
     PerformMC
------------------------------------------------------------------------------*/

void PerformMC(void)
{
    int i;
    SolveBlock *SolBlock;
    PutGetSolveBlock(1,&SolBlock); 
    SolBlock->GoFlag=1;

    
    for(i=0;i<10000;i++)
    {
        OpenInputMesoWin();
        MesoWin_cb_OK();
        OpenInputSolveWin();
        SolveWin_cb_OK();
        //SolverMainLoop();
        PostHomogen1();
    }
    
    SolBlock->GoFlag=0;
    
    // Only for Go-Mode
    //SolveBlock *SolBlock;
    //PutGetSolveBlock(1,&SolBlock); 
    //if (1==SolBlock->GoFlag) return;
    // End of Change  

    return;
}


