/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/*------------------------------------------------------------------------------
       InitMGData
-------------------------------------------------------------------------------*/
void InitMGData (MGBlock **Vector, int MaxBlocks)
{
    /*-------------------------------------------------------------------------
    Purpose: Allocate (MGBlock)-memory.

    Input variables :   
    **Vector                    -  This pointer contains the address
                                   of the pointer, which will be allocated. 
    MaxBlocks                   -  Size of array to be allocated.
    
    Output arguments:   
    *Vector  (RD_Block,sc,out)  -  This pointer contains the start address 
                                   of the allocated memory.
   ----------------------------------------------------------------------------*/
    
    *Vector=(MGBlock*) malloc(MaxBlocks*sizeof(MGBlock));
    return;
}



/*------------------------------------------------------------------------------
     InitMGBlock
-------------------------------------------------------------------------------*/

void InitMGBlock(double lx, double ly, int nex, int ney, MGBlock *MG)
{   
    int nnsize, sizeBC;
    int nnx, nny;
    
    nnx=nex+1;
    nny=ney+1;
    nnsize = nnx*nny;
    
    (MG->nnx) = nnx;
    (MG->nny) = nny;
    (MG->nex) = nex;
    (MG->ney) = ney;
    (MG->la)  = (lx/nex);
    (MG->lb)  = (ly/ney);
    
    InitVector_double(nnsize, &(MG->MatFieldE));
    InitVector_int(nnsize, &(MG->MatFieldInt));
    
    InitVector_double(nnsize, &(MG->Ux));      
    InitVector_double(nnsize, &(MG->Uy));       
    InitVector_double(nnsize, &(MG->dUx));	 
    InitVector_double(nnsize, &(MG->dUy));
    InitVector_double(nnsize, &(MG->Fx));      
    InitVector_double(nnsize, &(MG->Fy)); 
    InitVector_double(nnsize, &(MG->Dx));      
    InitVector_double(nnsize, &(MG->Dy)); 
    InitVector_double(nnsize, &(MG->Adx));
    InitVector_double(nnsize, &(MG->Ady));	
    InitVector_double(nnsize, &(MG->Mpc));		 	 
    
    InitVector_double(nnsize, &(MG->SwapX));	 
    InitVector_double(nnsize, &(MG->SwapY));
    
    InitVector_double(nnsize, &(MG->Swap));
    
    InitVector_double(nnsize, &(MG->Post1));
    InitVector_double(nnsize, &(MG->Post2));
    InitVector_double(nnsize, &(MG->Post3));
    InitVector_double(nnsize, &(MG->Post4));
    
    InitVector_double(nnsize, &(MG->EqStrain));
     
    InitVector_int(nnsize,      &(MG->BcTypeX));    
    InitVector_int(nnsize,      &(MG->BcTypeY));    
    InitVector_double(nnsize,   &(MG->BcValueX)); 
    InitVector_double(nnsize,   &(MG->BcValueY));
    
    InitVector_double(nnsize,   &(MG->Kappa)); 
    InitVector_double(nnsize,   &(MG->DamageOmega));
    
    
    if (nnx>nny) nnsize=nnx; else nnsize=nny;
    InitVector_double(nnsize,   &(MG->weights));
		
    return;
}

/*------------------------------------------------------------------------------
     FreeSolveBlock
-------------------------------------------------------------------------------*/

void FreeSolveBlock(SolveBlock *SB)
{   
    int i;
    int num_levels;
    
    num_levels=SB->num_levels;
      
    for(i=0; i<num_levels; i++)
    {
      
        FreeMGBlock2and3d(&(SB->Meshes[i]));
    }
    

    if (1==SB->PreMeshFlag)
    for(i=0; i<num_levels; i++)
    {
         FreeMGBlock2and3dPreCond(&(SB->PreMesh[i]));
    }
    SB->PreMeshFlag=0;

    free(SB->K_matrix);
		
    return;
}



/*------------------------------------------------------------------------------
     FreeMGBlock
-------------------------------------------------------------------------------*/

void FreeMGBlock(MGBlock *MG)
{   
   
    free(MG->MatFieldE);
    free(MG->MatFieldInt);

    
    free(MG->Ux);      
    free(MG->Uy);       
    free(MG->dUx);	 
    free(MG->dUy);
    free(MG->Fx);      
    free(MG->Fy); 
    free(MG->Dx);      
    free(MG->Dy); 
    free(MG->Adx);      
    free(MG->Ady); 	
    free(MG->Mpc); 	  
    
    free(MG->SwapX);	 
    free(MG->SwapY);
    
    free(MG->Swap); 
    
    free(MG->Post1); 
    free(MG->Post2); 
    free(MG->Post3); 
    free(MG->Post4); 
    
    free(MG->EqStrain); 
    
    free(MG->BcTypeX);
    free(MG->BcTypeY);      
    free(MG->BcValueX); 
    free(MG->BcValueY);
	
    free(MG->Kappa); 
    free(MG->DamageOmega);
   	
	free(MG->weights);
 	
    return;
}

/*------------------------------------------------------------------------------
     CopyMGBlock
-------------------------------------------------------------------------------*/

void CopyMGBlock(MGBlock *M, MGBlock *MG )
{   
    int nnsize; 
        
    (MG->nnx) = (M->nnx);
    (MG->nny) = (M->nny);
    (MG->nex) = (M->nex);
    (MG->ney) = (M->ney);
    (MG->la)  = (M->la);
    (MG->lb)  = (M->lb);
    
    nnsize = (MG->nnx)*(MG->nny);
    
    memcpy(MG->MatFieldE, M->MatFieldE, nnsize*sizeof(double));
    memcpy(MG->MatFieldInt, M->MatFieldInt, nnsize*sizeof(int));
    
    memcpy(MG->Ux,  M->Ux,  nnsize*sizeof(double));
    memcpy(MG->Uy,  M->Uy,  nnsize*sizeof(double));
    memcpy(MG->dUx, M->dUx, nnsize*sizeof(double));
    memcpy(MG->dUy, M->dUy, nnsize*sizeof(double));    
    memcpy(MG->Fx,  M->Fx,  nnsize*sizeof(double));
    memcpy(MG->Fy,  M->Fy,  nnsize*sizeof(double));  
    
    memcpy(MG->SwapX,  M->SwapX,  nnsize*sizeof(double));
    memcpy(MG->SwapY,  M->SwapY,  nnsize*sizeof(double)); 
    
    memcpy(MG->Swap,  M->Swap,  nnsize*sizeof(double));   

    memcpy(MG->Post1,  M->Post1,  nnsize*sizeof(double));
    memcpy(MG->Post2,  M->Post2,  nnsize*sizeof(double));
    memcpy(MG->Post3,  M->Post3,  nnsize*sizeof(double));
    memcpy(MG->Post4,  M->Post4,  nnsize*sizeof(double));
    
    memcpy(MG->EqStrain,  M->EqStrain,  nnsize*sizeof(double));
    memcpy(MG->Kappa,     M->Kappa,  nnsize*sizeof(double));
    memcpy(MG->DamageOmega,  M->DamageOmega,  nnsize*sizeof(double));
    
    
    memcpy(MG->BcTypeX,  M->BcTypeX,  nnsize*sizeof(int));
    memcpy(MG->BcTypeY,  M->BcTypeY,  nnsize*sizeof(int));  

    memcpy(MG->BcValueX,  M->BcValueX,  nnsize*sizeof(double));
    memcpy(MG->BcValueY,  M->BcValueY,  nnsize*sizeof(double));
   
    
    if (M->nnx>M->nny) nnsize=M->nnx; else nnsize=M->nny;
    memcpy(MG->weights,  M->weights,  nnsize*sizeof(double));
		
    return;
}


