/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*-------------------------------------------------------------------------------
	TransMatFieldToCoarse
---------------------------------------------------------------------------------*/

int TransMatFieldToCoarse(MGBlock *Mesh, int num_levels)
{
    int i,j,k;
    int level;
    int nex,ney,nez;
    double *Mfine, *Mcoarse;
    //int  *Mfine_int, *Mcoarse_int;
    int ModelDim;
    
    ModelDim=PutGetModelDim(1,123);
    
    level=num_levels-1;
    
if (2==ModelDim)    
    while (level > 0) 
    {    
        nex=Mesh[level-1].nex;
        ney=Mesh[level-1].ney;
    
        Mfine   = Mesh[level].MatFieldE;
        Mcoarse = Mesh[level-1].MatFieldE;
        
       // Mfine_int = Mesh[level].MatFieldInt;
       // Mcoarse_int = Mesh[level-1].MatFieldInt;
        
	for (i=0; i<nex; i++)
    {	
	    for (j=0; j<ney; j++)
	    {   
	       /* Mcoarse[i+nex*j]=1.0/(1.0/Mfine[(i*2)  +(nex*2)*(j*2)  ]+
		                          1.0/Mfine[(i*2)+1+(nex*2)*(j*2)  ]+
				                  1.0/Mfine[(i*2)  +(nex*2)*(j*2+1)]+
		                          1.0/Mfine[(i*2)+1+(nex*2)*(j*2+1)])*4*1.0; */

	        Mcoarse[i+nex*j]=(Mfine[(i*2)  +(nex*2)*(j*2)  ]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2)  ]+
				              Mfine[(i*2)  +(nex*2)*(j*2+1)]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2+1)])/4.0;

	        /*Mcoarse_int[i+nex*j]=(int)floor(((Mfine_int[(i*2)  +(nex*2)*(j*2)  ]+
		                          Mfine_int[(i*2)+1+(nex*2)*(j*2)  ]+
				                  Mfine_int[(i*2)  +(nex*2)*(j*2+1)]+
		                          Mfine_int[(i*2)+1+(nex*2)*(j*2+1)])/4.0)+0.5);	
            printf("%i",Mcoarse_int[i+nex*j]);	                             */
                             
        }   
	}
        
	level--;
    }
    
    if (3==ModelDim)    
    while (level > 0) 
    {    
        nex=Mesh[level-1].nex;
        ney=Mesh[level-1].ney;
        nez=Mesh[level-1].nez;
    
        Mfine   = Mesh[level].MatFieldE;
        Mcoarse = Mesh[level-1].MatFieldE;
        
       // Mfine_int = Mesh[level].MatFieldInt;
       // Mcoarse_int = Mesh[level-1].MatFieldInt;
    
    for (k=0; k<nez; k++)        
	for (i=0; i<nex; i++)
    {	
	    for (j=0; j<ney; j++)
	    {   
	       /* Mcoarse[i+nex*j]=1.0/(1.0/Mfine[(i*2)  +(nex*2)*(j*2)  ]+
		                          1.0/Mfine[(i*2)+1+(nex*2)*(j*2)  ]+
				                  1.0/Mfine[(i*2)  +(nex*2)*(j*2+1)]+
		                          1.0/Mfine[(i*2)+1+(nex*2)*(j*2+1)])*4*1.0; */

  Mcoarse[i+nex*j+k*nex*ney]=(Mfine[(i*2)  +(nex*2)*(j*2)  + (nex*ney*4)*(k*2)]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2)  + (nex*ney*4)*(k*2)]+
				              Mfine[(i*2)  +(nex*2)*(j*2+1)+ (nex*ney*4)*(k*2)]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2+1)+ (nex*ney*4)*(k*2)]+
                              Mfine[(i*2)  +(nex*2)*(j*2)  + (nex*ney*4)*(k*2+1)]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2)  + (nex*ney*4)*(k*2+1)]+
				              Mfine[(i*2)  +(nex*2)*(j*2+1)+ (nex*ney*4)*(k*2+1)]+
		                      Mfine[(i*2)+1+(nex*2)*(j*2+1)+ (nex*ney*4)*(k*2+1)])/8.0
                              *2.0; // IN 3D STIFFNESS INCREASES!!

	        /*Mcoarse_int[i+nex*j]=(int)floor(((Mfine_int[(i*2)  +(nex*2)*(j*2)  ]+
		                          Mfine_int[(i*2)+1+(nex*2)*(j*2)  ]+
				                  Mfine_int[(i*2)  +(nex*2)*(j*2+1)]+
		                          Mfine_int[(i*2)+1+(nex*2)*(j*2+1)])/4.0)+0.5);	
            printf("%i",Mcoarse_int[i+nex*j]);	                             */
                             
        }   
	}
        
	level--;
    }
    
    
    

    return(1);
}

/*---------------------------------------------------------
	TransForceOnCoarseMesh
----------------------------------------------------------*/

int TransForceOnCoarseMesh(int nnxfine, int nnyfine, double *FxFine, double *FyFine,
                 int nnxcoarse, int nnycoarse, double *FxCoarse, double *FyCoarse)
{
	int i,j,m,n, step;
	double step_db;
	int PosXFine, PosYFine;
	double x, y;
	double W;
	int ma,mb,na,nb;
    int nexcoarse, neycoarse;
    int p, q;
    
	
    nexcoarse=nnxcoarse-1;
    neycoarse=nnycoarse-1;
    step=(nnxfine-1)/(nnxcoarse-1);
    step_db=(double) step;

	for(j=0; j<nnycoarse; j++)
	{
		for(i=0; i<nnxcoarse; i++)
		{	
			p=i+j*nnxcoarse;
            FxCoarse[p]=0.0;
			FyCoarse[p]=0.0;
			
            /* An welcher Stelle im feinen Grid ?*/	
			PosXFine=step*i;
			PosYFine=step*j;
			
            ma=-(step-1);
			mb=(step-1);
			na=ma;
			nb=mb;
			
            if (0==i)         ma=0;
            if (nexcoarse==i) mb=0;
            if (0==j)         na=0;
            if (neycoarse==j) nb=0;
            
			
            for(n=na; n<=nb; n++)
			{
				for(m=ma; m<=mb; m++)
				{	
                    q= (PosXFine+m) + (PosYFine+n) * nnxfine;
                    x=(double) fabs(m);
					y=(double) fabs(n);
					W=1.00-(x+y-x*y/step_db)/step_db;
					
					FxCoarse[p]+=FxFine[q]*W ;
					FyCoarse[p]+=FyFine[q]*W ;
				}
			}
		}
	
	}


return(1);
}

/*---------------------------------------------------------
	TransForceOnCoarseMesh3d
----------------------------------------------------------*/

int TransForceOnCoarseMesh3d(int nnxfine, int nnyfine, int nnzfine, 
    double *FxFine, double *FyFine, double *FzFine,
    int nnxcoarse, int nnycoarse, int nnzcoarse,
    double *FxCoarse, double *FyCoarse, double *FzCoarse)
{
	int i,j,k,m,n, step;
	double step_db;
	int PosXFine, PosYFine, PosZFine;
	double x, y, z;
	double W;
	int ma,mb,pa,na,nb,pb,pp;
    int nexcoarse, neycoarse, nezcoarse, nxycoarse, nxyfine;
    int p, q;
    
	
    nexcoarse=nnxcoarse-1;
    neycoarse=nnycoarse-1;
    nezcoarse=nnzcoarse-1;
    nxycoarse=nnxcoarse*nnycoarse;
    nxyfine=nnxfine*nnyfine;
    
    step=(nnxfine-1)/(nnxcoarse-1);
    step_db=(double) step;
    
    for(k=0; k<nnzcoarse; k++)
	for(j=0; j<nnycoarse; j++)
	{
		for(i=0; i<nnxcoarse; i++)
		{	
			p=i+j*nnxcoarse+k*nxycoarse;
            FxCoarse[p]=0.0;
			FyCoarse[p]=0.0;
			FzCoarse[p]=0.0;
			
            /* An welcher Stelle im feinen Grid ?*/	
			PosXFine=step*i;
			PosYFine=step*j;
			PosZFine=step*k;
			
            ma=-(step-1);
			mb=(step-1);
			na=ma;
			nb=mb;
			pa=ma;
			pb=mb;
			
            if (0==i)         ma=0;
            if (nexcoarse==i) mb=0;
            if (0==j)         na=0;
            if (neycoarse==j) nb=0;
            if (0==k)         pa=0;
            if (nezcoarse==k) pb=0;
            
			for(pp=pa; pp<=pb; pp++)
            for(n=na; n<=nb; n++)
			{
				for(m=ma; m<=mb; m++)
				{	
                    q= (PosXFine+m) + (PosYFine+n) * nnxfine + (PosZFine+pp) * nxyfine;
                    x=(double) fabs(m);
					y=(double) fabs(n);
				    z=(double) fabs(pp);
					W=(1.0-(x/step_db))*(1.0-(y/step_db))*(1.0-(z/step_db));
					
					FxCoarse[p]+=FxFine[q]*W ;
					FyCoarse[p]+=FyFine[q]*W ;
					FzCoarse[p]+=FzFine[q]*W ;
				}
			}
		}
	
	}


return(1);
}

/*---------------------------------------------------------
	InterpolDispsOnFineMesh
----------------------------------------------------------*/

int InterpolDispsOnFineMesh(
     int nnxc, int nnyc, double *dUxC, double *dUyC,
     int nnxf, int nnyf, double *dUxF, double *dUyF, double *W)
{
	int i,j,step,k;
	double step_db;
	
    step=(nnxf-1)/(nnxc-1);
    step_db=(double) step;
	
    
    for(i=0; i<(step+1); i++)   W[i]=((double) i) / step_db;
    
    
    for(j=0; j<(nnyc-1); j++)
    {
        for(i=0; i<(nnxc); i++)
        {
             for(k=0; k<(step+1); k++)
             {    
                 dUxF[i*step+(j*step+k)*nnxf]=   dUxC[i+j*nnxc]    *(1.0-W[k])
                                     +  dUxC[i+(j+1)*nnxc]*W[k];
                                    
                 dUyF[i*step+(j*step+k)*nnxf]=   dUyC[i+j*nnxc]    *(1.0-W[k])
                                     +  dUyC[i+(j+1)*nnxc]*W[k];
             }
        }
    }
    
    for(j=0; j<(nnyf); j++)
    {
        for(i=0; i<(nnxf-1); i+=step)
        {
             for(k=1; k<(step); k++)
             {    
                 dUxF[i+k+j*nnxf]=   dUxF[i+j*nnxf]     *(1.0-W[k])
                                  +  dUxF[i+step+j*nnxf]*W[k];
                                    
                 dUyF[i+k+j*nnxf]=   dUyF[i+j*nnxf]     *(1.0-W[k])
                                  +  dUyF[i+step+j*nnxf]*W[k];
             }
        }
    }
	
	/*
	printf("\n");
    for(j=(nnyc-1); j>=0; j--)
    { 
        printf("\n");
        for(i=0; i<nnxc; i++)
        {
             printf("%g ",dUxC[i+j*nnxc]);
        }
    }
	
	printf("\n");
    for(j=(nnyf-1); j>=0; j--)
    { 
        printf("\n");
        for(i=0; i<nnxf; i++)
        {
             printf("%g ",dUxF[i+j*nnxf]);
        }
    }
	
	*/

	
return(1);
}

/*---------------------------------------------------------
	InterpolDispsOnFineMesh3d
----------------------------------------------------------*/

int InterpolDispsOnFineMesh3d(
     int nnxc, int nnyc, int nnzc, double *dUxC, double *dUyC, double *dUzC,
     int nnxf, int nnyf, int nnzf, double *dUxF, double *dUyF, double *dUzF, double *W)
{
	int i,j,k,si,sj,sk, step,nnxyf,nnxyc;
	double step_db;
	double a1,a2,b1,b2,c1,c2;
	
    step=(nnxf-1)/(nnxc-1);
    step_db=(double) step;
	
	nnxyf=nnxf*nnyf;
	nnxyc=nnxc*nnyc;
    
    for(i=0; i<(step+1); i++)   W[i]=((double) i) / step_db;
    
    for(i=0; i<(nnxc-1); i++)
    for(j=0; j<(nnyc-1); j++)
    for(k=0; k<(nnzc-1); k++)
    for(si=0; si<step+1; si++)
    for(sj=0; sj<step+1; sj++)
    for(sk=0; sk<step+1; sk++)    
    {
        a2=(double)si/step_db;
        b2=(double)sj/step_db;
        c2=(double)sk/step_db; 
        a1=1.0-a2;
        b1=1.0-b2;
        c1=1.0-c2;
        dUxF[(i*step+si) + (j*step+sj)*nnxf + (k*step+sk) *nnxyf]=
        dUxC[i   +     j*nnxc + k*nnxyc]    *a1*b1*c1+
        dUxC[i+1 +     j*nnxc + k*nnxyc]    *a2*b1*c1+
        dUxC[i+1 + (j+1)*nnxc + k*nnxyc]    *a2*b2*c1+
        dUxC[i   + (j+1)*nnxc + k*nnxyc]    *a1*b2*c1+
        dUxC[i   +     j*nnxc + (k+1)*nnxyc]*a1*b1*c2+
        dUxC[i+1 +     j*nnxc + (k+1)*nnxyc]*a2*b1*c2+
        dUxC[i+1 + (j+1)*nnxc + (k+1)*nnxyc]*a2*b2*c2+
        dUxC[i   + (j+1)*nnxc + (k+1)*nnxyc]*a1*b2*c2;
        
        dUyF[(i*step+si) + (j*step+sj)*nnxf + (k*step+sk) *nnxyf]=
        dUyC[i   +     j*nnxc + k*nnxyc]    *a1*b1*c1+
        dUyC[i+1 +     j*nnxc + k*nnxyc]    *a2*b1*c1+
        dUyC[i+1 + (j+1)*nnxc + k*nnxyc]    *a2*b2*c1+
        dUyC[i   + (j+1)*nnxc + k*nnxyc]    *a1*b2*c1+
        dUyC[i   +     j*nnxc + (k+1)*nnxyc]*a1*b1*c2+
        dUyC[i+1 +     j*nnxc + (k+1)*nnxyc]*a2*b1*c2+
        dUyC[i+1 + (j+1)*nnxc + (k+1)*nnxyc]*a2*b2*c2+
        dUyC[i   + (j+1)*nnxc + (k+1)*nnxyc]*a1*b2*c2;
        
        dUzF[(i*step+si) + (j*step+sj)*nnxf + (k*step+sk) *nnxyf]=
        dUzC[i   +     j*nnxc + k*nnxyc]    *a1*b1*c1+
        dUzC[i+1 +     j*nnxc + k*nnxyc]    *a2*b1*c1+
        dUzC[i+1 + (j+1)*nnxc + k*nnxyc]    *a2*b2*c1+
        dUzC[i   + (j+1)*nnxc + k*nnxyc]    *a1*b2*c1+
        dUzC[i   +     j*nnxc + (k+1)*nnxyc]*a1*b1*c2+
        dUzC[i+1 +     j*nnxc + (k+1)*nnxyc]*a2*b1*c2+
        dUzC[i+1 + (j+1)*nnxc + (k+1)*nnxyc]*a2*b2*c2+
        dUzC[i   + (j+1)*nnxc + (k+1)*nnxyc]*a1*b2*c2;
    }
    
	
return(1);
}



/*---------------------------------------------------------
	FinerScaleNodes
----------------------------------------------------------*/

int FinerScaleNodes(int nnxc, int nnyc, double *C,
                    int nnxf, int nnyf, double *F)
{
    int i,j,step;
    
    step=(nnxf-1)/(nnxc-1);
    
    for(j=0; j<nnyc; j++)
    for(i=0; i<nnxc; i++)
    F[i*step+j*step*nnxf]=C[i+j*nnxc];    

    return(1);
}

/*------------------------------------------------------------------------------
	GetLocalDisplacements_GS_real
--------------------------------------------------------------------------------*/ 

int GetLocalDisplacements_GS_real(
              int nnx, int nny, double *K, double *MatFieldE, 
              double *dUx, double *dUy, 
			  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, double lambda,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    
    (*err1) = 0;
    (*err2) = 0;
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; 
	         p--;               E2= MatFieldE[p]; 
	         p-=nex;            E3= MatFieldE[p]; 
	         p++;               E4= MatFieldE[p];
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K[0] *U9+K[1] *U10+K[2] *U11+K[3] *U12+K[4] *U17+K[5] *U18+K[6] *U15+K[7]*U16) 
             +E2*(K[16]*U7+K[17]*U8 +K[18]*U9 +K[19]*U10+K[20]*U15+K[21]*U16+K[22]*U13+K[23]*U14)
             +E3*(K[32]*U1+K[33]*U2 +K[34]*U3 +K[35]*U4 +K[36]*U9 +K[37]*U10+K[38]*U7 +K[39]*U8)
             +E4*(K[48]*U3+K[49]*U4 +K[50]*U5 +K[51]*U6 +K[52]*U11+K[53]*U12+K[54]*U9 +K[55]*U10);
         Fx-=BcValueX[p]; 
         (*err2)+=Fx*Fx; 
         MFx[p]=Fx;
         dUx[p]=(-Fx/(Esum*K[0]));
		 Ux[p]+=lambda*dUx[p]; 
		 } 
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K[8] *U9+K[9] *U10+K[10]*U11+K[11]*U12+K[12]*U17+K[13]*U18+K[14]*U15+K[15]*U16) 
		     +E2*(K[24]*U7+K[25]*U8 +K[26]*U9 +K[27]*U10+K[28]*U15+K[29]*U16+K[30]*U13+K[31]*U14)
		     +E3*(K[40]*U1+K[41]*U2 +K[42]*U3 +K[43]*U4 +K[44]*U9 +K[45]*U10+K[46]*U7 +K[47]*U8)
		     +E4*(K[56]*U3+K[57]*U4 +K[58]*U5 +K[59]*U6 +K[60]*U11+K[61]*U12+K[62]*U9 +K[63]*U10); 	 
		 Fy-=BcValueY[p]; 
         (*err2)+=Fy*Fy; 
         MFy[p]=Fy;
	     dUy[p]=(-Fy/(Esum*K[9]));
	     Uy[p]+=lambda*dUy[p];
	     }
       
      
         }
    }
   
    return(1);
}




/*------------------------------------------------------------------------------
	GetRestoringForce
--------------------------------------------------------------------------------*/ 

int GetRestoringForce(
              int nnx, int nny, double *K, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, 
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    
    (*err1) = 0;
    (*err2) = 0;
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; 
	         p--;               E2= MatFieldE[p]; 
	         p-=nex;            E3= MatFieldE[p]; 
	         p++;               E4= MatFieldE[p];
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K[0] *U9+K[1] *U10+K[2] *U11+K[3] *U12+K[4] *U17+K[5] *U18+K[6] *U15+K[7]*U16) 
             +E2*(K[16]*U7+K[17]*U8 +K[18]*U9 +K[19]*U10+K[20]*U15+K[21]*U16+K[22]*U13+K[23]*U14)
             +E3*(K[32]*U1+K[33]*U2 +K[34]*U3 +K[35]*U4 +K[36]*U9 +K[37]*U10+K[38]*U7 +K[39]*U8)
             +E4*(K[48]*U3+K[49]*U4 +K[50]*U5 +K[51]*U6 +K[52]*U11+K[53]*U12+K[54]*U9 +K[55]*U10);
         Fx-=BcValueX[p]; 
         (*err2)+=Fx*Fx; 
         MFx[p]=-Fx;
         } else MFx[p]=0.0;
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K[8] *U9+K[9] *U10+K[10]*U11+K[11]*U12+K[12]*U17+K[13]*U18+K[14]*U15+K[15]*U16) 
		     +E2*(K[24]*U7+K[25]*U8 +K[26]*U9 +K[27]*U10+K[28]*U15+K[29]*U16+K[30]*U13+K[31]*U14)
		     +E3*(K[40]*U1+K[41]*U2 +K[42]*U3 +K[43]*U4 +K[44]*U9 +K[45]*U10+K[46]*U7 +K[47]*U8)
		     +E4*(K[56]*U3+K[57]*U4 +K[58]*U5 +K[59]*U6 +K[60]*U11+K[61]*U12+K[62]*U9 +K[63]*U10); 	 
		 Fy-=BcValueY[p]; 
         (*err2)+=Fy*Fy; 
         MFy[p]=-Fy;
	     } else MFy[p]=0.0;
       
      
         }
    }
   
    return(1);
}

/*------------------------------------------------------------------------------
	GetRestoringForceCorrect
--------------------------------------------------------------------------------*/ 

int GetRestoringForceCorrect(
              int nnx, int nny, double *K, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, 
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    
    (*err1) = 0;
    (*err2) = 0;
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; 
	         p--;               E2= MatFieldE[p]; 
	         p-=nex;            E3= MatFieldE[p]; 
	         p++;               E4= MatFieldE[p];
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K[0] *U9+K[1] *U10+K[2] *U11+K[3] *U12+K[4] *U17+K[5] *U18+K[6] *U15+K[7]*U16) 
             +E2*(K[16]*U7+K[17]*U8 +K[18]*U9 +K[19]*U10+K[20]*U15+K[21]*U16+K[22]*U13+K[23]*U14)
             +E3*(K[32]*U1+K[33]*U2 +K[34]*U3 +K[35]*U4 +K[36]*U9 +K[37]*U10+K[38]*U7 +K[39]*U8)
             +E4*(K[48]*U3+K[49]*U4 +K[50]*U5 +K[51]*U6 +K[52]*U11+K[53]*U12+K[54]*U9 +K[55]*U10);
         Fx-=BcValueX[p]; 
         (*err2)+=Fx*Fx; 
         MFx[p]=-Fx;
         } else MFx[p]=0.0;
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K[8] *U9+K[9] *U10+K[10]*U11+K[11]*U12+K[12]*U17+K[13]*U18+K[14]*U15+K[15]*U16) 
		     +E2*(K[24]*U7+K[25]*U8 +K[26]*U9 +K[27]*U10+K[28]*U15+K[29]*U16+K[30]*U13+K[31]*U14)
		     +E3*(K[40]*U1+K[41]*U2 +K[42]*U3 +K[43]*U4 +K[44]*U9 +K[45]*U10+K[46]*U7 +K[47]*U8)
		     +E4*(K[56]*U3+K[57]*U4 +K[58]*U5 +K[59]*U6 +K[60]*U11+K[61]*U12+K[62]*U9 +K[63]*U10); 	 
		 Fy-=BcValueY[p]; 
         (*err2)+=Fy*Fy; 
         MFy[p]=-Fy;
	     } else MFy[p]=0.0;
       
      
         }
    }
   
    return(1);
}

/*------------------------------------------------------------------------------
	GetRestoringForce2
--------------------------------------------------------------------------------*/ 

int GetRestoringForce2(
              int nnx, int nny, double *MatFieldE, 
              int *MatFieldInt, MaterialBlock *Mat,
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    double *K1, *K2, *K3, *K4, *K;
    
    K1=Mat[0].K; K2=K1; K3=K1; K4=K1;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         Fx-=BcValueX[p]; 
         MFx[p]=-Fx;
         } else MFx[p]=0.0;
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 Fy-=BcValueY[p]; 
         MFy[p]=-Fy;
	     } else MFy[p]=0.0;
       
      
         }
    }
   
    return(1);
}

/*------------------------------------------------------------------------------
	GetRestoringForceIncrement
--------------------------------------------------------------------------------*/ 

// without considering external forces !!!

int GetRestoringForceIncrement(
              int nnx, int nny, double *K, double *MatFieldE, 
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
			  double *err1,double *err2, 
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    
    (*err1) = 0;
    (*err2) = 0;
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; 
	         p--;               E2= MatFieldE[p]; 
	         p-=nex;            E3= MatFieldE[p]; 
	         p++;               E4= MatFieldE[p];
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K[0] *U9+K[1] *U10+K[2] *U11+K[3] *U12+K[4] *U17+K[5] *U18+K[6] *U15+K[7]*U16) 
             +E2*(K[16]*U7+K[17]*U8 +K[18]*U9 +K[19]*U10+K[20]*U15+K[21]*U16+K[22]*U13+K[23]*U14)
             +E3*(K[32]*U1+K[33]*U2 +K[34]*U3 +K[35]*U4 +K[36]*U9 +K[37]*U10+K[38]*U7 +K[39]*U8)
             +E4*(K[48]*U3+K[49]*U4 +K[50]*U5 +K[51]*U6 +K[52]*U11+K[53]*U12+K[54]*U9 +K[55]*U10);
         //Fx-=BcValueX[p]; 
         (*err2)+=Fx*Fx; 
         MFx[p]=Fx; //CG
         } else MFx[p]=0.0;
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K[8] *U9+K[9] *U10+K[10]*U11+K[11]*U12+K[12]*U17+K[13]*U18+K[14]*U15+K[15]*U16) 
		     +E2*(K[24]*U7+K[25]*U8 +K[26]*U9 +K[27]*U10+K[28]*U15+K[29]*U16+K[30]*U13+K[31]*U14)
		     +E3*(K[40]*U1+K[41]*U2 +K[42]*U3 +K[43]*U4 +K[44]*U9 +K[45]*U10+K[46]*U7 +K[47]*U8)
		     +E4*(K[56]*U3+K[57]*U4 +K[58]*U5 +K[59]*U6 +K[60]*U11+K[61]*U12+K[62]*U9 +K[63]*U10); 	 
		 //Fy-=BcValueY[p]; 
         (*err2)+=Fy*Fy; 
         MFy[p]=Fy; //CG
	     } else MFy[p]=0.0;
       
      
         }
    }
   
    return(1);
}


/*------------------------------------------------------------------------------
	GetRestoringForceIncrement2
--------------------------------------------------------------------------------*/ 

int GetRestoringForceIncrement2(
              int nnx, int nny, double *MatFieldE, 
              int *MatFieldInt, MaterialBlock *Mat,
        	  double *Ux,  double *Uy,
			  double *MFx, double *MFy,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    double *K1, *K2, *K3, *K4, *K;
    
    K1=Mat[0].K; K2=K1; K3=K1; K4=K1;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         //Fx-=BcValueX[p]; 
         MFx[p]=Fx;
         } else MFx[p]=0.0;
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 //Fy-=BcValueY[p]; 
         MFy[p]=Fy;
	     } else MFy[p]=0.0;
       
      
         }
    }
   
    return(1);
}
/*------------------------------------------------------------------------------
	GetFactor
-------------------------------------------------------------------------------*/
int GetFactor(int nnx, int nny, 
			  double *Fx,  double *Fy,
			  double *dFx, double *dFy, 
              int *BcTypeX, int *BcTypeY,
              double *lambda)
{
    int i,j;
    int istart, jstart, iend, jend;
    int p;
    int nex, ney;
    double Nenner,Zaehler;
    
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
    
    Nenner=0.0;
    Zaehler=0.0;
    
    
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
            p=i+j*nnx;
	  
       if (0==BcTypeX[p]) 		
	    {
            Zaehler  += (dFx[p]*Fx[p]);
	        Nenner   += (dFx[p]*dFx[p]); 
        }
        
        if (0==BcTypeY[p]) 		
	    {
            Zaehler  += (dFy[p]*Fy[p]);
	        Nenner   += (dFy[p]*dFy[p]); 
        }
        
        }
    }
    
    (*lambda)=-(Zaehler/Nenner);
    
    return(1);
}

/*------------------------------------------------------------------------------
	AddIncrement
-------------------------------------------------------------------------------*/
int AddIncrement(int nnx, int nny, 
              double *Ux,  double *Uy,
			  double *dUx, double *dUy,
			  int *BcTypeX, int *BcTypeY,
			  double lambda)
{
    int i,j;
    int istart, jstart, iend, jend;
    int p;
    int nex, ney;   
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
 
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {    
	    p=i+j*nnx;
	    
	     if (0==BcTypeX[p])   Ux[p]+=dUx[p]*lambda;
	     if (0==BcTypeY[p])   Uy[p]+=dUy[p]*lambda;
        }
    }
 
    return(1);
}

/*------------------------------------------------------------------------------
	AddIncrement3d
-------------------------------------------------------------------------------*/
int AddIncrement3d(int nnx, int nny, int nnz,
              double *Ux,  double *Uy, double *Uz,
			  double *dUx, double *dUy, double *dUz,
			  int *BcTypeX, int *BcTypeY, int *BcTypeZ,
			  double lambda)
{
    int p;

    for (p = 0; p < nnx*nny*nnz; p++)   
    {    
	     if (0==BcTypeX[p])   Ux[p]+=dUx[p]*lambda;
	     if (0==BcTypeY[p])   Uy[p]+=dUy[p]*lambda;
    	 if (0==BcTypeZ[p])   Uz[p]+=dUz[p]*lambda;
    }
    return(1);
}




