/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

int MesoPixMainLoop()
{    
    MesoPixTmp TmpMemory;
    AggBlock AggData;
    
    int    *px, *p, *py, *pA, *map, *pmap, *obj_matrix;
    int    *length, *incl_matrix, *fillflag, *MatFieldInt;
    double *MatFieldE;
    
    int    i,j,k,l;
    int    num_incl,nex,ney, num_levels;
    int    agg_area_int, obj_area_int, aim_area_int;
    double a, b, phi, la, lb, lmin, lx, ly;

    double max_size_db;
    int    max_size_int;
    int    alloc_size;
    double rmax,rmin; // maximum radii;
   
    
    double E_Matrix, E_Inclusion;
    double Mu_Matrix, Mu_Inclusion;
    int valid;
    int X1, X2;
    int  tmp, tmpx, tmpy;
    MGBlock *Meshes;
    int trials;
    int trial_limit;
    clock_t start, ende;
    float mytime;
    double sumtrials=0.0;
    int MaterialNumBegin;
    int MaterialNumEnd;
    int AddFlag;
    int mat;
    MaterialBlock *LocalMat;
    int LocalMatNum;
    int MatId;
       
     
    AllWinStruct *AllWinData;
    MesoWinStruct *MesoWinData;
    SolveBlock *SolBlock;
    
    PutGetSolveBlock(1,&SolBlock); 
    PutGetAllWinStruct(1, &AllWinData);
    MesoWinData=AllWinData->MesoWinData;
    
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    la=SolBlock->Meshes[num_levels-1].la;
    lb=SolBlock->Meshes[num_levels-1].lb;
    nex=(SolBlock->Meshes[num_levels-1].nnx)-1;
    ney=(SolBlock->Meshes[num_levels-1].nny)-1; 
    trial_limit=10*nex*ney;
    
    rmax=(MesoWinData->MaxSize)/2;
    rmin=(MesoWinData->MinSize)/2;
    
    AggData.lw_ratio=MesoWinData->Compactness;
    AggData.Smoothness=MesoWinData->Smoothness;
    AggData.Roughness=MesoWinData->Roughness;
    AggData.Deformed=MesoWinData->Deformed;
    AggData.RoughFlag=MesoWinData->RoughFlag;
    AggData.DeformedFlag=MesoWinData->DeformedFlag;
    
    E_Inclusion=MesoWinData->E_Inclusion;
    E_Matrix=MesoWinData->E_Matrix;
    Mu_Inclusion=MesoWinData->Mu_Inclusion;
    Mu_Matrix=MesoWinData->Mu_Matrix;
    
    MatFieldInt=SolBlock->Meshes[num_levels-1].MatFieldInt;
    MatFieldE=SolBlock->Meshes[num_levels-1].MatFieldE;
    
     
   //printf("\nAnfang\n");getchar();
    
    obj_area_int=nex*ney;
    agg_area_int=0;
    aim_area_int=(int)((MesoWinData->VolFraction)*obj_area_int);
    
    max_size_db=sqrt(2*rmax*rmax)*2;  
    if (la>lb) lmin=lb; else lmin=la;
    max_size_int=(int) (max_size_db/lmin+2); 
    if(max_size_int>nex || max_size_int>ney)
    {
        printf("Aggregate size is too big compared too object dimensions");
        return 1;
    }
    InitVector_int(nex*ney,    &px);       TmpMemory.px=px; 
    InitVector_int(nex*ney,    &py);       TmpMemory.py=py;
    InitVector_int(nex*ney,    &pA);       TmpMemory.pA=pA;
    InitVector_int(nex*ney,    &p);        TmpMemory.p=p;
    InitVector_int(nex*ney,    &length);   TmpMemory.length=length;
    InitVector_int(nex*ney,    &fillflag); TmpMemory.fillflag=fillflag;
     
    InitVector_int(nex*ney,    &map);       TmpMemory.map=map;
    InitVector_int(nex*ney,  &obj_matrix);TmpMemory.obj_matrix=obj_matrix;
    
    
    AddFlag=(int) MesoWinData->AddFlag;
    if (1==AddFlag)
    {
        MaterialNumBegin= MesoWinData->MaterialNumBegin;
        MaterialNumEnd  = MesoWinData->MaterialNumEnd;
        if (0==SolBlock->NumMaterials)
        {
            LocalMatNum=3;
            MatId=3;
            InitMaterialBlock(LocalMatNum, &LocalMat);
        }    
        else
        {    
            LocalMatNum=((SolBlock->NumMaterials)+2);
            MatId=(SolBlock->NumMaterials);
            InitMaterialBlock(LocalMatNum, &LocalMat);
            //CopyMaterialBlock((SolBlock->NumMaterials),  LocalMat, SolBlock->Material);
        }
        memcpy(obj_matrix, MatFieldInt, nex*ney*sizeof(int)); 
    }
    else
    {
        MaterialNumBegin=0;
        MaterialNumEnd=0;
        LocalMatNum=3;
        MatId=1;
        InitMaterialBlock(LocalMatNum, &LocalMat);
    }  
    
   
   
   //obj_matrix=& MatFieldInt[0];
   // MatFieldInt=obj_matrix;
    
      pmap=&(map[(int)(nex*ney*0.5)]);
    p[0]=0;
    num_incl=0;
    printf("GENERATING!\n"); 
   
    start = clock();
    while (agg_area_int<aim_area_int)
    {     
        GenerateOneInclusion(rmax, rmin, &AggData);
        tmp=p[num_incl];
        
        BitmapInclEdge(la, lb, &AggData, &(length[num_incl]), 
                       &(px[tmp]), &(py[tmp]), &(pA[num_incl]), pmap,
                       &(fillflag[num_incl]));
        
        p[num_incl+1]=length[num_incl]+p[num_incl]+1;
       
        agg_area_int+=pA[num_incl];
      
        num_incl+=1;
    } 
    
     printf("Inclusions %i:\n", num_incl);

     printf("SORTING!\n");   
     printf("AREA %g\n", ( ((double) agg_area_int)/ ((double)obj_area_int) ));   
    InitVector_int(num_incl*4,    &incl_matrix);
    TmpMemory.incl_matrix=incl_matrix;
    
    for(i=0; i<num_incl; i++)
    {
        incl_matrix[i]  =p[i];
        incl_matrix[i+num_incl]=length[i];
        incl_matrix[i+num_incl*2]=fillflag[i];
        incl_matrix[i+num_incl*3]=pA[i];
    }
   
    shellsort3(incl_matrix, num_incl, 4, 4, 5);
    p=&(incl_matrix[0]);
    length=&(incl_matrix[num_incl]);
    fillflag=&(incl_matrix[2*num_incl]);
     
    for(i=0; i<nex; i++)
    {          
        obj_matrix[i+0*nex]=-1;
        obj_matrix[i+(ney-1)*nex]=-1;
    } 
    
    for(j=0; j<ney; j++)
    {          
        obj_matrix[0+j*nex]=-1;
        obj_matrix[(nex-1)+j*nex]=-1;
    } 
    trials=0;
    for(j=0; j<(num_incl); j++)
    {   
        
        trials=0;
        tmp=p[j];
        
        valid=0;
        do
        {
            valid=1; i=0;
            do
            {   
                X1=(int)((rand()/(RAND_MAX+1.0))*((double)(nex-1.0)));
                X2=(int)((rand()/(RAND_MAX+1.0))*((double)(ney-1.0)));
                tmpx=(px[tmp+i]+X1);
                tmpy=(py[tmp+i]+X2);
            } while ((tmpx<1) || (tmpy<1) || (tmpx>(nex-2)) 
                              || (tmpy>(ney-2)));
            
            while((i<length[j]) && (1==valid))
            {   
                tmpx=(px[tmp+i]+X1);
                tmpy=(py[tmp+i]+X2);
                //printf("%i\n",tmpx+tmpy*nex);
                
                if (MaterialNumBegin>obj_matrix[tmpx+tmpy*nex] 
                 || MaterialNumEnd<obj_matrix[tmpx+tmpy*nex] ) {valid=0;}
                
                /* ein pixel abstand? */
                
                else
                {   mat=obj_matrix[(tmpx+1)+(tmpy+0)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx+1)+(tmpy+1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx+1)+(tmpy-1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx-1)+(tmpy+0)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx-1)+(tmpy+1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx-1)+(tmpy-1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx)+(tmpy+1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                    mat=obj_matrix[(tmpx)+(tmpy-1)*nex];
                    if (MaterialNumBegin>mat || MaterialNumEnd<mat) {valid=0;}
                }
                i++;
            }
            
            trials++;
            
            if (trials>trial_limit)
            {
                printf("\nNo Solution - Number of trials:%i\n", trials);
                FreeMesoPixTmp(TmpMemory);
                return 1;
            }
        
        
        } while(0==valid);
        
        if((j/100.0)==floor(j/100.0)) printf("%i: %i \n", j, trials);
       
        //X1=100; X2=100;
        for(i=0; i<length[j]; i++)
        {   
            tmpx=(px[tmp+i]+X1);
            tmpy=(py[tmp+i]+X2);
            obj_matrix[tmpx+tmpy*nex]=MatId;
        }
        if(1==fillflag[j])  
        {RecFill2WithNumber(X1, X2, nex, obj_matrix, MatId, (MatId+1));}
        sumtrials=sumtrials+ ((double) trials);
    }
   //  printf("\nMItte2\n");getchar();
    printf("Sum of trials %g \n", sumtrials);
    
    for(i=0; i<nex; i++)
    {          
        obj_matrix[i+0*nex]=0;
        obj_matrix[i+(ney-1)*nex]=0;
    } 
    
    for(j=0; j<ney; j++)
    {          
        obj_matrix[0+j*nex]=0;
        obj_matrix[(nex-1)+j*nex]=0;
    } 
    
    memcpy(MatFieldInt, obj_matrix, nex*ney*sizeof(int));   
    
    if(0!=SolBlock->NumMaterials)
    {
       CopyMaterialBlock((SolBlock->NumMaterials), LocalMat, SolBlock->Material);
       FreeMaterialBlock((SolBlock->NumMaterials), SolBlock->Material);
    }
    
    SolBlock->Material=LocalMat;
    SolBlock->NumMaterials=MatId+2;
    
    if (0==AddFlag)
    {
        SolBlock->Material[0].E=E_Matrix;
        SolBlock->Material[0].Mu=Mu_Matrix;
    }
    
    SolBlock->Material[MatId].E=E_Inclusion;
    SolBlock->Material[MatId].Mu=Mu_Inclusion;
    SolBlock->Material[MatId+1].E=E_Inclusion;
    SolBlock->Material[MatId+1].Mu=Mu_Inclusion;
    
            
                            
    for(i=0; i<nex; i++)
    for(j=0; j<ney; j++)
    MatFieldE[i+j*nex]=SolBlock->Material[MatFieldInt[i+j*nex]].E;

   /*   
    for(i=0; i<nex; i++)
    {
        for(j=0; j<ney; j++)
        {
            if ((i<((double)nex)/2.0) && (j<((double)nex)/2.0))
            {
                MatFieldE[i+j*nex]=E_Inclusion;
                //if (1==obj_matrix[i+j*nex]) //MatFieldE[i+j*nex]=(E_Inclusion+E_Matrix)*0.5;
                //MatFieldE[i+j*nex]=0.0;
                //MatFieldE[i+j*nex]=E_Inclusion*0.5;
            }
            else
            MatFieldE[i+j*nex]=E_Matrix;
        }       
        
    } 
    */
    
    
    ende = clock();
	mytime = (float)(ende - start) / (float)CLOCKS_PER_SEC;
    printf("\n Generation Time %g\n", mytime);
    
    TransMatFieldToCoarse(Meshes, num_levels);
    
    //PrintMatrix_int(0, (nex-1), 0, (ney-1), MatFieldInt);
    FreeMesoPixTmp(TmpMemory);
   // printf("\nEnde\n");getchar();
    return 1;
}

/*------------------------------------------------------------------------------
     GenerateOneInclusion
------------------------------------------------------------------------------*/
void GenerateOneInclusion(double rmax, double rmin, AggBlock *AggData)
{
    double X1, X2;
    double x1, x2;
    double incl_size, cutted_size;
    double a,b,phi,rat;
    double pot1,pot2, smooth;
    
    rat=AggData->lw_ratio;
    smooth=AggData->Smoothness;
    
    x1=pow(rmax,(-1.5));
    x2=pow(rmin,(-1.5));
    
    X1=rand()/(RAND_MAX+1.0);
    X2=rand()/(RAND_MAX+1.0);

    incl_size=pow(x1-rand()/(RAND_MAX+1.0)*(x1-x2),(-2.0/3.0));
    cutted_size=pow(pow(incl_size,2)-pow(incl_size*(rand()/(RAND_MAX+1.0)),2),.5);
    a  =cutted_size;
    if (a<rmin) a=rmin;
    //(*a)  = pow(fabs(x1-X1*(x1-x2)),-(2.0/3.0))*sqrt(1-X2*X2);
    //(*a)  = rmin+X1*(rmax-rmin);
    
    X2=rand()/(RAND_MAX+1.0);
    //b  = a * (-X2*rat*0.5+1.0-rat*0.3);
    b  = a * (-X2*rat*0.1+1.0-rat*0.8);
    
    X2=rand()/(RAND_MAX+1.0);
    phi= X2 * 180.0;
    X2=rand()/(RAND_MAX+1.0);
    if (X2>0.5) {pot1=2.0-X2*smooth*0.6-smooth*0.4;} 
           else {pot1=2.0+X2*smooth*8.3+smooth*3;}
    if (X2>0.5) {pot2=2.0-X2*smooth*0.6-smooth*0.4;} 
           else {pot2=2.0+X2*smooth*8.3+smooth*3;}
    
    //a=20.0;
    //b=5.0;
    
    AggData->a=a;
    AggData->b=b;
    AggData->phi=phi;
    AggData->pot1=pot1;
    AggData->pot2=pot2;
    AggData->RoughFact=(rand()/(RAND_MAX+1.0))
             *(AggData->Roughness)*0.1+(AggData->Roughness)*0.05;
    AggData->DeformedFact=(rand()/(RAND_MAX+1.0))
             *(AggData->Deformed)*0.1+(AggData->Deformed)*0.05;
    
    AggData->e1=(rand()/(RAND_MAX+1.0))*1.570796327;
    AggData->e2=(rand()/(RAND_MAX+1.0))*1.570796327;
    
    AggData->num1=((int)((rand()/(RAND_MAX+1.0))*4.0+5.1))*2;
    AggData->num2=((int)((rand()/(RAND_MAX+1.0))*2.0+1.6))*2;
    return;
}


/*------------------------------------------------------------------------------
     InclFunc
------------------------------------------------------------------------------*/
void InclFunc(AggBlock *A,
              double la, double lb, int coorx, int coory, int *valid)
{
    double f;
    double x,y;
    double ang;
                                                                                                                                            
    x=la*coorx;
    y=lb*coory;
    
    f=    pow(( fabs ( x*(A->cosphi) + y*(A->sinphi))/(A->a) ) ,(A->pot1)) 
        + pow(( fabs (-x*(A->sinphi) + y*(A->cosphi))/(A->b) ) ,(A->pot1))
        -1.000;
    
    if (0!=A->RoughFlag || 0!=A->DeformedFlag)
    { 
        if (0!=x) {ang=atan(y/x);} else {ang=1.570796327;} 
        if (0!=A->RoughFlag)    
             f+= (A->RoughFact)*     sin((ang+(A->e1))*((double)(A->num1)));
        if (0!=A->DeformedFlag)    
             f+= (A->DeformedFact)*  sin((ang+(A->e2))*((double)(A->num2)));
    }

    if (f<0) (*valid)=1; else (*valid)=0;

    return;
}

/*------------------------------------------------------------------------------
     BitmapInclEdge
------------------------------------------------------------------------------*/
void BitmapInclEdge(double la, double lb, AggBlock *AggData, int *p,
                    int *px, int *py, int *pA, int *pmap, int *fillflag)
{
    int count;
    double cosphi, sinphi;
    double pi=3.14159265359;
    int sx,sy, start_x, start_y;
    int valid;
    int direct;
    int A;
    int xmin, xmax, ymin, ymax;
    int i,j;
    int xlength;
    int nofill;
    double phi;
    
    phi=(AggData->phi);
    
    
    A=0;
    sx=0; 
    sy=0;
    cosphi=cos(phi/180.0*pi);
	sinphi=sin(phi/180.0*pi);
	AggData->sinphi=sinphi;
	AggData->cosphi=cosphi;
	
    nofill=0;
    valid=1;
    count=0;
    
    //CHECK IF INCLUSION CAN BE FILLED
    InclFunc(AggData, la, lb, 1, -1, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, 1, 0, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, 1, 1, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, -1, -1, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, -1, 0, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, -1, 1, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, 0, -1, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, 0, 0, &valid); if (0==valid) nofill=1;
    InclFunc(AggData, la, lb, 0, 1, &valid); if (0==valid) nofill=1;
    
    if (1==nofill)
    {
        px[0]=0;
        py[0]=0;
        count++;
    }
   
    if (1==nofill)
    {
        for(i=-1; i<=1; i++)
        {
            for(j=-1; j<=1; j++)
            {
                InclFunc(AggData, la, lb, i, j, &valid); 
                if (0==i && 0==j) valid=1;
                if (1==valid) 
                {
                    px[count]=i;
                    py[count]=j;
                    count++;
                }
            }
        }
        *fillflag=0;
        *p=count;
        *pA=count;
        return; 
     }
   
    valid=1;
    while(1==valid)
    {
         sy-=1;
         InclFunc(AggData, la, lb, sx, sy, &valid);
    }
   
    sy+=1;
    start_x=sx;
    start_y=sy;
    px[count]=sx;
    py[count]=sy;
    direct=1;
    
    while((px[0]!=px[count] || py[0]!=py[count] || 0 ==count)) 
    {   
        if (5==direct) direct=1;
        if (0==direct) direct=4;
    
        
        
        if (1==direct) sx=px[count]+1;
        if (2==direct) sy=py[count]+1;
        if (3==direct) sx=px[count]-1;
        if (4==direct) sy=py[count]-1;
    
        InclFunc(AggData, la, lb, sx, sy, &valid);
        
        if (1==valid) 
        {
            count+=1;
            px[count]=sx;
            py[count]=sy;
            direct-=1;
        }
        else
        {
            direct+=1;
            sx=px[count];
            sy=py[count];
        }
    }
    
    *p=count;
    
   
    FindMinMax_int(count, px, &xmin, &xmax);
    FindMinMax_int(count, py, &ymin, &ymax);
    //printf("x:%i-%i  y:%i-%i\n",xmin,xmax,ymin,ymax);
    
    
    for(i = 0; i<count; i++)
    pmap[px[i]+py[i]*(xmax-xmin+1)]=1;
      
    
    //PrintMatrix_int(xmin, xmax, ymin, ymax, pmap);
    
    //CHECK IF INCLUSION CAN BE FILLED
    
    
    
  
    
    xlength=xmax-xmin+1; 
    //getchar();
    RecFill2(0,0,xlength, pmap);
    
    /*RecFill(0, 0, xlength, pmap);
    RecFill(0, 0, xlength, pmap);
    RecFill(0, 0, xlength, pmap);*/
                 
    //PrintMatrix_int(xmin, xmax, ymin, ymax, pmap);   
    CountPixelArea(xmin, xmax, ymin, ymax, pmap, &A);
    *pA=A;         
    *fillflag=1;
     
    return;
}

/*--------------------------------------------------------------------------------
	FindMinMax_int
--------------------------------------------------------------------------------*/
int FindMinMax_int(int length, int *vector, int *minimum, int *maximum)
{
   int i; 
   int min,max;
   
   min=vector[0];
   max=vector[0];
   
   for(i = 0;i < length;i++)
   {
	if (vector[i]<min) min=vector[i];
	if (vector[i]>max) max=vector[i];
   }
   
   *minimum=min;
   *maximum=max;

return(1);
}

/*--------------------------------------------------------------------------------
	PrintMatrix_int
--------------------------------------------------------------------------------*/
int PrintMatrix_int(int xmin, int xmax, int ymin, int ymax, int *vector)
{
   int i,j;
   
   for(j = ymax; j>=ymin; j--)
   {
       printf("\n");
       for(i = xmin; i<=xmax; i++)
	   {
           printf(" %i",vector[i+j*(xmax-xmin+1)]);
       }
   }

return(1);
}

/*--------------------------------------------------------------------------------
	RecFill - Problems if more than approx. 32000 recursive calls !!!
--------------------------------------------------------------------------------*/
void RecFill(int x, int y, int L, int *V)
{  
   if (1==V[x+y*L]) return;
   V[x+y*L]=2;
   
   if (0==V[(x+1)+y    *L]){RecFill(x+1, y  , L, V);}
   if (0==V[(x+1)+(y+1)*L]){RecFill(x+1, y+1, L, V);}
   if (0==V[x    +(y+1)*L]){RecFill(x  , y+1, L, V);}
   if (0==V[(x-1)+(y+1)*L]){RecFill(x-1, y+1, L, V);}   
   if (0==V[(x-1)+y    *L]){RecFill(x-1, y  , L, V);}
   if (0==V[(x-1)+(y-1)*L]){RecFill(x-1, y-1, L, V);} 
   if (0==V[x+(y-1)    *L]){RecFill(x  , y-1, L, V);}
   if (0==V[(x+1)+(y-1)*L]){RecFill(x+1, y-1, L, V);}   
 
   return;
}

/*--------------------------------------------------------------------------------
	RecFill2
--------------------------------------------------------------------------------*/
void RecFill2(int X1, int X2, int L, int *V)
{  
   int i,j,hits,p,k;
   int minX,minY,maxX,maxY;
   hits=0;
   minX=X1-1; maxX=X1+1;
   minY=X2-1; maxY=X2+1;
   V[X1+X2*L]=2;
   k=0;
   
   do
   {
   hits=0;
   for(j=minY; j<=maxY; j++)
   {
           for(i=minX; i<=maxX; i++)
           {
               p=i+j*L;
               if(2!=V[p] && 1!=V[p] && 
                 (2==V[p+1] || 2==V[p-1] || 2==V[p+L] || 2==V[p-L] ||
                  2==V[p+1+L] || 2==V[p+1-L] || 2==V[p-1+L] || 2==V[p-1-L]))
                  {
                      if(i==maxX) maxX++;
                      if(i==minX) minX--;
                      if(j==maxY) maxY++;
                      if(j==minY) minY--;
                      hits++;
                      V[p]=2; 
                      k=0;
                  } 
           }
   }
   if(0==hits && 0==k) k++; else k=0; 
   //printf("%i\n",hits);
   }
   while (hits>0 || k>0);
   
   return;
}

/*--------------------------------------------------------------------------------
	RecFill2WithNumber
--------------------------------------------------------------------------------*/
void RecFill2WithNumber(int X1, int X2, int L, int *V, int n1,  int n2)
{  
   int i,j,hits,p,k;
   int minX,minY,maxX,maxY;
   
   hits=0;
   minX=X1-1; maxX=X1+1;
   minY=X2-1; maxY=X2+1;
   V[X1+X2*L]=n2;
   k=0;
   
   do
   {
   hits=0;
   for(j=minY; j<=maxY; j++)
   {
           for(i=minX; i<=maxX; i++)
           {
               p=i+j*L;
               if(n2!=V[p] && n1!=V[p] && 
                 (n2==V[p+1] || n2==V[p-1] || n2==V[p+L] || n2==V[p-L] ||
                  n2==V[p+1+L] || n2==V[p+1-L] || n2==V[p-1+L] || n2==V[p-1-L]))
                  {
                      if(i==maxX) maxX++;
                      if(i==minX) minX--;
                      if(j==maxY) maxY++;
                      if(j==minY) minY--;
                      hits++;
                      V[p]=n2; 
                      k=0;
                  } 
           }
   }
   if(0==hits && 0==k) k++; else k=0; 
   //printf("%i\n",hits);
   }
   while (hits>0 || k>0);
   
   return;
}

/*--------------------------------------------------------------------------------
	CountPixelArea
--------------------------------------------------------------------------------*/
void CountPixelArea(int xmin, int xmax, int ymin, int ymax, int *pmap, int *A)
{
   int i,j,Area;
   Area=0;
   
   for(j = ymax; j>=ymin; j--)
   {
       for(i = xmin; i<=xmax; i++)
	   {
          if(0!=pmap[i+j*(xmax-xmin+1)]) 
          {
              Area+=1;
              pmap[i+j*(xmax-xmin+1)]=0;
          }
       }
   }
   
   *A=Area;
   return;
}

/*--------------------------------------------------------------------------------
	FreeMesoPixTmp
 --------------------------------------------------------------------------------*/
void FreeMesoPixTmp(MesoPixTmp Tmp)
{
   free(Tmp.px);
   free(Tmp.py);
   free(Tmp.pA);
   free(Tmp.p);
   free(Tmp.length);
   free(Tmp.fillflag);
   free(Tmp.map);
   free(Tmp.obj_matrix);
   free(Tmp.incl_matrix);
   
   return;
}


/*---------------------------------------------------------
	shellsort
----------------------------------------------------------*/


void shellsort3(int *array, int num_row, int num_col, int sort_col, int distanz)
{
 int i,j,n,h,k;
 int temp[num_col];
 int num_incl;
 
 num_incl=num_row;
 
 n=num_row-1;
 for(; n>0; n/=distanz)
 {
   for(i=n; i<num_row; i++)
     {
		   h=n;
			 for(k=0;k<num_col;k++)
			 {
         temp[k]=array[i+k*num_incl];
			 }
       for(j=i; j>=n && array[(j-n)+(sort_col-1)*num_incl]<temp[sort_col-1]; j-=n)
			 {
			   for(k=0;k<num_col;k++)
				 {
           array[j+k*num_incl]=array[(j-n)+k*num_incl];
				 }
			 }
  	   for(k=0;k<num_col;k++)
			 {
         array[j+k*num_incl]=temp[k];
			 }
     }
	}
	if (h>1)
	{
	  n=1;
    for(i=n; i<num_row; i++)
     {
			 for(k=0;k<num_col;k++)
			 {
         temp[k]=array[i+k*num_incl];
			 }
       for(j=i; j>=n && array[(j-n)+(sort_col-1)*num_incl]<temp[sort_col-1]; j-=n)
			 {
			   for(k=0;k<num_col;k++)
				 {
           array[j+k*num_incl]=array[(j-n)+k*num_incl];
				 }
			 }
  	   for(k=0;k<num_col;k++)
			 {
         array[j+k*num_incl]=temp[k];
			 }
     }
	}
}

/*--------------------------------------------------------------------------------
	SmoothMatField
--------------------------------------------------------------------------------*/
void SmoothMatField(int nex, int ney, double *MA, double *MB)
{
   int i,j;
   
   for(i=0; i<nex; i++)
    {
        for(j=0; j<ney; j++)
        {
            MB[i+j*nex]=MA[i+j*nex];
        }
    } 
   
   for(i=1; i<(nex-1); i++)
    {
        for(j=1; j<(ney-1); j++)
        {
            MA[i+j*nex]=(MB[i+j*nex]+
                         MB[i+(j+1)*nex]+
                         MB[i+(j-1)*nex]+
                         MB[i+1+j*nex]+
                         MB[i+1+(j+1)*nex]+
                         MB[i+1+(j-1)*nex]+
                         MB[i-1+j*nex]+
                         MB[i-1+(j+1)*nex]+
                         MB[i-1+(j-1)*nex]
                         )/9.0;
        }
    } 
      
   return;
}

/*--------------------------------------------------------------------------------
	MesoSmooth
--------------------------------------------------------------------------------*/
void MesoSmooth()
{    
    int num_levels;
    SolveBlock *SolBlock;
    int i,nex,ney;
    MGBlock *Meshes;
    double *Swap;
    int num;
    GraphicBlock *localblock;
    double *MatFieldE;
    
    PutGetGraphicBlock(1, &localblock);
    num= localblock->SmoothNum;
    PutGetSolveBlock(1,&SolBlock); 
      
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    nex=(SolBlock->Meshes[num_levels-1].nnx)-1;
    ney=(SolBlock->Meshes[num_levels-1].nny)-1; 
    Swap=SolBlock->Meshes[num_levels-1].Swap;
    MatFieldE=SolBlock->Meshes[num_levels-1].MatFieldE;
    
    printf("\nnum %i\n",num);    
    for(i=0; i<num; i++)
    {
        SmoothMatField(nex, ney, MatFieldE, Swap);
    }
    TransMatFieldToCoarse(Meshes, num_levels);
    localblock->NewObjectFlag=1; 
    UpdateGraphicWindow(); 
  return;
}
