/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


/*------------------------------------------------------------------------------
	PolyVal    !abgendert -a09-!
------------------------------------------------------------------------------*/
double PolyVal(PolynomStruct P, double x)
{ 			
    int i,n;
    double v;
    double *c;
        
    n=P.numcoeff-1;
    c=P.coefficient;
    v=c[n];
    
    if (x==0.0) return c[0];
    
    for(i=n; i>0; i--)
    {
        v=v*x+c[i-1];
    }
    
return v;	
}


/*------------------------------------------------------------------------------
	AddPolynoms  !neu -n21-!
------------------------------------------------------------------------------*/
void AddPolynoms( PolynomStruct *P1, PolynomStruct *P2)
{   
    int i;
    int numcoeff, numcoeff1, numcoeff2;
    double *zw;    
    
    if (P1->numcoeff >= P2->numcoeff) numcoeff=P1->numcoeff;
    else numcoeff=P2->numcoeff;
    numcoeff1=P1->numcoeff;
    numcoeff2=P2->numcoeff;
        
    InitVector_double(numcoeff, &zw);
    
    for(i=0; i<numcoeff1; i++)
    {
      zw[i] += (P1->coefficient[i]);       
    }         
    
    for(i=0; i<numcoeff2; i++)
    {
      zw[i] += (P2->coefficient[i]);       
    }
             
    DeletePolynomFunction(P1);         
    P1->coefficient = zw;             
    P1->numcoeff = numcoeff;
    
    return;
}


/*------------------------------------------------------------------------------
	CopyPolynoms  !neu -n22-!
------------------------------------------------------------------------------*/
void CopyPolynoms( PolynomStruct *P1, PolynomStruct *P2)
{   
    int i;
    int numcoeff;
    double *zw;
    
    numcoeff=P2->numcoeff;    
    InitVector_double(numcoeff, &zw);
    
    for(i=0; i<numcoeff; i++)
    {
      zw[i] = (P2->coefficient[i]);       
    }         
             
    DeletePolynomFunction(P1);         
    P1->coefficient = zw;             
    P1->numcoeff = numcoeff;
    
    return;
}


/*------------------------------------------------------------------------------
	DeletePolynomFunction  !neu -n07-!
------------------------------------------------------------------------------*/
void DeletePolynomFunction(PolynomStruct *P)
{   
    free(P->coefficient);
    P->coefficient=NULL;
    return;
}


/*------------------------------------------------------------------------------
	DerivePolynomStruct   !neu -n10-!
------------------------------------------------------------------------------*/
void DerivePolynomStruct(PolynomStruct *P)
{
    int i;
    int numcoeff;
    double *zw;
        
    numcoeff = P->numcoeff;
    InitVector_double(numcoeff-1, &zw);
    
    for(i=1; i<numcoeff; i++)
    {
      zw[i-1] = (double)i*(P->coefficient[i]);       
    }         
             
    DeletePolynomFunction(P);         
    P->coefficient = zw;             
    P->numcoeff = numcoeff-1;
    
    return; 
}


/*------------------------------------------------------------------------------
	InitPolynomFunction  !neu -n05-!
------------------------------------------------------------------------------*/
void InitPolynomFunction(PolynomStruct *P, int numcoeff)
{   
    InitVector_double(numcoeff,&(P->coefficient));
    P->numcoeff=numcoeff;
    return;
}


/*------------------------------------------------------------------------------
	InitPolynomStruct
------------------------------------------------------------------------------*/
void InitPolynomStruct(PolynomStruct **Vector, int numpoly)
{   
    *Vector=(PolynomStruct*) malloc(numpoly*sizeof(PolynomStruct));
    return;
}


/*------------------------------------------------------------------------------
	MultPolynoms  !neu -n23-!
------------------------------------------------------------------------------*/
void MultPolynoms( PolynomStruct *P1, PolynomStruct *P2)
{   
    int i,j;
    int numcoeff, numcoeff1, numcoeff2;
    double *zw;
    
    numcoeff1=P1->numcoeff; 
    numcoeff2=P2->numcoeff;
    numcoeff =numcoeff1+numcoeff2-1;    
    InitVector_double(numcoeff, &zw);
    
    for(i=0; i<numcoeff1; i++)
    for(j=0; j<numcoeff2; j++)
    {
      zw[i+j] += (P1->coefficient[i])*(P2->coefficient[j]);       
    }         
             
    DeletePolynomFunction(P1);         
    P1->coefficient = zw;             
    P1->numcoeff = numcoeff;
    
    return;
}


/*------------------------------------------------------------------------------
	PrintPolynomStruct  !neu -n24-!
------------------------------------------------------------------------------*/

void PrintPolynomStruct(PolynomStruct *P)
{
    int i,m;
    double *C;
    
    m = P->numcoeff;      
    C = &(P->coefficient[0]);
    
    printf("\n\n\n");
    
    for(i=0; i<m; i++)
    {             
    printf(" +%10.6f*x^%d\t",C[i] ,i);
    if ( floor(((double)i+1.0)/2.0) == ceil(((double)i+1.0)/2.0) ) printf("\n");
    }
    return; 
}


/*------------------------------------------------------------------------------
	PrintSplinePolynomStruct  !neu -n25-!
------------------------------------------------------------------------------*/

int PrintSplinePolynomStruct( SomeElements *SpElems, int e, int p)
{
    int i,m;
    double *C;
    
    m = SpElems->numcoeff;
    if ((SpElems->numElements <= e) || (SpElems->numpoly <= p)) return 0;
    if ((e <= -1) || (p <= -1)) return 0;  
    C = &(SpElems->Element[e].Polynom[p].coefficient[0]);
    
    printf("\n\n\n");
    
    for(i=0; i<m; i++)
    {             
    printf(" +%.20f*x^%d\t",C[i] ,i);
    if ( floor(((double)i+1.0)/2.0) == ceil(((double)i+1.0)/2.0) ) printf("\n");
    }
    return 1; 
}

