/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyMainWindowUI.h"
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>

/*********************************************************************************
                               MAIN PROGRAM OF POSTPROCESSING
********************************************************************************/
void PostMainLoop(void)
{    
    int     i,j,Factor;
    int     num_levels;
    int     show_level;
    MGBlock *Meshes, *M, *MF;
    double lambda;
    double Radius;
    SolveBlock *SolBlock;
    GraphicBlock *graphblock;
 
    AllWinStruct *AllWinData;
    SolveWinStruct *SolveWinData;
 /*--------- bspline declaration--------- */
    MGBlock *SplineMesh;
    int direction;   
    BSplineBlock *BSP;
    int ModelDim;
/*--------- end of declaration--------- */
   
    ModelDim=PutGetModelDim(1,123);
    if (3==ModelDim)
    {
    PostMainLoop3d();
    return;
    }
 


    PutGetAllWinStruct(1, &AllWinData); 
    SolveWinData=AllWinData->SolveWinData;    
    
    PutGetGraphicBlock(1, &graphblock); 
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    show_level=graphblock->show_level;
    M=&(Meshes[show_level]);
  

    if(0==strcmp("E-Modul",graphblock->SelectObject)) 
    ElemMatrix2ElemNodes(M->nnx, M->nny, M->MatFieldE,
              M->Post1, M->Post2, M->Post3, M->Post4); 

    if(0==strcmp("Damage",graphblock->SelectObject)) 
    ElemMatrix2ElemNodes(M->nnx, M->nny, M->DamageOmega,
              M->Post1, M->Post2, M->Post3, M->Post4);  
    
    
    if(0==strcmp("Material Type",graphblock->SelectObject)) 
    ElemMatrixInt2ElemNodes(M->nnx, M->nny, M->MatFieldInt,
              M->Post1, M->Post2, M->Post3, M->Post4); 
              
    if(0==strcmp("Color Index",graphblock->SelectObject)) 
    ElemMatrixInt2ElemNodes(M->nnx, M->nny, M->ColorIndex,
              M->Post1, M->Post2, M->Post3, M->Post4); 

/* Q4-Element */
if (0==(SolveWinData->BSplineFlag))
{    
    if(0==strcmp("Displacement Ux",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->Ux,
              M->Post1, M->Post2, M->Post3, M->Post4); 
              
    if(0==strcmp("Displacement Uy",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->Uy,
              M->Post1, M->Post2, M->Post3, M->Post4); 
    
    if(0==strcmp("Increment dUx",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->dUx,
              M->Post1, M->Post2, M->Post3, M->Post4); 
    
    if(0==strcmp("Increment dUy",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->dUy,
              M->Post1, M->Post2, M->Post3, M->Post4); 
 
    if(0==strcmp("Unbalanced Fx",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->Fx,
              M->Post1, M->Post2, M->Post3, M->Post4); 

    if(0==strcmp("Unbalanced Fy",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->Fy,
              M->Post1, M->Post2, M->Post3, M->Post4);        
    
    if(0==strcmp("Increment dFx",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->SwapX,
              M->Post1, M->Post2, M->Post3, M->Post4); 

    if(0==strcmp("Increment dFy",graphblock->SelectObject)) 
    NodeMatrix2ElemNodes(M->nnx, M->nny, M->SwapY,
              M->Post1, M->Post2, M->Post3, M->Post4);         

    if(0==strcmp("Strain Exx",graphblock->SelectObject)) 
    PostStrainExx(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, 
    M->Post1, M->Post2, M->Post3, M->Post4); 
    
    if(0==strcmp("Strain Eyy",graphblock->SelectObject)) 
    PostStrainEyy(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, 
    M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Strain Exy",graphblock->SelectObject)) 
    PostStrainExy(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, 
    M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
    PostStressSxx(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, M->MatFieldE, 
    M->MatFieldInt, SolBlock->Material, M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
    PostStressSyy(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, M->MatFieldE, 
    M->MatFieldInt, SolBlock->Material, M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
    PostStressSxy(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, M->MatFieldE, 
    M->MatFieldInt, SolBlock->Material, M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Mises Stress",graphblock->SelectObject)) 
    PostStressMises(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, M->MatFieldE, 
    M->MatFieldInt, SolBlock->Material, M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Eq Strain",graphblock->SelectObject)) 
    PostEqStrain(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy, M->MatFieldE,
                 M->MatFieldInt, M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Eq Strain (C)",graphblock->SelectObject)) 
    {
       PostEqStrainCenter(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy,
        M->MatFieldE, M->MatFieldInt, M->EqStrain);
       ElemMatrix2ElemNodes(M->nnx, M->nny, M->EqStrain,
              M->Post1, M->Post2, M->Post3, M->Post4);  
    }
    
    if(0==strcmp("NLEq Strain",graphblock->SelectObject)) 
    {
       //printf("out1\n");
       PostEqStrainCenter(M->nnx, M->nny, M->la, M->lb, M->Ux, M->Uy,
        M->MatFieldE, M->MatFieldInt, M->Swap);
        Radius=SolveWinData->NLRadius;
       PostEqStrainNonLocal(M->nnx, M->nny, M->la, M->lb, 
              M->MatFieldInt, Radius, M->Swap, M->EqStrain);
       ElemMatrix2ElemNodes(M->nnx, M->nny, M->EqStrain,
              M->Post1, M->Post2, M->Post3, M->Post4);  
    }
    

    
}    
/*else // B-Spline Element
{    
    SplineMesh=SolBlock->SplineMesh;
    BSP=&(SolBlock->BSplines);
    
    if(0==strcmp("Displacement Ux",graphblock->SelectObject)) 
    {
    direction=0;
    TransformSplinesOnMesh2(0,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    }
    
    if(0==strcmp("Displacement Uy",graphblock->SelectObject)) 
    {
    direction=0;
    TransformSplinesOnMesh2(1,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    }

    if(0==strcmp("Strain Exx",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(0,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Strain Eyy",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(1,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Strain Exy",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(2,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    
    if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(3,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(4,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);
    if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
    TransformSplinesOnMesh3(5,   M->nnx, M->nny, M->la, M->lb, 
                            SplineMesh->Ux, SplineMesh->Uy, M->MatFieldE, BSP,
                            M->Post1, M->Post2, M->Post3, M->Post4);

}*/
else /* B-Spline Element order k*/
{    
    SplineMesh=SolBlock->SplineMesh;
    BSP=&(SolBlock->BSplines);
    M=&(Meshes[num_levels-1]);
    
    if (1 == SolBlock->FinerSplineFlag)
    { 
        MF=SolBlock->FinerSplineMesh;
        Factor=SolBlock->FinerSpMeshFactor;
            
        if(0==strcmp("Displacement Ux",graphblock->SelectObject)) 
        TransformSplinesOnMesh4(0, Factor, M->nex, M->ney, M->la, M->lb,
                                SplineMesh->Ux, SplineMesh->Uy, SplineMesh->nnx, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
            
        if(0==strcmp("Displacement Uy",graphblock->SelectObject)) 
        TransformSplinesOnMesh4(1, Factor, M->nex, M->ney, M->la, M->lb,
                                SplineMesh->Ux, SplineMesh->Uy, SplineMesh->nnx, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
        
        if(0==strcmp("Strain Exx",graphblock->SelectObject)) 
        TransformSplinesOnMesh5(0, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
        
        if(0==strcmp("Strain Eyy",graphblock->SelectObject)) 
        TransformSplinesOnMesh5(1, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
        
        if(0==strcmp("Strain Exy",graphblock->SelectObject)) 
        TransformSplinesOnMesh5(2, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
        
        
        if(1 == AllWinData->SolveWinData->YMsmoothFlag)
        {
         if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
         TransformSplinesOnMesh5a(3, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
         if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
         TransformSplinesOnMesh5a(4, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
         if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
         TransformSplinesOnMesh5a(5, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);   
        }
        else
        {
         if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
         TransformSplinesOnMesh5(3, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
         if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
         TransformSplinesOnMesh5(4, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
         if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
         TransformSplinesOnMesh5(5, Factor, M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                MF->Post1, MF->Post2, MF->Post3, MF->Post4);
        } 
    }
    else 
    {   //temporary: Fx
        if(0==strcmp("Unbalanced Fx",graphblock->SelectObject)) 
        NodeMatrix2ElemNodes(M->nnx, M->nny, M->Fx,
              M->Post1, M->Post2, M->Post3, M->Post4); 
        //temporary:Fx
        
        if(0==strcmp("Displacement Ux",graphblock->SelectObject)) 
        TransformSplinesOnMesh2_K(0, M->nex, M->ney, M->la, M->lb, 
                                SplineMesh->Ux, SplineMesh->Uy, SplineMesh->nnx, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
            
        if(0==strcmp("Displacement Uy",graphblock->SelectObject)) 
        TransformSplinesOnMesh2_K(1, M->nex, M->ney, M->la, M->lb, 
                                SplineMesh->Ux, SplineMesh->Uy, SplineMesh->nnx, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
        
        if(0==strcmp("Strain Exx",graphblock->SelectObject)) 
        TransformSplinesOnMesh3_K(0,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
        
        if(0==strcmp("Strain Eyy",graphblock->SelectObject)) 
        TransformSplinesOnMesh3_K(1,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
        
        if(0==strcmp("Strain Exy",graphblock->SelectObject)) 
        TransformSplinesOnMesh3_K(2,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
        
        
        
        if(1 == AllWinData->SolveWinData->YMsmoothFlag)
        {
         if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
         TransformSplinesOnMesh3a(3,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
         if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
         TransformSplinesOnMesh3a(4,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
         if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
         TransformSplinesOnMesh3a(5,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->KnMatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);    
        }
        else
        {
         if(0==strcmp("Stress Sxx",graphblock->SelectObject)) 
         TransformSplinesOnMesh3_K(3,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
         if(0==strcmp("Stress Syy",graphblock->SelectObject)) 
         TransformSplinesOnMesh3_K(4,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
         if(0==strcmp("Stress Sxy",graphblock->SelectObject)) 
         TransformSplinesOnMesh3_K(5,   M->nex, M->ney, M->la, M->lb, SplineMesh->Ux,
                                SplineMesh->Uy, SplineMesh->nnx, M->MatFieldE, BSP,
                                M->Post1, M->Post2, M->Post3, M->Post4);
        }                        
    }  
}
/* End of Bspline Element */

    if (1==(graphblock->PostSmoothFlag)) 
    {
        PostSmooth(M->nnx, M->nny, 
        M->Post1, M->Post2, M->Post3, M->Post4);
        printf("\nsmooth\n");
    }
    else printf("\nNONsmooth\n");
    
    
    return;
}

/*------------------------------------------------------------------------------
     NodeMatrix2ElemNodes
------------------------------------------------------------------------------*/
int NodeMatrix2ElemNodes(int nnx, int nny, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    int nex;
    size_t nexSize;
    nex=nnx-1;       
    nexSize=sizeof(double)*nex;
                
    for (j = 0; j < (nny-1); j++) 
    {     
        memcpy(&(Post1[j*nex]),&(M[j*nnx])      ,nexSize);
        memcpy(&(Post2[j*nex]),&(M[j*nnx+1])    ,nexSize);
        memcpy(&(Post3[j*nex]),&(M[(j+1)*nnx+1]),nexSize);
        memcpy(&(Post4[j*nex]),&(M[(j+1)*nnx])  ,nexSize);
    }
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     ElemMatrix2ElemNodes
------------------------------------------------------------------------------*/
int ElemMatrix2ElemNodes(int nnx, int nny, double *M,
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    size_t FullSize;
      
    FullSize=sizeof(double)*(nnx-1)*(nny-1);
    
        memcpy(Post1,M,FullSize);
        memcpy(Post2,M,FullSize);
        memcpy(Post3,M,FullSize);
        memcpy(Post4,M,FullSize);
        
    return(1);
}

/*------------------------------------------------------------------------------
     ElemMatrixInt2ElemNodes
------------------------------------------------------------------------------*/
int ElemMatrixInt2ElemNodes(int nnx, int nny, int *M,
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double tmp;
    size_t FullSize;
      
    FullSize=sizeof(double)*(nnx-1)*(nny-1);
    
    for(i=0;i<(nnx-1);i++)
    for(j=0;j<(nny-1);j++)
    {
        p=i+j*(nnx-1);
        tmp=(double)M[p];
        Post1[p]=tmp;
        Post2[p]=tmp;
        Post3[p]=tmp;
        Post4[p]=tmp;
        
    }        
    return(1);
}

/*------------------------------------------------------------------------------
     PostStrainExx
------------------------------------------------------------------------------*/
int PostStrainExx(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, 
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    
    nex=nnx-1;       
          
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; p++;
            u2=Ux[p]; p+=nnx;
            u3=Ux[p]; p--;
            u4=Ux[p]; 
	        
	        
	        //node 1
            Post1[i+j*nex]=(u2-u1)/la; 
            //node 2
            Post2[i+j*nex]=(u2-u1)/la; 
            //node 3
            Post3[i+j*nex]=(u3-u4)/la; 
            //node 4
            Post4[i+j*nex]=(u3-u4)/la; 
         }
    }
  	    
    return(1);
}
/*------------------------------------------------------------------------------
     PostStrainEyy
------------------------------------------------------------------------------*/
int PostStrainEyy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy,  
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    
    nex=nnx-1;       
           
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            v1=Uy[p]; p++;
            v2=Uy[p]; p+=nnx;
            v3=Uy[p]; p--;
            v4=Uy[p]; 
	        
	        
	        //node 1
            Post1[i+j*nex]=(v4-v1)/lb; 
            //node 2
            Post2[i+j*nex]=(v3-v2)/lb; 
            //node 3
            Post3[i+j*nex]=(v3-v2)/lb; 
            //node 4
            Post4[i+j*nex]=(v4-v1)/lb; 
         }
    }
  	    
    return(1);
}
/*------------------------------------------------------------------------------
     PostStrainExy
------------------------------------------------------------------------------*/
int PostStrainExy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy,  
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    
    nex=nnx-1;       
             
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; p++;
            u2=Ux[p]; p+=nnx;
            u3=Ux[p]; p--;
            u4=Ux[p]; 
	        
            p=i+j*nnx;
            
            v1=Uy[p]; p++;
            v2=Uy[p]; p+=nnx;
            v3=Uy[p]; p--;
            v4=Uy[p]; 
            
            
	        //node 1
            Post1[i+j*nex]=(u4-u1)/lb+(v2-v1)/la; 
            //node 2
            Post2[i+j*nex]=(u3-u2)/lb+(v2-v1)/la; 
            //node 3
            Post3[i+j*nex]=(u3-u2)/lb+(v3-v4)/la; 
            //node 4
            Post4[i+j*nex]=(u4-u1)/lb+(v3-v4)/la; ; 
         }
    }
  	    
    return(1);
}
/*------------------------------------------------------------------------------
     PostStressSxx
------------------------------------------------------------------------------*/
int PostStressSxx(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat, 
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    double mu,N;
    
    //mu=0.2;
   
    
    
    nex=nnx-1;       
         
    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; p++;
            u2=Ux[p]; p+=nnx;
            u3=Ux[p]; p--;
            u4=Ux[p]; 
	        
	        p=i+j*nnx;
            
            v1=Uy[p]; p++;
            v2=Uy[p]; p+=nnx;
            v3=Uy[p]; p--;
            v4=Uy[p]; 
	        
	        p=i+j*nex;
	        mu=mat[MatInt[p]].Mu;
	         N=1.0/(1.0-mu*mu);
	        //node 1
            Post1[p]=      ((u2-u1)/la + mu*(v4-v1)/lb) *N*E[p]; 
            //node 2
            Post2[p]=    ((u2-u1)/la + mu*(v3-v2)/lb) *N*E[p]; 
            //node 3
            Post3[p]=((u3-u4)/la + mu*(v3-v2)/lb) *N*E[p]; 
            //node 4
            Post4[p]=  ((u3-u4)/la + mu*(v4-v1)/lb) *N*E[p]; 
         }
    }                
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostStressSyy
------------------------------------------------------------------------------*/
int PostStressSyy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MatInt, MaterialBlock *mat, 
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    double mu,N;
    
    //mu=0.2;
    
    
    nex=nnx-1;       
            
    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; p++;
            u2=Ux[p]; p+=nnx;
            u3=Ux[p]; p--;
            u4=Ux[p]; 
	        
	        p=i+j*nnx;
            
            v1=Uy[p]; p++;
            v2=Uy[p]; p+=nnx;
            v3=Uy[p]; p--;
            v4=Uy[p]; 
	        
	        p=i+j*nex;
	        mu=mat[MatInt[p]].Mu;
	        N=1.0/(1.0-mu*mu);
	        //node 1
            Post1[p]=      (mu*(u2-u1)/la + (v4-v1)/lb) *N*E[p]; 
            //node 2
            Post2[p]=    (mu*(u2-u1)/la + (v3-v2)/lb) *N*E[p]; 
            //node 3
            Post3[p]=(mu*(u3-u4)/la + (v3-v2)/lb) *N*E[p]; 
            //node 4
            Post4[p]=  (mu*(u3-u4)/la + (v4-v1)/lb) *N*E[p]; 
         }
    }  
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostStressSxy
------------------------------------------------------------------------------*/
int PostStressSxy(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E,  int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
  
    int nex;
    double mu,M;
    
    //mu=0.2;
    
    
   
    nex=nnx-1;                
    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; p++;
            u2=Ux[p]; p+=nnx;
            u3=Ux[p]; p--;
            u4=Ux[p]; 
	        
	        p=i+j*nnx;
            
            v1=Uy[p]; p++;
            v2=Uy[p]; p+=nnx;
            v3=Uy[p]; p--;
            v4=Uy[p]; 
	        
	        p=i+j*nex;
	        mu=mat[MatInt[p]].Mu;
	        M=1.0/(1.0-mu*mu)*(1.0-mu)*0.5;
	        //node 1
            Post1[p]=((u4-u1)/lb+(v2-v1)/la)*M*E[p]; 
            //node 2
            Post2[p]=((u3-u2)/lb+(v2-v1)/la)*M*E[p]; 
            //node 3
            Post3[p]=((u3-u2)/lb+(v3-v4)/la)*M*E[p]; 
            //node 4
            Post4[p]=((u4-u1)/lb+(v3-v4)/la)*M*E[p]; 
         }
    }  
  	    
    return(1);
}

/*------------------------------------------------------------------------------
     PostStressMises
------------------------------------------------------------------------------*/
int PostStressMises(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E,  int *MatInt, MaterialBlock *mat,
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    int nex;
    double mu,N,M;
    double Sxx, Syy, Sxy;
        
    nex=nnx-1;        
                    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; v1=Uy[p]; p++;
            u2=Ux[p]; v2=Uy[p]; p+=nnx;
            u3=Ux[p]; v3=Uy[p]; p--;
            u4=Ux[p]; v4=Uy[p];
	        
	        p=i+j*nex;
	        mu=mat[MatInt[p]].Mu;
            N=1.0/(1.0-mu*mu);
            M=1.0/(1.0-mu*mu)*(1.0-mu)*0.5;
            //node 1
            Sxx= ((u2-u1)/la + mu*(v4-v1)/lb) *N*E[p]; 
            Syy= (mu*(u2-u1)/la + (v4-v1)/lb) *N*E[p];  
            Sxy= ((u4-u1)/lb+(v2-v1)/la)*M*E[p];              
            Post1[p]=sqrt(Sxx*Sxx-Sxx*Syy+Syy*Syy+3*Sxy*Sxy);
            
            //node 2
            Sxx= ((u2-u1)/la + mu*(v3-v2)/lb) *N*E[p]; 
            Syy= (mu*(u2-u1)/la + (v3-v2)/lb) *N*E[p];
            Sxy= ((u3-u2)/lb+(v2-v1)/la)*M*E[p]; 
            Post2[p]=sqrt(Sxx*Sxx-Sxx*Syy+Syy*Syy+3*Sxy*Sxy);
            
            //node 3
            Sxx= ((u3-u4)/la + mu*(v3-v2)/lb) *N*E[p]; 
            Syy= (mu*(u3-u4)/la + (v3-v2)/lb) *N*E[p]; 
            Sxy=((u3-u2)/lb+(v3-v4)/la)*M*E[p]; 
            Post3[p]=sqrt(Sxx*Sxx-Sxx*Syy+Syy*Syy+3*Sxy*Sxy);
            
            //node 4
            Sxx= ((u3-u4)/la + mu*(v4-v1)/lb) *N*E[p]; 
            Syy= (mu*(u3-u4)/la + (v4-v1)/lb) *N*E[p]; 
            Sxy=((u4-u1)/lb+(v3-v4)/la)*M*E[p]; 
            Post4[p]=sqrt(Sxx*Sxx-Sxx*Syy+Syy*Syy+3*Sxy*Sxy);
         }
    }
  
    return(1);
}

/*------------------------------------------------------------------------------
     PostEqStrain
------------------------------------------------------------------------------*/

int PostEqStrain(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MType,  
              double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    
    int nex;
    double mu,N,M;
    double Sxx, Syy, Sxy;
        
    
    nex=nnx-1;        
    mu=0.2;
    N=1.0/(1.0-mu*mu);
    M=1.0/(1.0-mu*mu)*(1.0-mu)*0.5;
                    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; v1=Uy[p]; p++;
            u2=Ux[p]; v2=Uy[p]; p+=nnx;
            u3=Ux[p]; v3=Uy[p]; p--;
            u4=Ux[p]; v4=Uy[p];
	        
	        p=i+j*(nnx-1);
            
            if (MType[p]>0)
            {
            Post1[p]=0.0;
            Post2[p]=0.0;
            Post3[p]=0.0;
            Post4[p]=0.0;
            }
            else
            {                     
            //node 1
            Sxx= ((u2-u1)/la + mu*(v4-v1)/lb) *N*E[p]; 
            Syy= (mu*(u2-u1)/la + (v4-v1)/lb) *N*E[p];  
            Sxy= ((u4-u1)/lb+(v2-v1)/la)*M*E[p];        
            if (Sxx<0.0) Sxx=0.0;   
            if (Syy<0.0) Syy=0.0;  
            if (Sxy<0.0) Sxy=0.0;     
            Post1[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];
            
            //node 2
            Sxx= ((u2-u1)/la + mu*(v3-v2)/lb) *N*E[p]; 
            Syy= (mu*(u2-u1)/la + (v3-v2)/lb) *N*E[p];
            Sxy= ((u3-u2)/lb+(v2-v1)/la)*M*E[p]; 
            if (Sxx<0.0) Sxx=0.0;   
            if (Syy<0.0) Syy=0.0;  
            if (Sxy<0.0) Sxy=0.0;
            Post2[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];
            
            //node 3
            Sxx= ((u3-u4)/la + mu*(v3-v2)/lb) *N*E[p]; 
            Syy= (mu*(u3-u4)/la + (v3-v2)/lb) *N*E[p]; 
            Sxy=((u3-u2)/lb+(v3-v4)/la)*M*E[p]; 
            if (Sxx<0.0) Sxx=0.0;   
            if (Syy<0.0) Syy=0.0;  
            if (Sxy<0.0) Sxy=0.0;
            Post3[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];
            
            //node 4
            Sxx= ((u3-u4)/la + mu*(v4-v1)/lb) *N*E[p]; 
            Syy= (mu*(u3-u4)/la + (v4-v1)/lb) *N*E[p]; 
            Sxy=((u4-u1)/lb+(v3-v4)/la)*M*E[p]; 
            if (Sxx<0.0) Sxx=0.0;   
            if (Syy<0.0) Syy=0.0;  
            if (Sxy<0.0) Sxy=0.0;
            Post4[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];
            }
         }
    }
      
    return(1);
}

/*------------------------------------------------------------------------------
     PostEqStrainCenter
------------------------------------------------------------------------------*/

int PostEqStrainCenter(int nnx, int nny, double la, double lb, 
              double *Ux, double *Uy, double *E, int *MType,  
              double *EqStrain)
{
    int i,j,p;
    double u1,u2,u3,u4,v1,v2,v3,v4;
    
    int nex;
    double mu,N,M;
    double Sxx, Syy, Sxy;
    double S1,S2,tmp,tmp1,tmp2;
        
    
    nex=nnx-1;        
    mu=0.2;
    N=1.0/(1.0-mu*mu);
    M=1.0/(1.0-mu*mu)*(1.0-mu)*0.5;
                    
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            p=i+j*nnx;
            
            u1=Ux[p]; v1=Uy[p]; p++;
            u2=Ux[p]; v2=Uy[p]; p+=nnx;
            u3=Ux[p]; v3=Uy[p]; p--;
            u4=Ux[p]; v4=Uy[p];
	        
	        p=i+j*(nnx-1);
            if (MType[p]>0)
            {
                EqStrain[p]=0.0;
            }
            else
            {
            Sxx=((u2-u1+u3-u4)/la + mu*(v4-v1+v3-v2)/lb)*N*E[p]*0.5;
            Syy=(mu*(u2-u1+u3-u4)/la + (v4-v1+v3-v2)/lb)*N*E[p]*0.5; 
            Sxy=((u4-u1+u3-u2)/lb+(v2-v1+v3-v4)/la)*M*E[p]*0.5;    
                           
            /*if (Sxx<0.0) Sxx=0.0;   
            if (Syy<0.0) Syy=0.0;  
            if (Sxy<0.0) Sxy=0.0;     
            EqStrain[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];*/
            
            tmp=(Sxx-Syy)/2.0;
            tmp1=sqrt(tmp*tmp+Sxy*Sxy);
            tmp2=(Sxx+Syy)/2.0;
            S1=tmp2+tmp1;
            S2=tmp2-tmp1;
            if (S1<0.0) S1=0.0;  
            if (S2<0.0) S2=0.0; 
            EqStrain[p]=sqrt(Sxx*Sxx+Syy*Syy+0.5*Sxy*Sxy)/E[p];
            
            }
          }
    }
      
    return(1);
}


/*------------------------------------------------------------------------------
     PostEqStrainNonLocal
------------------------------------------------------------------------------*/

int PostEqStrainNonLocal(int nnx, int nny, double la, double lb, 
              int *MType, double Radius, double *Swap, double *EqStrain)
{
    int i,j,k,p,q,num;
    int mc, nc;
    int ra,rb,listlen;
    int nex,ney;
    int global, lokal;
    double r;
    
    int nlx,nly;
    int *M, *N;
    double *W;
    double w, wsum;
    double tmp1, tmp2;
    double RadSQ;
     
    nex=nnx-1; 
    ney=nny-1; 
    RadSQ=Radius*Radius;      
    
    ra=(int)(ceil(Radius/la));
    rb=(int)(ceil(Radius/lb));
    nlx=ra*2+1;
    nly=rb*2+1;
    listlen=nlx*nly;
    
    mc=ra;
    nc=rb;
     
    InitVector_int(listlen, &M);
    InitVector_int(listlen, &N);
    InitVector_double(listlen, &W);
    
    for (j = 0; j < nly; j++) 
    {     
        for (i = 0; i < nlx; i++)   
        {
          num=i+nlx*j;
          
          M[num]=(i-mc);  
          N[num]=(j-nc);
          
          tmp1=(double)M[num];    
          tmp2=(double)N[num];
         
          tmp1*=la;
          tmp2*=lb;
          
          r=sqrt(tmp1*tmp1+tmp2*tmp2);
         
          if (r>Radius) W[num]=0.0;
              else
              {
                  tmp1=(1.0-(r*r)/RadSQ);
                  W[num]=tmp1*tmp1;
              }   
        }
    }
                                                               
                                                                                                  
    for (j = 0; j < ney; j++) 
    {     
        for (i = 0; i < nex; i++)   
        {
            wsum=0.0;
            global=i+j*nex;
            EqStrain[global]=0.0;
                
            if (MType[global]<1)
            {
                for (k = 0; k < listlen; k++)   
                {
                    p=i+M[k];
                    q=j+N[k]; 
                    lokal=p+q*nex;
                
                    if((p>(-1)) && (p<nex) && (q>(-1)) && (q<ney))
                    {
                        if (MType[lokal]<1)
                        {
                            w=W[k]; 
                            wsum+=w;
                            EqStrain[global]+=Swap[lokal]*w;
                        }
                    }
                }   
                EqStrain[global]/=wsum;
            }
        }
    }
    
      
    free(M);
    free(N);
    free(W);      
    return(1);
}



/*------------------------------------------------------------------------------
     PostSmooth
------------------------------------------------------------------------------*/
int PostSmooth(int nnx, int nny, double *Post1, double *Post2, double *Post3, double *Post4)
{
    int i,j;
    int nex;
    double tmp;
       
    nex=nnx-1;       
              
    for (j = 0; j < (nny-1); j++) 
    {     
        for (i = 0; i < (nnx-2); i++)   
        {
            tmp=(Post2[i+j*nex]    + Post1[i+1+j*nex])*0.5;
            Post2[i+j*nex]  =tmp;
            Post1[i+1+j*nex]=tmp;  
            
            tmp=(Post3[i+j*nex]    + Post4[i+1+j*nex])*0.5;
            Post3[i+j*nex]  =tmp;
            Post4[i+1+j*nex]=tmp;  
        }
    }
    
    for (j = 0; j < (nny-2); j++) 
    {     
        for (i = 0; i < (nnx-1); i++)   
        {
            tmp=(Post4[i+j*nex]    + Post1[i+(j+1)*nex])*0.5;
            Post4[i+j*nex]   = tmp;
            Post1[i+(j+1)*nex]=tmp;
            
            tmp=(Post3[i+j*nex]    + Post2[i+(j+1)*nex])*0.5;
            Post3[i+j*nex]   = tmp;
            Post2[i+(j+1)*nex]=tmp;  
        }
    }
    
                  	    
    return(1);


}
/*------------------------------------------------------------------------------
     PostSave
------------------------------------------------------------------------------*/
int PostSave(void)
{
    int i,j; 
    FILE    *outp ;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    GraphicBlock *graphblock;
    int Pstep;
    
    printf("..............Saving Post....................\n");
    outp = fopen("./mises.mg","wb");   
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    PutGetGraphicBlock(1, &graphblock);
    
    Pstep=graphblock->PostSaveStep;
    if (Pstep<1) Pstep=1;
    
    if (outp == NULL)  
   	{   
  		printf("Error: cannot open the file ");
   	    getchar();
   	}
    else
    {        
      for(i=0; i<M->nnx-1; i+=Pstep)
      {
        for(j=0; j<M->nny-1;j+=Pstep)
        {
            fprintf(outp,"%g\n",(M->Post1[i+j*(M->nnx-1)]));
        }
      }
      fclose(outp);   
    }
    return(1);
}


/*------------------------------------------------------------------------------
     PostSaveProfile
------------------------------------------------------------------------------*/
int PostSaveProfile(void)
{
    int i,j;
    double w1,w2,w3,w4,w5; 
    FILE    *outp;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    GraphicBlock *graphblock;
    int Pstep;
    
    printf("..............Saving Post....................\n");
    outp = fopen("./profile.txt","wb");  
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    PutGetGraphicBlock(1, &graphblock);
    
    if (outp == NULL)  
   	{   
  		printf("Error: cannot open the file ");
   	    getchar();
   	}
    else
    {        
      
      if ((M->ney)%2==0) 
      {
       i=(M->ney)/2-1;
       fprintf(outp,"Mittelschnitt in x-Richtung: \n\n");
       fprintf(outp,"Wert 1 \t");
       fprintf(outp,"Wert 2 \t");
       fprintf(outp,"Wert 3 \t");
       fprintf(outp,"Wert 4 \t");
       fprintf(outp,"gemittelt \n\n");
       
       for(j=0; j<=(M->nex); j++)
       {
        if (j-1 > -1) w1=(M->Post3[j-1+i*(M->nex)]); else w1=0.0;
        if (j < (M->nex)) w2=(M->Post4[j+i*(M->nex)]); else w2=0.0;
        if (j < (M->nex)) w3=(M->Post1[j+(i+1)*(M->nex)]); else w3=0.0;
        if (j-1 > -1) w4=(M->Post2[j-1+(i+1)*(M->nex)]); else w4=0.0;
        
        if (j > 0 && j < (M->nex)) w5=(w1+w2+w3+w4)*0.25;
        else w5=(w1+w2+w3+w4)*0.5;
        
        fprintf(outp,"%g \t",w1);
        fprintf(outp,"%g \t",w2);
        fprintf(outp,"%g \t",w3);
        fprintf(outp,"%g \t",w4);
        fprintf(outp,"%g \n",w5);        
       }
      }
      
      if ((M->nex)%2==0) 
      {
       j=(M->nex)/2-1;
       fprintf(outp,"\n\nMittelschnitt in y-Richtung: \n\n");
       fprintf(outp,"Wert 1 \t");
       fprintf(outp,"Wert 2 \t");
       fprintf(outp,"Wert 3 \t");
       fprintf(outp,"Wert 4 \t");
       fprintf(outp,"gemittelt \n\n");
       
       for(i=0; i<=(M->ney); i++)
       {
        if (i-1 > -1) w1=(M->Post3[j+(i-1)*(M->nex)]); else w1=0.0;
        if (i-1 > -1) w2=(M->Post4[j+1+(i-1)*(M->nex)]); else w2=0.0;
        if (i < (M->ney)) w3=(M->Post1[j+1+i*(M->nex)]); else w3=0.0;
        if (i < (M->ney)) w4=(M->Post2[j+i*(M->nex)]); else w4=0.0;
        
        if (i > 0 && i < (M->ney)) w5=(w1+w2+w3+w4)*0.25;
        else w5=(w1+w2+w3+w4)*0.5;
        
        fprintf(outp,"%g \t",w1);
        fprintf(outp,"%g \t",w2);
        fprintf(outp,"%g \t",w3);
        fprintf(outp,"%g \t",w4);
        fprintf(outp,"%g \n",w5);        
       }
      }
       
      if ((M->ney)%2==0) 
      {
       i=(M->ney)/2-1;
       fprintf(outp,"\n\nMittelschnitt gemittelt in x-Richtung: \n\n");
       
       for(j=0; j<=(M->nex); j++)
       {
        if (j-1 > -1) w1=(M->Post3[j-1+i*(M->nex)]); else w1=0.0;
        if (j < (M->nex)) w2=(M->Post4[j+i*(M->nex)]); else w2=0.0;
        if (j < (M->nex)) w3=(M->Post1[j+(i+1)*(M->nex)]); else w3=0.0;
        if (j-1 > -1) w4=(M->Post2[j-1+(i+1)*(M->nex)]); else w4=0.0;
        
        if (j > 0 && j < (M->nex)) w5=(w1+w2+w3+w4)*0.25;
        else w5=(w1+w2+w3+w4)*0.5;
        
        fprintf(outp,"%g \n",w5);        
       }
      }
      
      if ((M->nex)%2==0) 
      {
       j=(M->nex)/2-1;
       fprintf(outp,"\n\nMittelschnitt gemittelt in y-Richtung: \n\n");
       
       for(i=0; i<=(M->ney); i++)
       {
        if (i-1 > -1) w1=(M->Post3[j+(i-1)*(M->nex)]); else w1=0.0;
        if (i-1 > -1) w2=(M->Post4[j+1+(i-1)*(M->nex)]); else w2=0.0;
        if (i < (M->ney)) w3=(M->Post1[j+1+i*(M->nex)]); else w3=0.0;
        if (i < (M->ney)) w4=(M->Post2[j+i*(M->nex)]); else w4=0.0;
        
        if (i > 0 && i < (M->ney)) w5=(w1+w2+w3+w4)*0.25;
        else w5=(w1+w2+w3+w4)*0.5;
        
        fprintf(outp,"%g \n",w5);        
       }
      }       
      fclose(outp);   
    }
    return(1);
}

/*------------------------------------------------------------------------------
     cb_PrintSectionvalues
------------------------------------------------------------------------------*/

void cb_PrintSectionvalues(void)
{
    int i,ix,iy;
    int nex,ney;
    int level;
    double w1,w2,w3,w4;
    double *li,*mi,*re;
    char tmp[30];
    
    MGBlock *M;
    MyMainWindowUI *mainwin;
    SolveBlock *SolBlock;
    
  
    PutGetMainWindow(1, &mainwin);
    PutGetSolveBlock(1,&SolBlock);
    
    if (SolBlock->FinerSplineFlag == 1) 
    {
        cb_PrintSectionvaluesFiner(); return;
    }
    
    level = (int)(mainwin->Detail->value());
    M = &(SolBlock->Meshes[level]);
    nex=M->nex;
    ney=M->ney;    
    ix=(int)mainwin->x_section->value();
    iy=(int)mainwin->y_section->value();
    
    //Schnitt auf x-Achse (in y-Richtung)
    InitVector_double(2*ney,&li);
    InitVector_double(1+ney,&mi);
    InitVector_double(2*ney,&re);
    
    if (ix != 0) //Vektor links
    for(i=0; i<ney; i++)
    {
       li[i*2]=M->Post2[(ix-1)+i*nex];
       li[i*2+1]=M->Post3[(ix-1)+i*nex];
    }
    
    if (ix != nex) //Vektor rechts
    for(i=0; i<ney; i++)
    {
       re[i*2]=M->Post1[ix+i*nex];
       re[i*2+1]=M->Post4[ix+i*nex];
    }
    
    for(i=0; i<ney+1; i++) //Berechnung Vektor mitte
    {
       if (i != 0) w1=li[i*2-1]; else w1=li[i*2];
       if (i != 0) w2=re[i*2-1]; else w2=re[i*2];
       if (i != ney) w3=li[i*2]; else w3=li[i*2-1];
       if (i != ney) w4=re[i*2]; else w4=re[i*2-1];
       if (ix > 0 && ix < nex) mi[i]=(w1+w2+w3+w4)*0.25;
       else mi[i]=(w1+w2+w3+w4)*0.5;
    }
    
    //Ausgabe
    sprintf(tmp, "./X_section_%i_li.txt",ix);
    PrintMatrixToFile(2*ney,1,li,tmp);
    sprintf(tmp, "./X_section_%i_mi.txt",ix);
    PrintMatrixToFile(1+ney,1,mi,tmp);
    sprintf(tmp, "./X_section_%i_re.txt",ix);
    PrintMatrixToFile(2*ney,1,re,tmp);
    
    free(li);
    free(mi);
    free(re);
    
    //Schnitt auf y-Achse (in x-Richtung)
    InitVector_double(2*nex,&li);
    InitVector_double(1+nex,&mi);
    InitVector_double(2*nex,&re);
    
    if (iy != ney) //Vektor links
    for(i=0; i<nex; i++)
    {
       li[i*2]=M->Post1[i+iy*nex];
       li[i*2+1]=M->Post2[i+iy*nex];
    }
    
    if (iy != 0) //Vektor rechts
    for(i=0; i<nex; i++)
    {
       re[i*2]=M->Post4[i+(iy-1)*nex];
       re[i*2+1]=M->Post3[i+(iy-1)*nex];
    }
    
    for(i=0; i<nex+1; i++) //Berechnung Vektor mitte
    {
       if (i != 0) w1=li[i*2-1]; else w1=li[i*2];
       if (i != 0) w2=re[i*2-1]; else w2=re[i*2];
       if (i != nex) w3=li[i*2]; else w3=li[i*2-1];
       if (i != nex) w4=re[i*2]; else w4=re[i*2-1];
       if (iy > 0 && iy < ney) mi[i]=(w1+w2+w3+w4)*0.25;
       else mi[i]=(w1+w2+w3+w4)*0.5;
    }
    
    //Ausgabe
    sprintf(tmp, "./Y_section_%i_li.txt",iy);
    PrintMatrixToFile(2*nex,1,li,tmp);
    sprintf(tmp, "./Y_section_%i_mi.txt",iy);
    PrintMatrixToFile(1+nex,1,mi,tmp);
    sprintf(tmp, "./Y_section_%i_re.txt",iy);
    PrintMatrixToFile(2*nex,1,re,tmp);
    
    free(li);
    free(mi);
    free(re);
}

/*------------------------------------------------------------------------------
     cb_PrintSectionvaluesFiner
------------------------------------------------------------------------------*/

void cb_PrintSectionvaluesFiner(void)
{
    int i,ix,iy;
    int nex,ney;
    int level;
    double w1,w2,w3,w4;
    double *li,*mi,*re;
    char tmp[30];
    int fact;
    
    MGBlock *M;
    MGBlock *P;
    MyMainWindowUI *mainwin;
    SolveBlock *SolBlock;
    
    
    PutGetMainWindow(1, &mainwin);
    PutGetSolveBlock(1,&SolBlock);
    fact=SolBlock->FinerSpMeshFactor;
    
    level = (int)(mainwin->Detail->value());
    //M = &(SolBlock->Meshes[level]);
    M = SolBlock->FinerSplineMesh;
    nex=M->nnx-1;
    ney=M->nny-1;    
    ix=(int)mainwin->x_section->value()*fact;
    iy=(int)mainwin->y_section->value()*fact;
    
    //Schnitt auf x-Achse (in y-Richtung)
    InitVector_double(2*ney,&li);
    InitVector_double(1+ney,&mi);
    InitVector_double(2*ney,&re);
    
    if (ix != 0) //Vektor links
    for(i=0; i<ney; i++)
    {
       li[i*2]=M->Post2[(ix-1)+i*nex];
       li[i*2+1]=M->Post3[(ix-1)+i*nex];
    }
    
    if (ix != nex) //Vektor rechts
    for(i=0; i<ney; i++)
    {
       re[i*2]=M->Post1[ix+i*nex];
       re[i*2+1]=M->Post4[ix+i*nex];
    }
    
    for(i=0; i<ney+1; i++) //Berechnung Vektor mitte
    {
       if (i != 0) w1=li[i*2-1]; else w1=li[i*2];
       if (i != 0) w2=re[i*2-1]; else w2=re[i*2];
       if (i != ney) w3=li[i*2]; else w3=li[i*2-1];
       if (i != ney) w4=re[i*2]; else w4=re[i*2-1];
       if (ix > 0 && ix < nex) mi[i]=(w1+w2+w3+w4)*0.25;
       else mi[i]=(w1+w2+w3+w4)*0.5;
    }
    
    //Ausgabe
    sprintf(tmp, "./X_section_%i_li.txt",ix);
    PrintMatrixToFile(2*ney,1,li,tmp);
    sprintf(tmp, "./X_section_%i_mi.txt",ix);
    PrintMatrixToFile(1+ney,1,mi,tmp);
    sprintf(tmp, "./X_section_%i_re.txt",ix);
    PrintMatrixToFile(2*ney,1,re,tmp);
    
    free(li);
    free(mi);
    free(re);
    
    //Schnitt auf y-Achse (in x-Richtung)
    InitVector_double(2*nex,&li);
    InitVector_double(1+nex,&mi);
    InitVector_double(2*nex,&re);
    
    if (iy != ney) //Vektor links
    for(i=0; i<nex; i++)
    {
       li[i*2]=M->Post1[i+iy*nex];
       li[i*2+1]=M->Post2[i+iy*nex];
    }
    
    if (iy != 0) //Vektor rechts
    for(i=0; i<nex; i++)
    {
       re[i*2]=M->Post4[i+(iy-1)*nex];
       re[i*2+1]=M->Post3[i+(iy-1)*nex];
    }
    
    for(i=0; i<nex+1; i++) //Berechnung Vektor mitte
    {
       if (i != 0) w1=li[i*2-1]; else w1=li[i*2];
       if (i != 0) w2=re[i*2-1]; else w2=re[i*2];
       if (i != nex) w3=li[i*2]; else w3=li[i*2-1];
       if (i != nex) w4=re[i*2]; else w4=re[i*2-1];
       if (iy > 0 && iy < ney) mi[i]=(w1+w2+w3+w4)*0.25;
       else mi[i]=(w1+w2+w3+w4)*0.5;
    }
    
    //Ausgabe
    sprintf(tmp, "./Y_section_%i_li.txt",iy);
    PrintMatrixToFile(2*nex,1,li,tmp);
    sprintf(tmp, "./Y_section_%i_mi.txt",iy);
    PrintMatrixToFile(1+nex,1,mi,tmp);
    sprintf(tmp, "./Y_section_%i_re.txt",iy);
    PrintMatrixToFile(2*nex,1,re,tmp);
    
    free(li);
    free(mi);
    free(re);
    return;
}



/*------------------------------------------------------------------------------
     PostHomogen1
------------------------------------------------------------------------------*/
int PostHomogen1(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Sxx1, Syy1, Sxx2, Syy2, Sxx, Syy;
    double EHomogen, muHomogen;
    double H,W;
    double eps0;
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
        
    PostRestoringForce2(M->nnx, M->nny, M->MatFieldE,               
                        M->MatFieldInt, SolBlock->Material,
                        M->Ux,  M->Uy,
			            M->SwapX, M->SwapY);            
    //PostRestoringForce(M->nnx, M->nny, SolBlock->K_matrix, M->MatFieldE, 
    //                   M->Ux, M->Uy, M->SwapX, M->SwapY);
   
    Syy1=0; Syy2=0;
    Sxx1=0; Sxx2=0;
    
    for(i=0; i<M->nnx; i++)
    {
        Syy1+=M->SwapY[i+0*M->nnx];
        Syy2+=M->SwapY[i+(M->nny-1)*M->nnx];
    }
    
    for(j=0; j<M->nny; j++)
    {
        Sxx1+=M->SwapX[0+j*M->nnx];
        Sxx2+=M->SwapX[(M->nnx-1)+j*M->nnx];
    }
    
    Sxx=Sxx2;///(M->lb*M->ney);
    Syy=Syy2;///(M->la*M->nex);
    
    W=M->nex*M->la;
    H=M->ney*M->lb;
    
    eps0=(M->Ux[(M->nnx-1)])/(M->nex*M->la);// printf("eps0: %g",eps0);
    
    /* Plane stress */
    EHomogen=Sxx/H*(1-(Syy*H/Sxx/W)*(Syy*H/Sxx/W))/eps0;
    muHomogen=Syy*H/Sxx/W; 
    
    // Plane strain 
    //muHomogen=Syy/(Sxx+Syy); 
    //EHomogen=(Sxx*Sxx+Sxx*Syy-2*Syy*Syy)/(Sxx+Syy);        
 
    
    FILE    *outp ;
    char filename[30];
    
    strcpy(filename,"MC.dat");
    outp = fopen(filename,"a");
    
    printf("\nSxx1=%18.12g   Sxx2=%18.12g\n", Sxx1, Sxx2);
    printf("\nSyy1=%18.12g   Syy2=%18.12g\n", Syy1, Syy2);
    printf("E(homogen)=%18.12g\n", EHomogen);
    printf("mu(homogen)=%18.12g\n", muHomogen);
  	
    if (outp == NULL)  
   	{   
  		printf("Error: cannot open the file ");
   	    //getchar();
   	}
    else 
	{   
	    
   		fprintf(outp,"%18.12g ; %18.12g   \n", Sxx, Syy);	
        fclose(outp);
	}   
           
                   
    return(1);
}

/*------------------------------------------------------------------------------
     PostHomogenY
------------------------------------------------------------------------------*/
int PostHomogenY(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Sxx1, Syy1, Sxx2, Syy2, Sxx, Syy;
    double EHomogen, muHomogen;
    double H,W;
    double eps0;
    double tmp;
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
        
    PostRestoringForce2(M->nnx, M->nny, M->MatFieldE,               
                        M->MatFieldInt, SolBlock->Material,
                        M->Ux,  M->Uy,
			            M->SwapX, M->SwapY);            
    //PostRestoringForce(M->nnx, M->nny, SolBlock->K_matrix, M->MatFieldE, 
    //                   M->Ux, M->Uy, M->SwapX, M->SwapY);
   
    Syy1=0; Syy2=0;
    Sxx1=0; Sxx2=0;
    
    for(i=0; i<M->nnx; i++)
    {
        Syy1+=M->SwapY[i+0*M->nnx];
        Syy2+=M->SwapY[i+(M->nny-1)*M->nnx];
    }
    
    for(j=0; j<M->nny; j++)
    {
        Sxx1+=M->SwapX[0+j*M->nnx];
        Sxx2+=M->SwapX[(M->nnx-1)+j*M->nnx];
    }
    
    Sxx=Sxx2;///(M->lb*M->ney);
    Syy=Syy2;///(M->la*M->nex);
    
    //Swap for y-direction
    tmp=Sxx; Sxx=Syy; Syy=tmp;
    
    W=M->nex*M->la;
    H=M->ney*M->lb;
    
    //Swap for y-direction
    tmp=W; W=H; H=tmp;
    
    eps0=(M->Uy[(M->nny-1)*M->nnx])/(M->ney*M->lb);// printf("eps0: %g",eps0);
    
    /* Plane stress */
    EHomogen=Sxx/H*(1-(Syy*H/Sxx/W)*(Syy*H/Sxx/W))/eps0;
    muHomogen=Syy*H/Sxx/W; 
    
    // Plane strain 
    //muHomogen=Syy/(Sxx+Syy); 
    //EHomogen=(Sxx*Sxx+Sxx*Syy-2*Syy*Syy)/(Sxx+Syy);        
 
    
    printf("\nSxx1=%18.12g   Sxx2=%18.12g\n", Sxx1, Sxx2);
    printf("\nSyy1=%18.12g   Syy2=%18.12g\n", Syy1, Syy2);
    printf("E(homogen)=%18.12g\n", EHomogen);
    printf("mu(homogen)=%18.12g\n", muHomogen);
  	    
    return(1);
}


/*------------------------------------------------------------------------------
     PostSystemEnergy
------------------------------------------------------------------------------*/

double PostSystemEnergy(void)
{
    int     i,j;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
    PostRestoringForce2(M->nnx, M->nny, M->MatFieldE,               
                        M->MatFieldInt, SolBlock->Material,
                        M->Ux,  M->Uy,
			            M->SwapX, M->SwapY);      
                
    //PostRestoringForce(M->nnx, M->nny, SolBlock->K_matrix, M->MatFieldE, 
    //                   M->Ux, M->Uy, M->SwapX, M->SwapY);
    Energy=0.0;
    for(i=0; i<M->nnx; i++)
    {
        for(j=0; j<M->nny; j++)
        {
            if (1==(M->BcTypeX[i+j*M->nnx])) Energy+=((M->Ux[i+j*M->nnx])*(M->SwapX[i+j*M->nnx]));
            if (1==(M->BcTypeY[i+j*M->nnx])) Energy+=((M->Uy[i+j*M->nnx])*(M->SwapY[i+j*M->nnx]));
            if (0==(M->BcTypeX[i+j*M->nnx])) Energy+=((M->Ux[i+j*M->nnx])*(M->BcValueX[i+j*M->nnx]));
            if (0==(M->BcTypeY[i+j*M->nnx])) Energy+=((M->Uy[i+j*M->nnx])*(M->BcValueY[i+j*M->nnx]));
        }
    }
    //printf("\nEnergy: %g\n", Energy);
    
    return(Energy);
}



/*------------------------------------------------------------------------------
     PostErrorEnergy
------------------------------------------------------------------------------*/
double PostErrorEnergy(void)
{
    int     i,j,k;
    int     num_levels;
    MGBlock *Meshes, *M;
    SolveBlock *SolBlock;
    double Energy;
    
    
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    num_levels=SolBlock->num_levels;
    M=&(Meshes[num_levels-1]);
    
        
    PostRestoringForce2(M->nnx, M->nny, M->MatFieldE,               
                        M->MatFieldInt, SolBlock->Material,
                        M->Ux,  M->Uy,
			            M->SwapX, M->SwapY);              
    //PostRestoringForce(M->nnx, M->nny, SolBlock->K_matrix, M->MatFieldE, 
    //                   M->Ux, M->Uy, M->SwapX, M->SwapY);
    Energy=0;
    for(j=0; j<M->nny; j++)
    {
        for(i=0; i<M->nnx; i++)
        {
            if (0==(M->BcTypeX[i+j*M->nnx])) Energy+=fabs((M->Ux[i+j*M->nnx])*
                                             ( (M->SwapX[i+j*M->nnx])-(M->BcValueX[i+j*M->nnx]) ) );
            if (0==(M->BcTypeY[i+j*M->nnx])) Energy+=fabs((M->Uy[i+j*M->nnx])*
                                             ( (M->SwapY[i+j*M->nnx])-(M->BcValueY[i+j*M->nnx]) ) );
        }
    }
    //printf("\nEnergy: %g\n", Energy);
    
    return(Energy);
}


/*------------------------------------------------------------------------------
	PostRestoringForce
--------------------------------------------------------------------------------*/ 

int PostRestoringForce(int nnx, int nny, double *K, double *MatFieldE, 
                       double *Ux,  double *Uy,
			           double *MFx, double *MFy)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    
 
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; 
	         p--;               E2= MatFieldE[p]; 
	         p-=nex;            E3= MatFieldE[p]; 
	         p++;               E4= MatFieldE[p];
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
          
         Fx = E1*(K[0] *U9+K[1] *U10+K[2] *U11+K[3] *U12+K[4] *U17+K[5] *U18+K[6] *U15+K[7]*U16) 
             +E2*(K[16]*U7+K[17]*U8 +K[18]*U9 +K[19]*U10+K[20]*U15+K[21]*U16+K[22]*U13+K[23]*U14)
             +E3*(K[32]*U1+K[33]*U2 +K[34]*U3 +K[35]*U4 +K[36]*U9 +K[37]*U10+K[38]*U7 +K[39]*U8)
             +E4*(K[48]*U3+K[49]*U4 +K[50]*U5 +K[51]*U6 +K[52]*U11+K[53]*U12+K[54]*U9 +K[55]*U10);
         MFx[p]=Fx;
       
   
         Fy = E1*(K[8] *U9+K[9] *U10+K[10]*U11+K[11]*U12+K[12]*U17+K[13]*U18+K[14]*U15+K[15]*U16) 
		     +E2*(K[24]*U7+K[25]*U8 +K[26]*U9 +K[27]*U10+K[28]*U15+K[29]*U16+K[30]*U13+K[31]*U14)
		     +E3*(K[40]*U1+K[41]*U2 +K[42]*U3 +K[43]*U4 +K[44]*U9 +K[45]*U10+K[46]*U7 +K[47]*U8)
		     +E4*(K[56]*U3+K[57]*U4 +K[58]*U5 +K[59]*U6 +K[60]*U11+K[61]*U12+K[62]*U9 +K[63]*U10); 	 
		 MFy[p]=Fy; 
        }
    }
    return(1);
}

/*------------------------------------------------------------------------------
	PostRestoringForce2
--------------------------------------------------------------------------------*/ 

int PostRestoringForce2(int nnx, int nny, double *MatFieldE,               
                        int *MatFieldInt, MaterialBlock *Mat,
                       double *Ux,  double *Uy,
			           double *MFx, double *MFy)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy;
    double *K1, *K2, *K3, *K4, *K;
    
    K1=Mat[0].K; K2=K1; K3=K1; K4=K1;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0;   
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p]; K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p]; K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p]; K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p]; K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         MFx[p]=Fx;
   
   
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
         MFy[p]=Fy;
 
      
         }
    }
   
    return(1);
}
/*------------------------------------------------------------------------------
	PostRestoringForce3d
--------------------------------------------------------------------------------*/ 
int PostRestoringForce3d(
              int nnx, int nny, int nnz, MaterialBlock *Material,
              double *MatFieldE, int *MatFieldInt, 
        	  double *Ux,  double *Uy, double *Uz,
			  double *Fx,  double *Fy, double *Fz)
{
    int i,j,k,m;
    int nex, ney, nez, nns;
    double E;
    double U[24],F[24];
    double *K;
    int p;
    
    nex=nnx-1;
    ney=nny-1;
    nez=nnz-1;
    
    SetVectorToZero_double((nnx*nny*nnz), Fx);
    SetVectorToZero_double((nnx*nny*nnz), Fy);
    SetVectorToZero_double((nnx*nny*nnz), Fz);
    
    for (k = 0; k < nez; k++) 
    for (j = 0; j < ney; j++) 
    {     
        for (i = 0; i < nex; i++)   
        {
            p=i+nex*j+nex*ney*k;
            K=Material[MatFieldInt[p]].K;
            
            p=i+nnx*j+nnx*nny*(k);
            U[0]=Ux[p];
            U[1]=Uy[p];
            U[2]=Uz[p];
            
            p++;
            U[3]=Ux[p];
            U[4]=Uy[p];
            U[5]=Uz[p];
            
            p+=nnx;
            U[6]=Ux[p];
            U[7]=Uy[p];
            U[8]=Uz[p];
            
            p--;
            U[9]=Ux[p];
            U[10]=Uy[p];
            U[11]=Uz[p];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            U[12]=Ux[p];
            U[13]=Uy[p];
            U[14]=Uz[p];
            
            p++;
            U[15]=Ux[p];
            U[16]=Uy[p];
            U[17]=Uz[p];
            
            p+=nnx;
            U[18]=Ux[p];
            U[19]=Uy[p];
            U[20]=Uz[p];
            
            p--;
            U[21]=Ux[p];
            U[22]=Uy[p];
            U[23]=Uz[p];
            
            MatrixMultMatrix(24,24,1,K,U,F);
            E= MatFieldE[i + j*nex + k*nex*ney];
            MatrixMultScalar(24,1, F, E);
          
            p=i+nnx*j+nnx*nny*(k);
            Fx[p]+=F[0];
            Fy[p]+=F[1];
            Fz[p]+=F[2];
            
            p++;
            Fx[p]+=F[3];
            Fy[p]+=F[4];
            Fz[p]+=F[5];
            
            p+=nnx;
            Fx[p]+=F[6];
            Fy[p]+=F[7];
            Fz[p]+=F[8];
            
            p--;
            Fx[p]+=F[9];
            Fy[p]+=F[10];
            Fz[p]+=F[11];
            
            p=i+nnx*j+nnx*nny*(k+1);
            
            Fx[p]+=F[12];
            Fy[p]+=F[13];
            Fz[p]+=F[14];
            
            p++;
            Fx[p]+=F[15];
            Fy[p]+=F[16];
            Fz[p]+=F[17];
            
            p+=nnx;
            Fx[p]+=F[18];
            Fy[p]+=F[19];
            Fz[p]+=F[20];
            
            p--;
            Fx[p]+=F[21];
            Fy[p]+=F[22];
            Fz[p]+=F[23];
            
           // for(m=0;m<24;m++)
           // if (0!=F[m]) printf("f[m]: %f \n",F[m]);  
           // for(m=0;m<24;m++)
           // if (0!=U[m]) printf("U[m]: %f \n",U[m]);     

         }
    }
    
    return(1);
}

/*------------------------------------------------------------------------------
	TransformDetail2d
--------------------------------------------------------------------------------*/

void TransformDetail2and3d(void)
{
    int     i,j,k;
    int     show_level, detail_level, diff_level;
    MGBlock *Meshes, *M, *P;
    SolveBlock *SolBlock;
    GraphicBlock *graphblock;
    int Pnex, Pney, Pnez, Pxy, Mnex, Mney, Mnez, Mxy, fact; 
    int i1,i2,j1,j2,k1,k2,p;
    int ModelDim;
    
    ModelDim=PutGetModelDim(1,123);
    
    PutGetGraphicBlock(1, &graphblock); 
    PutGetSolveBlock(1,&SolBlock); 
    Meshes=SolBlock->Meshes;
    show_level=graphblock->show_level;
    detail_level=graphblock->detail_level;
    diff_level=show_level-detail_level;
    M=&(Meshes[show_level]); 
    P=&(Meshes[detail_level]); 
    
    Pnex=P->nex;
    Pney=P->ney;
    Mnex=M->nex;
    Mney=M->ney;
        
    fact=2;
    for (i=0; i<(diff_level-1); i++) fact*=2;


if (2==ModelDim)
{    
    for (i=0; i<Pnex; i++)
    for (j=0; j<Pney; j++)
    {
        i1=i*fact;
        j1=j*fact;
        i2=(i+1)*fact-1;
        j2=(j+1)*fact-1;
        p=i+j*Pnex;
        P->Post1[p]=M->Post1[i1+j1*Mnex];
        P->Post2[p]=M->Post2[i2+j1*Mnex];
        P->Post3[p]=M->Post3[i2+j2*Mnex];
        P->Post4[p]=M->Post4[i1+j2*Mnex];
    } 
}
else
{
    Pnez=P->nez;
    Pxy=(P->nex)*(P->ney);
    Mxy=(M->nex)*(M->ney);
    
    for (i=0; i<Pnex; i++)
    for (j=0; j<Pney; j++)
    for (k=0; k<Pnez; k++)
    {
        i1=i*fact;
        j1=j*fact;
        k1=k*fact;
        i2=(i+1)*fact-1;
        j2=(j+1)*fact-1;
        k2=(k+1)*fact-1;
        
        p=i+j*Pnex+k*Pxy;
        P->Post1[p]=M->Post1[i1+j1*Mnex+k1*Mxy];
        P->Post2[p]=M->Post2[i2+j1*Mnex+k1*Mxy];
        P->Post3[p]=M->Post3[i2+j2*Mnex+k1*Mxy];
        P->Post4[p]=M->Post4[i1+j2*Mnex+k1*Mxy];
        P->Post5[p]=M->Post5[i1+j1*Mnex+k2*Mxy];
        P->Post6[p]=M->Post6[i2+j1*Mnex+k2*Mxy];
        P->Post7[p]=M->Post7[i2+j2*Mnex+k2*Mxy];
        P->Post8[p]=M->Post8[i1+j2*Mnex+k2*Mxy];
    } 

}    


return;
}


