/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner (Email: info@mulgrido.de)
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/*------------------------------------------------------------------------------
	Smooth_GS
--------------------------------------------------------------------------------*/ 

int Smooth_GS(
              int nnx, int nny, 
              MaterialBlock *Mat,
              double *MatFieldE, 
              int *MatFieldInt,
              double *Ux,  double *Uy,
			  double lambda,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy, dUx, dUy;
    double *K,*K1, *K2, *K3, *K4 ;
    double Nenner;
    double K11,K12,K22;
    
    K=Mat->K;
    K1=Mat->K;
    K2=Mat->K;
    K3=Mat->K;
    K4=Mat->K;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0; 
    
    //Forward Sweep
      
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p];  K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p];  K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p];  K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p];  K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         Fx-=BcValueX[p]; 
         
         dUx=(-Fx/(E1*K1[0]+E2*K2[0]+E3*K3[0]+E4*K4[0]));
	     Ux[p]+=lambda*dUx; 
          
		 } 
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 Fy-=BcValueY[p]; 
  
         dUy=(-Fy/(E1*K1[9]+E2*K2[9]+E3*K3[9]+E4*K4[9]));
	     Uy[p]+=lambda*dUy;
          
         }
      
         }
    }
    

   
    return(1);
}

/*------------------------------------------------------------------------------
	Smooth_GS_block
--------------------------------------------------------------------------------*/ 

int Smooth_GS_block(
              int nnx, int nny, 
              MaterialBlock *Mat,
              double *MatFieldE, 
              int *MatFieldInt,
              double *Ux,  double *Uy,
			  double lambda,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy, dUx, dUy;
    double *K,*K1, *K2, *K3, *K4 ;
    double Nenner;
    double K11,K12,K22;
    
    K=Mat->K;
    K1=Mat->K;
    K2=Mat->K;
    K3=Mat->K;
    K4=Mat->K;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0; 
    
    //Forward Sweep
      
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p];  K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p];  K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p];  K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p];  K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         Fx-=BcValueX[p]; 
         
         if (0!=BcTypeY[p])
           {
              dUx=(-Fx/(E1*K1[0]+E2*K2[0]+E3*K3[0]+E4*K4[0]));
		      Ux[p]+=lambda*dUx; 
           }
		 } 
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 Fy-=BcValueY[p]; 
  
         if (0!=BcTypeX[p])
           {
	          dUy=(-Fy/(E1*K1[9]+E2*K2[9]+E3*K3[9]+E4*K4[9]));
	          Uy[p]+=lambda*dUy;
           }
         }
         
         if (0==BcTypeX[p] && 0==BcTypeY[p])
         {
             K11=E1*K1[0]+E2*K2[0]+E3*K3[0]+E4*K4[0];
             K22=E1*K1[9]+E2*K2[9]+E3*K3[9]+E4*K4[9];
             K12=E1*K1[1]-E2*K2[1]+E3*K3[1]-E4*K4[1];
             Nenner=K11*K22-K12*K12;
             dUx=-( K22*Fx-K12*Fy)/Nenner;
             dUy=-(-K12*Fx+K11*Fy)/Nenner;
             Ux[p]+=lambda*dUx; 
             Uy[p]+=lambda*dUy; 
         
         }
      
         }
    }
    

   
    return(1);
}

/*------------------------------------------------------------------------------
	Smooth_SSOR
--------------------------------------------------------------------------------*/ 

int Smooth_SSOR(
              int nnx, int nny, 
              MaterialBlock *Mat,
              double *MatFieldE, 
              int *MatFieldInt,
              double *Ux,  double *Uy,
			  double lambda,
              int *BcTypeX, int *BcTypeY,
              double *BcValueX, double *BcValueY)
{
    register int i,j;
    int istart, jstart, iend, jend;
    register int p;
    int nex, ney;
    double E1, E2, E3, E4, Esum;
    double U1, U2, U3, U4, U5, U6, U7, U8, U9, U10, U11, U12, U13, U14, U15, U16, U17, U18;
    double a11, a12, a22, b9,b10;
    double I11, I12, I22;
    double det;
    double Fx, Fy, dUx, dUy;
    double *K,*K1, *K2, *K3, *K4 ;
    
    K=Mat->K;
    K1=Mat->K;
    K2=Mat->K;
    K3=Mat->K;
    K4=Mat->K;
    
    nex=nnx-1;
    ney=nny-1;
    
    istart=1;
    jstart=1;
    iend=nnx-1;
    jend=nny-1;
     
    U1=0.0;U2=0.0;U3=0.0;U4=0.0;U5=0.0;U6=0.0;U7=0.0;U8=0.0;U9=0.0;U10=0.0;
    U11=0.0;U12=0.0;U13=0.0;U14=0.0;U15=0.0;U16=0.0;U17=0.0;U18=0.0; 
    
    //Forward Sweep
      
    for (j = 0; j < nny; j++) 
    {     
        for (i = 0; i < nnx; i++)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p];  K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p];  K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p];  K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p];  K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         Fx-=BcValueX[p]; 
         
         
         dUx=(-Fx/(E1*K1[0]+E2*K2[0]+E3*K3[0]+E4*K4[0]));
		 Ux[p]+=lambda*dUx; 
		 U9=Ux[p];
		 } 
   
   
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 Fy-=BcValueY[p]; 
  
  
	     dUy=(-Fy/(E1*K1[9]+E2*K2[9]+E3*K3[9]+E4*K4[9]));
	     Uy[p]+=lambda*dUy;
	     }
       
      
         }
    }
    
    //Backward Sweep
    
    for (j = nny-1; j >= 0; j--) 
    {     
        for (i = nnx-1; i >= 0; i--)   
        {
         
         if(0==i || 0==j || i==nex || j==ney)
         {
             if(i!=0 && j!=0)      {p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];}
	         if(j!=0)              {p = (i-1) + (j-1)*nnx+1;  U3 = Ux[p];   U4 = Uy[p];}
	         if(i!=nex && j!=0)    {p = (i-1) + (j-1)*nnx+2;  U5 = Ux[p];   U6 = Uy[p];}
         
             if(i!=0)              {p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];}
	                                p = (i-1) + j*nnx+1;      U9  = Ux[p];  U10 = Uy[p]; 
	         if(i!=nex)            {p = (i-1) + j*nnx+2;      U11 = Ux[p];  U12 = Uy[p];}
         
             if(i!=0 && j!=ney)    {p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];}
	         if(j!=ney)            {p = (i-1) + (j+1)*nnx+1;  U15 = Ux[p];  U16 = Uy[p];}
             if(i!=nex && j!=ney)  {p = (i-1) + (j+1)*nnx+2;  U17 = Ux[p];  U18 = Uy[p];}
	         
	         
             if(i!=nex && j!=ney)    {p= i + j*nex;      E1= MatFieldE[p];K1=Mat[MatFieldInt[p]].K;} else E1=0.0; 
	         if(i!=0   && j!=ney)    {p= i + j*nex-1;    E2= MatFieldE[p];K2=Mat[MatFieldInt[p]].K;} else E2=0.0; 
	         if(i!=0   && j!=0)      {p= i + j*nex-1-nex;E3= MatFieldE[p];K3=Mat[MatFieldInt[p]].K;} else E3=0.0; 
	         if(i!=nex && j!=0)      {p= i + j*nex-nex;  E4= MatFieldE[p];K4=Mat[MatFieldInt[p]].K;} else E4=0.0;
	     }
	     else
	     {
	         p = (i-1) + (j-1)*nnx;    U1 = Ux[p];   U2 = Uy[p];
	         p++;                      U3 = Ux[p];   U4 = Uy[p];
	         p++;                      U5 = Ux[p];   U6 = Uy[p];
         
             p = (i-1) + j*nnx;        U7  = Ux[p];  U8  = Uy[p];
             p++;                      U9  = Ux[p];  U10 = Uy[p]; 
	         p++;                      U11 = Ux[p];  U12 = Uy[p];
         
             p = (i-1) + (j+1)*nnx;    U13 = Ux[p];  U14 = Uy[p];
	         p++;                      U15 = Ux[p];  U16 = Uy[p];
             p++;                      U17 = Ux[p];  U18 = Uy[p];
	     
	         
             p= i + j*nex;      E1= MatFieldE[p];  K1=Mat[MatFieldInt[p]].K;
	         p--;               E2= MatFieldE[p];  K2=Mat[MatFieldInt[p]].K;
	         p-=nex;            E3= MatFieldE[p];  K3=Mat[MatFieldInt[p]].K;
	         p++;               E4= MatFieldE[p];  K4=Mat[MatFieldInt[p]].K;
	     }
	     
	     Esum=E1+E2+E3+E4;
	     
	     /*------------------------------------------------------------------------------ 
	     Element order: 1 : upper right; 2 : upper left; 3 : lower left ; 4 : lower right 
         -------------------------------------------------------------------------------*/
         p = i + j*nnx;  
         
         if (0==BcTypeY[p])
         {
         Fy = E1*(K1[8] *U9+K1[9] *U10+K1[10]*U11+K1[11]*U12+K1[12]*U17+K1[13]*U18+K1[14]*U15+K1[15]*U16) 
		     +E2*(K2[24]*U7+K2[25]*U8 +K2[26]*U9 +K2[27]*U10+K2[28]*U15+K2[29]*U16+K2[30]*U13+K2[31]*U14)
		     +E3*(K3[40]*U1+K3[41]*U2 +K3[42]*U3 +K3[43]*U4 +K3[44]*U9 +K3[45]*U10+K3[46]*U7 +K3[47]*U8)
		     +E4*(K4[56]*U3+K4[57]*U4 +K4[58]*U5 +K4[59]*U6 +K4[60]*U11+K4[61]*U12+K4[62]*U9 +K4[63]*U10); 	 
		 Fy-=BcValueY[p]; 
  
  
	     dUy=(-Fy/(E1*K1[9]+E2*K2[9]+E3*K3[9]+E4*K4[9]));
	     Uy[p]+=lambda*dUy;
	     U10=Uy[p];
	     }
         
         if (0==BcTypeX[p])
         {
         Fx = E1*(K1[0] *U9+K1[1] *U10+K1[2] *U11+K1[3] *U12+K1[4] *U17+K1[5] *U18+K1[6] *U15+K1[7]*U16) 
             +E2*(K2[16]*U7+K2[17]*U8 +K2[18]*U9 +K2[19]*U10+K2[20]*U15+K2[21]*U16+K2[22]*U13+K2[23]*U14)
             +E3*(K3[32]*U1+K3[33]*U2 +K3[34]*U3 +K3[35]*U4 +K3[36]*U9 +K3[37]*U10+K3[38]*U7 +K3[39]*U8)
             +E4*(K4[48]*U3+K4[49]*U4 +K4[50]*U5 +K4[51]*U6 +K4[52]*U11+K4[53]*U12+K4[54]*U9 +K4[55]*U10);
         Fx-=BcValueX[p]; 
         
         
         dUx=(-Fx/(E1*K1[0]+E2*K2[0]+E3*K3[0]+E4*K4[0]));
		 Ux[p]+=lambda*dUx; 
		 
		 } 
         }
    }
   
    return(1);
}
