/*
-------------------------------------------------------------------------
Mulgrido, a grid-based finite element program. 
Copyright (C) 2008 Stefan Hfner
Coding Support: Marco Kessel and David Schneider
Supervision: Carsten Knke, Professor for Statics of Structures, 
Institute of Structural Mechanics, Bauhaus-University Weimar.

This file is part of Mulgrido.

    Mulgrido is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Mulgrido is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Mulgrido.  If not, see <http://www.gnu.org/licenses/>.
-------------------------------------------------------------------------
*/
/* 
LINUX VERSION ONLY WITH #include <GL/glut.h> !!!
Windows VERSION AT HOME ONLY WITH #include <FL/glut.h> !!!
*/


#include <FL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <FL/Fl_Gl_Window.H>

#include "MyMainWindowUI.h"
#include "MyDataStructures.h"
#include "MyDeclarations.h"

#include <math.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#define frand() ((double) rand()/(RAND_MAX+1.0))

/*------------------------------------------------------------------------------
     MySetViewport
------------------------------------------------------------------------------*/
void MySetViewport(int w, int h) 
{
    GraphicBlock *localblock;
   
    PutGetGraphicBlock(1, &localblock);
    localblock->WinPixelWidth=(int) w;
    localblock->WinPixelHeight=(int) h;
    
    glViewport(0, 0, w, h);  // (w,h) Pixels of OpenGL Window
        
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();  
    
}

/*------------------------------------------------------------------------------
     MyGLContext
------------------------------------------------------------------------------*/
void MyGlContext() 
{
   GLfloat no_mat[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat mat_ambient[] = { 0.7, 0.7, 0.7, 1.0 };
   GLfloat mat_ambient_color[] = { 0.8, 0.8, 0.2, 1.0 };
   GLfloat mat_diffuse[] = { 0.1, 0.5, 0.8, 1.0 };
   GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
   GLfloat no_shininess[] = { 0.0 };
   GLfloat low_shininess[] = { 5.0 };
   GLfloat high_shininess[] = { 100.0 };
   GLfloat mat_emission[] = {0.3, 0.2, 0.2, 0.0};
   
       //GLfloat mat_specular[] = {0.3,0.4, 0.3, 1.0};
    GLfloat mat_shininess[] = {3.0};
    //GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat white_light[] = {0.6,0.6,0.6,1.0};
    GLfloat lmodel_ambient[] = {0.1,0.1,0.1,1.0};
    //GLfloat mat_emission[] = {0.3, 0.2, 0.2, 0.0};
    
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, no_mat);
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, mat_diffuse);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, high_shininess);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, no_mat);
    
    glEnable(GL_NORMALIZE);
    glClearColor(1.0,1.0,1.0,0);    
    glShadeModel(GL_SMOOTH);  
    
       
}

/*------------------------------------------------------------------------------
     DrawStructure
------------------------------------------------------------------------------*/
void DrawStructure() 
{
    GraphicBlock *localblock;
    PutGetGraphicBlock(1, &localblock);
    SolveBlock *SolBlock;
    int num_levels; 
    int show_level;
    int ModelDim;
    int i;
    
    PutGetSolveBlock(1,&SolBlock); 
    num_levels=SolBlock->num_levels;
    show_level=localblock->show_level;
    // Actual Level becomes show_level! Dirty Hack!!!
    //SolBlock->num_levels=show_level;
    
    ModelDim=PutGetModelDim(1,123);
    
    if(0==strcmp("Nothing",localblock->SelectObject)) 
    {
        DrawNothing(); 
        return;
    }
    else
    {
        if (1==localblock->NewObjectFlag) 
        {
            PostMainLoop();
            localblock->NewObjectFlag=0;
        }
        
        if (localblock->detail_level!=localblock->show_level)
        {TransformDetail2and3d();}
        
                        
        if (2==ModelDim) DrawMesoStructureDeformed();
        if (3==ModelDim && localblock->RotateFlag==0) DrawMesoStructure3d();
        if (3==ModelDim && localblock->RotateFlag==1) DrawMesoStructureDeformed();
        
      
    }
    

   // DrawScale();
    return;
}

/*------------------------------------------------------------------------------
     DrawNothing
------------------------------------------------------------------------------*/
void DrawNothing() 
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();   
    gluOrtho2D(0,100,0,100);
    
    glClear(GL_COLOR_BUFFER_BIT); 
    
return;
}


/*------------------------------------------------------------------------------
     DrawMesoStructureDeformed
------------------------------------------------------------------------------*/
void DrawMesoStructureDeformed() 
{
    int i,j,k,p;
    float stpx,stpy; 
    float x,y;

    float zoomvar;
    float rotx;
    float rotz;
    double la,lb;
    int nx,ny,nxv,nyv;
    double w,h;
    double ratio_pixels, ratio_real;
    GLdouble left, right, bottom, top;
    double width, height;
    double transX, transY;
    int optimStep;
    double px,pex,pey;
    int OptimStepX,OptimStepY;
    int istart,iend;
    int jstart,jend;
    SolveBlock *SolBlock;
    int num_levels;
    int show_level; 
    double *Post1;
    double *Post2;
    double *Post3;
    double *Post4;
    double minimum, maximum;
    double minimum2, maximum2;
    double val;
    int col;
    float *r, *g, *b;
    int NumColor;
    float df;
    int act;
    GraphicBlock *graphblock;
    MyMainWindowUI   *mainwin;
    double *Ux, *Uy;
    int nex, ney;
    double zdepth;
    double ExtrudeZ, Extrude;
    int postflag;  
    float LoadScaleU;
    float LoadScaleF;
    float LoadLen;
    float Umax,Umin,Urange,Fmax,Fmin,Frange;
    int *BcTypeX, *BcTypeY;
    double *BcValueX, *BcValueY;
    
    int NumIsoLines;
    double *IsoValues;
    int MinIso, MaxIso;
    MaterialBlock *Material;
    int *MatFieldInt;
    
    GLUquadricObj *qobj;
    
    //GLfloat mat_specular[] = {0.3,0.4, 0.3, 1.0};
    GLfloat mat_shininess[] = {3.0};
    //GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat white_light[] = {0.6,0.6,0.6,1.0};
    GLfloat lmodel_ambient[] = {0.1,0.1,0.1,1.0};
    //GLfloat mat_emission[] = {0.3, 0.2, 0.2, 0.0};
    
    
    GLfloat v1[3];
    GLfloat v2[3];
    GLfloat v3[3];
    GLfloat v4[3];    
    GLfloat vn[3];
    GLfloat vcol[3];
    
    GLfloat v1iso[3];
    GLfloat v2iso[3];
    GLfloat v3iso[3];
    GLfloat v4iso[3]; 
    
    GLfloat vp1[3];
    GLfloat vp2[3];
    
    /*----------------- for bitmap -----*/    
    GLfloat rastercolor[4];
    GLfloat rasterposition[4];
   
    
    /*--------------------- Get data from SOLVEBLOCK -------------------------*/   
    PutGetSolveBlock(1,&SolBlock);
    PutGetGraphicBlock(1, &graphblock); 
    num_levels=SolBlock->num_levels;
    
    // either show_level=detail_level or detail_level is relevant, thus ...
    show_level=graphblock->detail_level; 
    
    if (SolBlock->FinerSplineFlag == 1)
    {
        la=SolBlock->FinerSplineMesh->la;
        lb=SolBlock->FinerSplineMesh->lb;
        nx=SolBlock->FinerSplineMesh->nnx;
        ny=SolBlock->FinerSplineMesh->nny;
        Ux=SolBlock->FinerSplineMesh->Ux;
        Uy=SolBlock->FinerSplineMesh->Uy;
        
        BcTypeX=SolBlock->FinerSplineMesh->BcTypeX;
        BcTypeY=SolBlock->FinerSplineMesh->BcTypeY;
        BcValueX=SolBlock->FinerSplineMesh->BcValueX;
        BcValueY=SolBlock->FinerSplineMesh->BcValueY;
    
        Post1=SolBlock->FinerSplineMesh->Post1;
        Post2=SolBlock->FinerSplineMesh->Post2;
        Post3=SolBlock->FinerSplineMesh->Post3;
        Post4=SolBlock->FinerSplineMesh->Post4;
        
        MatFieldInt=SolBlock->FinerSplineMesh->MatFieldInt;
        Material=SolBlock->Material;                          
    }
    else
    {
        la=SolBlock->Meshes[show_level].la;
        lb=SolBlock->Meshes[show_level].lb;
        nx=SolBlock->Meshes[show_level].nnx;
        ny=SolBlock->Meshes[show_level].nny;
        Ux=SolBlock->Meshes[show_level].Ux;
        Uy=SolBlock->Meshes[show_level].Uy;
        
        BcTypeX=SolBlock->Meshes[show_level].BcTypeX;
        BcTypeY=SolBlock->Meshes[show_level].BcTypeY;
        BcValueX=SolBlock->Meshes[show_level].BcValueX;
        BcValueY=SolBlock->Meshes[show_level].BcValueY;
    
        Post1=SolBlock->Meshes[show_level].Post1;
        Post2=SolBlock->Meshes[show_level].Post2;
        Post3=SolBlock->Meshes[show_level].Post3;
        Post4=SolBlock->Meshes[show_level].Post4;
        
        MatFieldInt=SolBlock->Meshes[show_level].MatFieldInt;
        Material=SolBlock->Material;
    }
    
    nex=nx-1;
    ney=ny-1;
   
    nxv=nx-1;
    nyv=ny-1;
        
    
    FindMinMaxVector(nxv*nyv, Post1, &minimum, &maximum);
    FindMinMaxVector(nxv*nyv, Post2, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv, Post3, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv, Post4, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
      
    printf("\nMinimum: %.12e  Maximum: %.12e\n",minimum,maximum);
          
    stpx=la;  
    stpy=lb;  

    
    /*--------------------- Get data from GRAPHBLOCK -------------------------*/ 
    PutGetGraphicBlock(1, &graphblock);
    zoomvar=graphblock->Zoomfactor;
    rotx=graphblock->RotateAngleX;
    rotz=graphblock->RotateAngleZ;
    ExtrudeZ=graphblock->ExtrudeZ;
    Extrude=graphblock->Extrude;
    df=graphblock->Deflect;
    NumIsoLines=graphblock->IsoLines;
    
    w=(double)graphblock->WinPixelWidth;
    h=(double)graphblock->WinPixelHeight;
    
    transX=(double)graphblock->transX;
    transY=(double)graphblock->transY;
    
    
    /*------------------------ Write to GRAPHBLOCK ---------------------------*/
    if (minimum==0.0 && maximum==0.0){minimum=-1.0;maximum=1.0;}
    if (0.0!=minimum)
    if ( ((maximum-minimum)/minimum) <  0.000001 &&
         ((maximum-minimum)/minimum) > -0.000001) 
    {
        minimum*=0.99999;
        maximum*=1.00001;
    }

    graphblock->minimum=minimum;
    graphblock->maximum=maximum;
    
    ratio_pixels=(h/w);
    ratio_real=((ny*lb)/(nx*la));
    
    
    /*-------------------Calculate visible real world range-------------------*/
    if (ratio_pixels>ratio_real)
    {
        width =nex*la*1.1;
        height=ratio_pixels*width;
    }
    else
    {
        height=ney*lb*1.1;
        width =height/ratio_pixels;
    }
    
    width/=zoomvar;
    height/=zoomvar;
    
    
    PutGetMainWindow(1, &mainwin); 
    left=(nex*la/2)-width/2+nex*la*transX;     mainwin->MinX->value((float)left);
    right=(nex*la/2)+width/2+nex*la*transX;    mainwin->MaxX->value((float)right);
    
    bottom=(ney*lb/2)-height/2-ney*lb*transY;  mainwin->MinY->value((float)bottom);
    top=(ney*lb/2)+height/2-ney*lb*transY;;    mainwin->MaxY->value((float)top);


    OptimStepX=1;
    OptimStepY=1;
    istart=0;
    jstart=0;
    iend=nx-2;
    jend=ny-2;
    
    
    
    /*------------------------- Get Color Palette ----------------------------*/
    act=graphblock->act_palette;
    r=graphblock->CurrentPalette[act].r;
    g=graphblock->CurrentPalette[act].g;
    b=graphblock->CurrentPalette[act].b;
    NumColor=graphblock->CurrentPalette[act].NumColor;

    /*------------------------- Draw BMP if flag is on -----------------------*/
    
    
    if (0<graphblock->BmpDataFlag)
    {
        WriteBmp(nex, ney, Post1, Post2, Post3, Post4,
                 minimum, maximum, graphblock->CurrentPalette[act]);
        if (1==graphblock->BmpDataFlag) graphblock->BmpDataFlag=0;
    }
    
    
    /*---------------------------- Draw Patches ------------------------------*/
    zdepth=sqrt(width*width+height*height);
   
    
        
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();  
    
    glOrtho(left,right,bottom,top,-zdepth,zdepth);
    glTranslatef( (nex*la/2) , (ney*lb/2) , ((double)0.0) );
    glRotatef(rotx,1.0,0.0,0.0);
    glRotatef(rotz,0.0,0.0,1.0);
    glTranslatef( (-nex*la/2) , (-ney*lb/2) , ((double)0.0) );
  
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
 
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, white_light);
    glLightfv(GL_LIGHT0, GL_SPECULAR, white_light);

    //glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
    //glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_COLOR_MATERIAL);
    glEnable(GL_DEPTH_TEST);
    
    InitVector_double((NumIsoLines+3), &(IsoValues));
    
    if (NumIsoLines>1)
    for(i=0; i<(NumIsoLines+1); i++)           
    {
        IsoValues[i]=(maximum-minimum)/NumIsoLines*i+minimum; 
    }   
    
    
         
        
        for (i=0; i<iend+1; i+=OptimStepX)
        {
            x=la*i;
            for (j=0; j<jend+1; j+=OptimStepY)
            {
               y=lb*j;
            if (1==Material[MatFieldInt[i+nex*j]].Visible)               
            {   
                  
                   //glRectf(x,y,x+stpx,y+stpy);
                   
               val=Post1[i+nxv*j];col=PickColor(NumColor, minimum, maximum, val);    
               v1[0]=(GLfloat)(x+Ux[i+nx*j]*df);
               v1[1]=(GLfloat)(y+Uy[i+nx*j]*df);
               v1[2]=(GLfloat)(val-minimum)/(maximum-minimum)*ExtrudeZ;    
               
               v1iso[2]=(GLfloat) (v1[2]+Extrude);
                       
               val=Post2[i+nxv*j];col=PickColor(NumColor, minimum, maximum, val);    
               v2[0]=(GLfloat)(x+stpx+Ux[i+1+nx*j]*df);
               v2[1]=(GLfloat)(y+Uy[i+1+nx*j]*df);
               v2[2]=(GLfloat)(val-minimum)/(maximum-minimum)*ExtrudeZ;     
               
               v2iso[2]=(GLfloat)(v2[2]+Extrude);
               
               val=Post3[i+nxv*j]; col=PickColor(NumColor, minimum, maximum, val);  
               v3[0]=(GLfloat)(x+stpx+Ux[i+1+nx*(j+1)]*df);
               v3[1]=(GLfloat)(y+stpy+Uy[i+1+nx*(j+1)]*df);
               v3[2]=(GLfloat)(val-minimum)/(maximum-minimum)*ExtrudeZ;      
               
               v3iso[2]=(GLfloat)(v3[2]+Extrude);
                        
               val=Post4[i+nxv*j]; col=PickColor(NumColor, minimum, maximum, val);   
               v4[0]=(GLfloat)(x+Ux[i+nx*(j+1)]*df);
               v4[1]=(GLfloat)(y+stpy+Uy[i+nx*(j+1)]*df);
               v4[2]=(GLfloat)(val-minimum)/(maximum-minimum)*ExtrudeZ;         
               
               v4iso[2]=(GLfloat)(v4[2]+Extrude);
               
               if (1==graphblock->DrawPatchFlag)
               {
               glBegin (GL_POLYGON); 
               val=Post1[i+nxv*j];col=PickColor(NumColor, minimum, maximum, val); 
               vcol[0]=r[col]; vcol[1]=g[col]; vcol[2]=b[col];
               CompNormalVector((float*) v4, (float*) v1, (float*) v2, (float*) vn); 
               glColor3fv (vcol);  
               glNormal3fv (vn);
               glVertex3fv (v1);              
               
               val=Post2[i+nxv*j];col=PickColor(NumColor, minimum, maximum, val); 
               vcol[0]=r[col]; vcol[1]=g[col]; vcol[2]=b[col];
               CompNormalVector((float*) v1, (float*) v2, (float*) v3, (float*) vn);
               glColor3fv (vcol); 
               glNormal3fv (vn);
               glVertex3fv (v2);    
                           
               val=Post3[i+nxv*j]; col=PickColor(NumColor, minimum, maximum, val); 
               vcol[0]=r[col]; vcol[1]=g[col]; vcol[2]=b[col];
               CompNormalVector((float*) v2, (float*) v3, (float*) v4, (float*) vn); 
               glColor3fv (vcol);
               glNormal3fv (vn);
               glVertex3fv (v3);
               
               val=Post4[i+nxv*j]; col=PickColor(NumColor, minimum, maximum, val); 
               vcol[0]=r[col]; vcol[1]=g[col]; vcol[2]=b[col];
               CompNormalVector((float*) v3, (float*) v4, (float*) v1, (float*) vn);
               glColor3fv (vcol);     
               glNormal3fv (vn);
               glVertex3fv (v4); 
               
               val=Post1[i+nxv*j]; col=PickColor(NumColor, minimum, maximum, val); 
               vcol[0]=r[col]; vcol[1]=g[col]; vcol[2]=b[col];
               CompNormalVector((float*) v4, (float*) v1, (float*) v2, (float*) vn); 
               glColor3fv (vcol);   
               glNormal3fv (vn);
               glVertex3fv (v1);             
             
               glEnd ();  
               }

               glColor3f(1.0, 1.0, 1.0); 
               glLineWidth((GLfloat)(graphblock->LineThickness)); 
               v1[2]=v1iso[2];
               v2[2]=v2iso[2];
               v3[2]=v3iso[2];
               v4[2]=v4iso[2];
               
               if (1==graphblock->DrawGridFlag)
               {
                  glColor3f(0.0, 0.0, 0.0);   
                  glBegin (GL_LINE_STRIP); 
                  glVertex3fv (v1);          
                  glVertex3fv (v2);          
                  glVertex3fv (v3);          
                  glVertex3fv (v4);          
                  glVertex3fv (v1);          
                  glEnd(); 
               }
               
               if (1==graphblock->DrawNodeFlag)
               {
               qobj=gluNewQuadric();
               glPushMatrix();
               glEnable(GL_LIGHTING);
               glEnable(GL_LIGHT0);
               glEnable(GL_DEPTH_TEST);
               glShadeModel(GL_SMOOTH);
               gluQuadricDrawStyle(qobj, GLU_FILL);
               gluQuadricNormals(qobj, GLU_SMOOTH);
               glTranslatef(v1[0],v1[1],v1[2]);
               gluSphere(qobj, (GLfloat)(graphblock->NodeSize),15,10);
               glPopMatrix();
               gluDeleteQuadric(qobj);
               
               qobj=gluNewQuadric();
               glPushMatrix();
               glEnable(GL_LIGHTING);
               glEnable(GL_LIGHT0);
               glEnable(GL_DEPTH_TEST);
               glShadeModel(GL_SMOOTH);
               gluQuadricDrawStyle(qobj, GLU_FILL);
               gluQuadricNormals(qobj, GLU_SMOOTH);
               glTranslatef(v2[0],v2[1],v2[2]);
               gluSphere(qobj, (GLfloat)(graphblock->NodeSize),15,10);
               glPopMatrix();
               gluDeleteQuadric(qobj);
               
               qobj=gluNewQuadric();
               glPushMatrix();
               glEnable(GL_LIGHTING);
               glEnable(GL_LIGHT0);
               glEnable(GL_DEPTH_TEST);
               glShadeModel(GL_SMOOTH);
               gluQuadricDrawStyle(qobj, GLU_FILL);
               gluQuadricNormals(qobj, GLU_SMOOTH);
               glTranslatef(v3[0],v3[1],v3[2]);
               gluSphere(qobj, (GLfloat)(graphblock->NodeSize),15,10);
               glPopMatrix();
               gluDeleteQuadric(qobj);
               
               qobj=gluNewQuadric();
               glPushMatrix();
               glEnable(GL_LIGHTING);
               glEnable(GL_LIGHT0);
               glEnable(GL_DEPTH_TEST);
               glShadeModel(GL_SMOOTH);
               gluQuadricDrawStyle(qobj, GLU_FILL);
               gluQuadricNormals(qobj, GLU_SMOOTH);
               glTranslatef(v4[0],v4[1],v4[2]);
               gluSphere(qobj, (GLfloat)(graphblock->NodeSize),15,10);
               glPopMatrix();
               gluDeleteQuadric(qobj);
               
               }
               
               glLineWidth(1.0);
               
               
               if (NumIsoLines>1) 
               {                                               
                 p=i+nxv*j;
                 minimum2=Post1[p];
                 maximum2=Post1[p];
                 if (Post2[p]>Post1[p]) maximum2=Post2[p]; else minimum2=Post2[p];
                 if (Post3[p]>maximum2) maximum2=Post3[p]; 
                 if (Post3[p]<minimum2) minimum2=Post3[p]; 
                 if (Post4[p]>maximum2) maximum2=Post4[p]; 
                 if (Post4[p]<minimum2) minimum2=Post4[p];
                 
                 MinIso=(int) (floor(minimum2-minimum)/(maximum-minimum)*NumIsoLines-1.0);
                 MaxIso=(int) (ceil(maximum2-minimum)/(maximum-minimum)*NumIsoLines+1.0);
                 if (MinIso<0) MinIso=0;
                 if (MaxIso>(NumIsoLines+1)) MaxIso=(NumIsoLines+1);
                 
                 for(k=MinIso; k<MaxIso; k++)           
                 {
                   
                   //case1
                   if (
                      (Post1[p]<IsoValues[k] && 
                      Post2[p]>IsoValues[k] &&
                      Post3[p]<IsoValues[k])
                      ||
                      (Post1[p]>IsoValues[k] && 
                      Post2[p]<IsoValues[k] &&
                      Post3[p]>IsoValues[k])
                      )
                      
                   {
                       CompIsoVector(v1, v2, Post1[p],  Post2[p],
                                     v2, v3, Post2[p],  Post3[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
                   
                   //case2
                   if (
                      (Post1[p]<IsoValues[k] && 
                       Post4[p]<IsoValues[k] && 
                       Post2[p]>IsoValues[k] &&
                       Post3[p]>IsoValues[k])
                      ||
                      (Post1[p]>IsoValues[k] && 
                       Post4[p]>IsoValues[k] && 
                       Post2[p]<IsoValues[k] &&
                       Post3[p]<IsoValues[k])
                      )
                      
                   {
                       CompIsoVector(v1, v2, Post1[p],  Post2[p],
                                     v4, v3, Post4[p],  Post3[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
                   
                   //case3
                   if (
                      (Post1[p]<IsoValues[k] && 
                       Post4[p]>IsoValues[k] && 
                       Post2[p]>IsoValues[k]) 
                      ||
                      (Post1[p]>IsoValues[k] && 
                       Post4[p]<IsoValues[k] && 
                       Post2[p]<IsoValues[k]) 
                      )
                      
                   {
                       CompIsoVector(v1, v2, Post1[p],  Post2[p],
                                     v1, v4, Post1[p],  Post4[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
                   
                   //case4
                   if (
                      (Post3[p]<IsoValues[k] && 
                       Post4[p]>IsoValues[k] && 
                       Post2[p]>IsoValues[k]) 
                      ||
                      (Post3[p]>IsoValues[k] && 
                       Post4[p]<IsoValues[k] && 
                       Post2[p]<IsoValues[k]) 
                      )
                      
                   {
                       CompIsoVector(v3, v2, Post3[p],  Post2[p],
                                     v3, v4, Post3[p],  Post4[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
                   
                   //case5
                   if (
                      (Post1[p]<IsoValues[k] && 
                       Post2[p]<IsoValues[k] && 
                       Post3[p]>IsoValues[k] &&
                       Post4[p]>IsoValues[k]) 
                      ||
                      (Post1[p]>IsoValues[k] && 
                       Post2[p]>IsoValues[k] && 
                       Post3[p]<IsoValues[k] &&
                       Post4[p]<IsoValues[k]) 
                      )
                      
                   {
                       CompIsoVector(v2, v3, Post2[p],  Post3[p],
                                     v1, v4, Post1[p],  Post4[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
                   
                   
                   //case6
                   if (
                      (Post1[p]<IsoValues[k] && 
                       Post3[p]<IsoValues[k] && 
                       Post4[p]>IsoValues[k]) 
                      ||
                      (Post1[p]>IsoValues[k] && 
                       Post3[p]>IsoValues[k] && 
                       Post4[p]<IsoValues[k]) 
                      )
                      
                   {
                       CompIsoVector(v1, v4, Post1[p],  Post4[p],
                                     v3, v4, Post3[p],  Post4[p],
                                     IsoValues[k], vp1, vp2);
                       
                       glBegin(GL_LINES);
                       glVertex3fv(vp1);
                       glVertex3fv(vp2);  
                       glEnd();
                   }
               
                 }
                 }
               }
            }
        }   
      
   
    
    
    istart=0;
    iend=nx;
    jstart=0;
    jend=ny;
    
    Umax=0.0;
    Umin=0.0;
    Fmax=0.0;
    Fmin=0.0;
    
    free(IsoValues);
    
    if (1==graphblock->DrawBCFlag)
    {   
        for (i=istart; i<iend; i+=OptimStepX)
        {
            x=la*i;
            for (j=jstart; j<jend; j+=OptimStepY)
            {
               y=lb*j;
               
               if (1==BcTypeX[i+nx*j] && 0.0!=BcValueX[i+nx*j]) 
               {
                   if (Umin>BcValueX[i+nx*j]) Umin=(float)BcValueX[i+nx*j];
                   if (Umax<BcValueX[i+nx*j]) Umax=(float)BcValueX[i+nx*j];
               }
               if (1==BcTypeY[i+nx*j]&& 0.0!=BcValueY[i+nx*j]) 
               {
                   if (Umin>BcValueY[i+nx*j]) Umin=(float)BcValueY[i+nx*j];
                   if (Umax<BcValueY[i+nx*j]) Umax=(float)BcValueY[i+nx*j];
               }
               if (0==BcTypeX[i+nx*j] && 0.0!=BcValueX[i+nx*j]) 
               {
                   if (Fmin>BcValueX[i+nx*j]) Fmin=(float)BcValueX[i+nx*j];
                   if (Fmax<BcValueX[i+nx*j]) Fmax=(float)BcValueX[i+nx*j];
               }
               if (0==BcTypeY[i+nx*j] && 0.0!=BcValueY[i+nx*j]) 
               {
                   if (Fmin>BcValueY[i+nx*j]) Fmin=(float)BcValueY[i+nx*j];
                   if (Fmax<BcValueY[i+nx*j]) Fmax=(float)BcValueY[i+nx*j];
               }
            }
        } 
        
        if (fabs(Umin)>fabs(Umax)) Urange=(float)fabs(Umin); else Urange=(float)fabs(Umax); 
        if (fabs(Fmin)>fabs(Fmax)) Frange=(float)fabs(Fmin); else Frange=(float)fabs(Fmax); 
        //if (Urange!=0.0) LoadScaleU=sqrt(la*nx*lb*ny)*0.1/Urange; else LoadScaleU=0.0;
        Urange=sqrt(la*nx*lb*ny)*0.03;
        if (Frange!=0.0) LoadScaleF=sqrt(la*nx*lb*ny)*0.1/Frange; else LoadScaleF=0.0;
                
        glColor3f(0.0, 0.0, 0.0); 
        for (i=istart; i<iend; i+=OptimStepX)
        {
            x=la*i;
            for (j=jstart; j<jend; j+=OptimStepY)
            {
               y=lb*j;
               
               if (0==BcTypeX[i+nx*j] && 0.0!=BcValueX[i+nx*j]) 
               {
                   
                   LoadLen=(float)(LoadScaleF*BcValueX[i+nx*j]);
                   DrawArrow((float)(x+Ux[i+nx*j]*df), 
                           (float)(y+Uy[i+nx*j]*df), LoadLen, 0);
               }           
               if (0==BcTypeY[i+nx*j] && 0.0!=BcValueY[i+nx*j]) 
               {
                   
                   LoadLen=(float)(LoadScaleF*BcValueY[i+nx*j]);
                   DrawArrow((float)(x+Ux[i+nx*j]*df), 
                           (float)(y+Uy[i+nx*j]*df), LoadLen, 1);
               }  
               if (1==BcTypeX[i+nx*j]) 
               {
                   
                   LoadLen=(float)(LoadScaleF*BcValueX[i+nx*j]);
                   DrawTriangle((float)(x+Ux[i+nx*j]*df), 
                           (float)(y+Uy[i+nx*j]*df), Urange, 0);
               }           
               if (1==BcTypeY[i+nx*j]) 
               {
                   
                   LoadLen=(float)(LoadScaleF*BcValueY[i+nx*j]);
                   DrawTriangle((float)(x+Ux[i+nx*j]*df), 
                           (float)(y+Uy[i+nx*j]*df), Urange, 1);
               }
            
            
            }
        }   
      
      
      
      
      
      
      
      }
    
    DrawScale();
     
    if (0<graphblock->BmpScreenFlag)
    {
        WriteBmp2((int) w, (int) h);
        if (1==graphblock->BmpScreenFlag) graphblock->BmpScreenFlag=0;
    }
     //WriteBmp2((int) w, (int) h);
    /* bitmap */
    //unsigned char* pixels = (unsigned char*)malloc((3*((unsigned int)w)+4)*
    //                         ((unsigned int)h)*sizeof(unsigned char*));
    //glReadPixels(0, 0, ((unsigned int)w), ((unsigned int)h), GL_RGB, GL_UNSIGNED_BYTE, pixels);
    //WriteBmp2((int) w, (int) h,  pixels);
    //free(pixels);
    
    /*
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();   
    
    w=(double)graphblock->WinPixelWidth;
    h=(double)graphblock->WinPixelHeight;
    gluOrtho2D(0,(GLfloat)w, 0, (GLfloat)h);
    
    glRasterPos3f(0.4,0.4,0.01);
    glGetFloatv(GL_CURRENT_RASTER_COLOR, rastercolor);
    glFlush();
    printf("rgb: %f %f %f\n", (double) rastercolor[0],(double) rastercolor[1],(double) rastercolor[2]); */
return;
}

/*------------------------------------------------------------------------------
     CompNormalvector
------------------------------------------------------------------------------*/
void CompNormalVector(float *v1, float *v2, float *v3, float *vn) 
{
    float a[3],b[3];
    
    a[0]=v1[0]-v2[0];
    a[1]=v1[1]-v2[1];
    a[2]=v1[2]-v2[2];
    
    b[0]=v2[0]-v3[0];
    b[1]=v2[1]-v3[1];
    b[2]=v2[2]-v3[2];
  
    vn[0]=a[1]*b[2]-a[2]*b[1];
    vn[1]=a[2]*b[0]-a[0]*b[2];
    vn[2]=a[0]*b[1]-a[1]*b[0];
  
   return; 
}


/*------------------------------------------------------------------------------
     CompIsoVector
------------------------------------------------------------------------------*/
void CompIsoVector(float *v1, float *v2, float w1,  float w2,
                   float *v3, float *v4, float w3,  float w4,
                   float wiso, 
                   float *vp1, float *vp2)
{   
    GLfloat vd[3];
    float s;
    
    vd[0]=v2[0]-v1[0];
    vd[1]=v2[1]-v1[1];
    vd[2]=v2[2]-v1[2];
    
    s=(wiso-w1)/(w2-w1);
    
    vp1[0]=s*vd[0]+v1[0];
    vp1[1]=s*vd[1]+v1[1];
    vp1[2]=s*vd[2]+v1[2];
    
    vd[0]=v4[0]-v3[0];
    vd[1]=v4[1]-v3[1];
    vd[2]=v4[2]-v3[2];
    
    s=(wiso-w3)/(w4-w3);
    
    vp2[0]=s*vd[0]+v3[0];
    vp2[1]=s*vd[1]+v3[1];
    vp2[2]=s*vd[2]+v3[2];
  
   return; 
}

/*------------------------------------------------------------------------------
     DrawScale
------------------------------------------------------------------------------*/
void DrawScale() 
{
    int i;
    double w,h,stp;
    float *r, *g, *b;
    int NumColor;
    int Numvalue;
    int act_palette;
    int ShowValues;
    int act;
    char s[80];
    double Valuestep, Pixelstep;
    
    GraphicBlock *graphblock;
    PutGetGraphicBlock(1, &graphblock);
   
    act=graphblock->act_palette;
    ShowValues=graphblock->ShowValues;
    r=graphblock->CurrentPalette[act].r;
    g=graphblock->CurrentPalette[act].g;
    b=graphblock->CurrentPalette[act].b;
    NumColor=graphblock->CurrentPalette[act].NumColor;
    
    w=(double)graphblock->WinPixelWidth;
    h=(double)graphblock->WinPixelHeight;
       
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();   
    
    glDisable(GL_LIGHTING);
    glDisable(GL_LIGHT0);
    glDisable(GL_COLOR_MATERIAL);
    glDisable(GL_DEPTH_TEST);
    
    gluOrtho2D(0,(GLfloat)w, 0, (GLfloat)h);
 
    stp=(float) 0.9*h/NumColor;
    act_palette=graphblock->act_palette;
    switch (act_palette)
            {
                case 0: Numvalue=32; break;
                case 1: Numvalue=32; break;
                case 2: Numvalue=32; break;
                case 3: Numvalue=9;  break;
                case 4: Numvalue=32; break;
                case 5: Numvalue=32; break;
                case 6: Numvalue=8;  break;
                case 7: Numvalue=32; break;
                case 8: Numvalue=32; break;
                case 9: Numvalue=3; break;
             }
    Pixelstep=0.9*h/Numvalue;
    for (i=0; i<NumColor; i++)
    {
        glColor3f (r[i], g[i], b[i]);
        glRectf(0.96*w,0.05*h+stp*i,0.99*w,0.05*h+stp*(i+1));      
    } 
    
    glColor3f(1.0, 1.0, 1.0);  
    
    
    if (act_palette!=5 && 1==ShowValues)
    {
        glBegin(GL_LINES);
        for (i=0; i<(Numvalue+1); i++)
        {
            glVertex2f(0.961*w,0.05*h+Pixelstep*i);
            glVertex2f(0.99*w,0.05*h+Pixelstep*i);
        } 
        glEnd();
    }
    
    
    glColor3f(0.0, 0.0, 0.0);
    sprintf(s, "%+4.4E", graphblock->minimum); 
	DrawText((int)(0.96*w-85),(int)(0.05*h+2), s);
    sprintf(s, "%+4.4E", graphblock->maximum); 
	DrawText((int)(0.96*w-85),(int)(0.95*h-9), s);
	
	Pixelstep=(graphblock->maximum - graphblock->minimum)/Numvalue;
    
    if (1==ShowValues)
    for (i=1; i<Numvalue; i++)
    {
        sprintf(s, "%+4.4E", (Pixelstep*(double)i+graphblock->minimum)); 
	    DrawText((int)(0.96*w-85),(int)( ((0.95*h-9)-(0.05*h+2))/Numvalue*i+(0.05*h+2)), s);
    
    }
	
   
return;
} 


/*------------------------------------------------------------------------------
     DrawText
------------------------------------------------------------------------------*/
void DrawText(int x, int y, char *string) {
  int len, i;

  glRasterPos2f(x, y);
  len = (int) strlen(string);
  /*for (i = 0; i < len; i++)
	  glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12 , string[i]);*/
}

/*------------------------------------------------------------------------------
     DrawArrow
------------------------------------------------------------------------------*/
void DrawArrow(float x, float y, float length, int direct) 
{
  float height;
  height=fabs(length); 
  //length=10;
  if (0==direct)
  {
      glBegin(GL_LINES);
               
               glVertex3f(x,y,height);
               glVertex3f(x+length,y,height);  
               
               glVertex3f(x+length,y,height);
               glVertex3f(x+length*0.8,y+length*0.1,height);
               
               glVertex3f(x+length,y,height);
               glVertex3f(x+length*0.8,y-length*0.1,height);
      glEnd();
  }
  else
  {
      glBegin(GL_LINES);
               
               glVertex3f(x,y,height);
               glVertex3f(x,y+length,height);  
               
               glVertex3f(x,y+length,height);
               glVertex3f(x+length*0.1,y+length*0.8,height);
               
               glVertex3f(x,y+length,height);
               glVertex3f(x-length*0.1,y+length*0.8,height);
      glEnd();
  }
  
  return;
}

/*------------------------------------------------------------------------------
     DrawTriangle
------------------------------------------------------------------------------*/
void DrawTriangle(float x, float y, float length, int direct) 
{
  float height;
  height=fabs(length); 
  //length=10;
  if (0==direct)
  {
      glBegin(GL_LINES);
               glVertex3f(x,y,height);
               glVertex3f(x-length/1.5,y-length/4.0,height);  
               
               glVertex3f(x-length/1.5,y-length/4.0,height);  
               glVertex3f(x-length/1.5,y+length/4.0,height);
               
               glVertex3f(x-length/1.5,y+length/4.0,height);
               glVertex3f(x,y,height);
      glEnd();
  }
  else
  {
      glBegin(GL_LINES);
               glVertex3f(x,y,height);
               glVertex3f(x-length/4.0,y-length/1.5,height);  
               
               glVertex3f(x-length/4.0,y-length/1.5,height);  
               glVertex3f(x+length/4.0,y-length/1.5,height);
               
               glVertex3f(x+length/4.0,y-length/1.5,height);
               glVertex3f(x,y,height);
      glEnd();
  }
  
  return;
}


/*------------------------------------------------------------------------------
     PickColor
------------------------------------------------------------------------------*/
int PickColor(int NumColor, double min, double max, double val) 
{
    return ((int)((val-min)/(max-min)*(NumColor-1)+0.499));
}  


/*------------------------------------------------------------------------------
     DrawMesoStructure3d
------------------------------------------------------------------------------*/
void DrawMesoStructure3d() 
{
    int i,j,k,p;
    float stpx,stpy,stpz; 
    float x,y,z;

    float zoomvar;
    float rotx;
    float rotz;
    double la,lb,lc;
    int nx,ny,nz,nxv,nyv,nzv;
    double w,h;
    double ratio_pixels, ratio_real;
    GLdouble left, right, bottom, top;
    double width, height;
    double transX, transY;
    int optimStep;
    double px,pex,pey;
    int OptimStepX,OptimStepY;
    int istart,iend;
    int jstart,jend;
    int kstart,kend;
    SolveBlock *SolBlock;
    int num_levels;
    int show_level; 
    double *Post1;
    double *Post2;
    double *Post3;
    double *Post4;
    double *Post5;
    double *Post6;
    double *Post7;
    double *Post8;
    double minimum, maximum;
    double minimum2, maximum2;
    double val;
    int col;
    float *r, *g, *b;
    int NumColor;
    float df;
    int act;
    GraphicBlock *graphblock;
    MyMainWindowUI   *mainwin;
    double *Ux, *Uy;
    int nex, ney, nez;
    double zdepth;
    double ExtrudeZ, Extrude;
    int postflag;  
    float LoadScaleU;
    float LoadScaleF;
    float LoadLen;
    float Umax,Umin,Urange,Fmax,Fmin,Frange;
    int *BcTypeX, *BcTypeY;
    double *BcValueX, *BcValueY;
    
    int NumIsoLines;
    double *IsoValues;
    int MinIso, MaxIso;
    MaterialBlock *Material;
    int NumMaterials;
    int *MatFieldInt;
    int AllVisible=1;
    int tmp, tmp2;
    
    GLUquadricObj *qobj;
    
    //GLfloat mat_specular[] = {0.3,0.4, 0.3, 1.0};
    GLfloat mat_shininess[] = {3.0};
    //GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat light_position[] = {1.0,1.0,30.0,0.0};
    GLfloat white_light[] = {0.6,0.6,0.6,1.0};
    GLfloat lmodel_ambient[] = {0.1,0.1,0.1,1.0};
    //GLfloat mat_emission[] = {0.3, 0.2, 0.2, 0.0};
    
    
    GLfloat v1[3];
    GLfloat v2[3];
    GLfloat v3[3];
    GLfloat v4[3];   
    GLfloat v5[3];
    GLfloat v6[3];
    GLfloat v7[3];
    GLfloat v8[3]; 
    GLfloat vn[3];
    GLfloat vcol1[3];
    GLfloat vcol2[3];
    GLfloat vcol3[3];
    GLfloat vcol4[3];
    GLfloat vcol5[3];
    GLfloat vcol6[3];
    GLfloat vcol7[3];
    GLfloat vcol8[3];
    
   
    GLfloat vp1[3];
    GLfloat vp2[3];
    
    /*----------------- for bitmap -----*/    
    GLfloat rastercolor[4];
    GLfloat rasterposition[4];
   
    
    /*--------------------- Get data from SOLVEBLOCK -------------------------*/   
    PutGetSolveBlock(1,&SolBlock);
    PutGetGraphicBlock(1, &graphblock); 
    num_levels=SolBlock->num_levels;
    
    // either show_level==detail_level or detail_level is relevant, thus ...
    show_level=graphblock->detail_level;

     
    la=SolBlock->Meshes[show_level].la;
    lb=SolBlock->Meshes[show_level].lb;
    lc=SolBlock->Meshes[show_level].lc;
    nx=SolBlock->Meshes[show_level].nnx;
    ny=SolBlock->Meshes[show_level].nny;
    nz=SolBlock->Meshes[show_level].nnz;

    
    Post1=SolBlock->Meshes[show_level].Post1;
    Post2=SolBlock->Meshes[show_level].Post2;
    Post3=SolBlock->Meshes[show_level].Post3;
    Post4=SolBlock->Meshes[show_level].Post4;

    Post5=SolBlock->Meshes[show_level].Post5;
    Post6=SolBlock->Meshes[show_level].Post6;
    Post7=SolBlock->Meshes[show_level].Post7;
    Post8=SolBlock->Meshes[show_level].Post8;   
    
        
    MatFieldInt=SolBlock->Meshes[show_level].MatFieldInt;
    Material=SolBlock->Material; 
    NumMaterials=SolBlock->NumMaterials;
    for (i=0; i<NumMaterials; i++) if (0==Material[i].Visible) AllVisible=0;
    
    
    nex=nx-1;
    ney=ny-1;
    nez=nz-1;
   
    nxv=nx-1;
    nyv=ny-1;
    nzv=nz-1;
        
    
    FindMinMaxVector(nxv*nyv*nzv, Post1, &minimum, &maximum);
    FindMinMaxVector(nxv*nyv*nzv, Post2, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post3, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post4, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post5, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post6, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post7, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    FindMinMaxVector(nxv*nyv*nzv, Post8, &minimum2, &maximum2);
    if (minimum2<minimum) minimum=minimum2;
    if (maximum2>maximum) maximum=maximum2;
    
       
       
    stpx=la;  
    stpy=lb;  
    stpz=lc; 

    
    /*--------------------- Get data from GRAPHBLOCK -------------------------*/ 
    PutGetGraphicBlock(1, &graphblock);
    zoomvar=graphblock->Zoomfactor;
    rotx=graphblock->RotateAngleX;
    rotz=graphblock->RotateAngleZ;
    ExtrudeZ=graphblock->ExtrudeZ;
    Extrude=graphblock->Extrude;
    df=graphblock->Deflect;
    NumIsoLines=graphblock->IsoLines;
    
    w=(double)graphblock->WinPixelWidth;
    h=(double)graphblock->WinPixelHeight;
    
    transX=(double)graphblock->transX;
    transY=(double)graphblock->transY;
    
    
    /*------------------------ Write to GRAPHBLOCK ---------------------------*/
    if (minimum==0.0 && maximum==0.0){minimum=-1.0;maximum=1.0;}
    if (0.0!=minimum)
    if ( ((maximum-minimum)/minimum) <  0.000001 &&
         ((maximum-minimum)/minimum) > -0.000001) 
    {
        minimum*=0.99999;
        maximum*=1.00001;
    }
    
    graphblock->minimum=minimum;
    graphblock->maximum=maximum;
    
    ratio_pixels=(h/w);
    ratio_real=((ny*lb)/(nx*la));
    
    
    /*-------------------Calculate visible real world range-------------------*/
    if (ratio_pixels>ratio_real)
    {
        width =nex*la*1.1;
        height=ratio_pixels*width;
    }
    else
    {
        height=ney*lb*1.1;
        width =height/ratio_pixels;
    }
    
    width/=zoomvar;
    height/=zoomvar;
    
    
    PutGetMainWindow(1, &mainwin); 
    left=(nex*la/2)-width/2+nex*la*transX;     mainwin->MinX->value((float)left);
    right=(nex*la/2)+width/2+nex*la*transX;    mainwin->MaxX->value((float)right);
    
    bottom=(ney*lb/2)-height/2-ney*lb*transY;  mainwin->MinY->value((float)bottom);
    top=(ney*lb/2)+height/2-ney*lb*transY;;    mainwin->MaxY->value((float)top);


    OptimStepX=1;
    OptimStepY=1;
    istart=0;
    jstart=0;
    kstart=0;
    iend=nx-2;
    jend=ny-2;
    kend=nz-2;
    
    
    
    /*------------------------- Get Color Palette ----------------------------*/
    act=graphblock->act_palette;
    r=graphblock->CurrentPalette[act].r;
    g=graphblock->CurrentPalette[act].g;
    b=graphblock->CurrentPalette[act].b;
    NumColor=graphblock->CurrentPalette[act].NumColor;

    /*------------------------- Draw BMP if flag is on -----------------------*/
    
    
    if (0<graphblock->BmpDataFlag)
    {
        WriteBmp(nex, ney, Post1, Post2, Post3, Post4,
                 minimum, maximum, graphblock->CurrentPalette[act]);
        if (1==graphblock->BmpDataFlag) graphblock->BmpDataFlag=0;
    }
    
    
    /*---------------------------- Draw Patches ------------------------------*/
    zdepth=sqrt(width*width+height*height);
   
    
        
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();  
    
    glOrtho(left,right,bottom,top,-zdepth,zdepth);
    glTranslatef( (nex*la/2) , (ney*lb/2) , ((double)0.0) );
    glRotatef(rotx,1.0,0.0,0.0);
    glRotatef(rotz,0.0,0.0,1.0);
    glTranslatef( (-nex*la/2) , (-ney*lb/2) , ((double)0.0) );
  
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);
    
    
    if (1==AllVisible)
    {
        for (k=0; k<kend+1; k+=kend)
        for (i=0; i<iend+1; i++)
        {
            for (j=0; j<jend+1; j++)
            {
                p=i+nxv*j+nxv*nyv*k;    
                if (1==Material[MatFieldInt[p]].Visible)               
                {     
                DrawOneCube3d(i, j, k, la, lb, lc, 
                Post1[p], Post2[p], Post3[p],  Post4[p], Post5[p], Post6[p], Post7[p], Post8[p],
                minimum, maximum, graphblock, NumColor, r, g, b); 
                }
            }
        }  
          
        
        for (i=0; i<iend+1; i+=iend)
        for (k=0; k<kend+1; k++)
        {
            for (j=0; j<jend+1; j++)
            {
                p=i+nxv*j+nxv*nyv*k;    
                if (1==Material[MatFieldInt[p]].Visible)               
                {
                DrawOneCube3d(i, j, k, la, lb, lc, 
                Post1[p], Post2[p], Post3[p],  Post4[p], Post5[p], Post6[p], Post7[p], Post8[p],
                minimum, maximum, graphblock, NumColor, r, g, b); 
                }
            }
        }  
     
        
        for (j=0; j<jend+1; j+=jend)
        for (k=0; k<kend+1; k++)
        {
            for (i=0; i<iend+1; i++)
            {
                p=i+nxv*j+nxv*nyv*k;
                if (1==Material[MatFieldInt[p]].Visible)               
                {    
                DrawOneCube3d(i, j, k, la, lb, lc, 
                Post1[p], Post2[p], Post3[p],  Post4[p], Post5[p], Post6[p], Post7[p], Post8[p],
                minimum, maximum, graphblock, NumColor, r, g, b); 
                }
            }
        }  
    }
    else
    {
        for (k=0; k<kend+1; k++)
        for (i=0; i<iend+1; i++)
        {
            for (j=0; j<jend+1; j++)
            {
                p=i+nxv*j+nxv*nyv*k;    
                if (1==Material[MatFieldInt[p]].Visible)               
                {     
                DrawOneCube3d(i, j, k, la, lb, lc, 
                Post1[p], Post2[p], Post3[p],  Post4[p], Post5[p], Post6[p], Post7[p], Post8[p],
                minimum, maximum, graphblock, NumColor, r, g, b); 
                }
            }
        } 
    
    }
    
    if (1==graphblock->DrawFrameFlag)
    {
          tmp=graphblock->DrawPatchFlag;
          graphblock->DrawPatchFlag=0;
          
          tmp2=graphblock->DrawGridFlag;
          graphblock->DrawGridFlag=1;
          
          DrawOneCube3d(0, 0, 0, (nx-1)*la, (ny-1)*lb, (nz-1)*lc, 
          0.1, 0.1, 0.1,  0.1, 0.1, 0.1, 0.1, 0.1,
          0.0, 1.0, graphblock, NumColor, r, g, b);
          
          graphblock->DrawPatchFlag=tmp;
          graphblock->DrawGridFlag=tmp2;
    }
    
    
    DrawScale();
    
  
      
    if (0<graphblock->BmpScreenFlag)
    {
        WriteBmp2((int) w, (int) h);
        if (1==graphblock->BmpScreenFlag) graphblock->BmpScreenFlag=0;
    }

return;
}

/*------------------------------------------------------------------------------
     DrawOneCube3d
------------------------------------------------------------------------------*/
void DrawOneCube3d(int i, int j, int k, double la, double lb, double lc,  
                   double Post1,  double Post2, double Post3,  double Post4,
                   double Post5,  double Post6, double Post7,  double Post8,
                   double minimum, double maximum, GraphicBlock *graphblock,
                   int NumColor, float *r, float *g, float *b) 
{
    int col;
    double val;
    double x,y,z;
    double dx,dy,dz;
    GLfloat v1[3];
    GLfloat v2[3];
    GLfloat v3[3];
    GLfloat v4[3];   
    GLfloat v5[3];
    GLfloat v6[3];
    GLfloat v7[3];
    GLfloat v8[3];
    GLfloat vcol1[3];
    GLfloat vcol2[3];
    GLfloat vcol3[3];
    GLfloat vcol4[3];
    GLfloat vcol5[3];
    GLfloat vcol6[3];
    GLfloat vcol7[3];
    GLfloat vcol8[3]; 
    double shrink=0.00;
    
               
               x=la*i;
               y=lb*j;
               z=lc*k;
               dx=shrink*la;
               dy=shrink*lb;
               dz=shrink*lc;
               
               v1[0]=(GLfloat)(x+dx);
               v1[1]=(GLfloat)(y+dy);
               v1[2]=(GLfloat)(z+dz);    
               
               v2[0]=(GLfloat)(x+la-dx);
               v2[1]=(GLfloat)(y+dy);
               v2[2]=(GLfloat)(z+dz);     
               
               v3[0]=(GLfloat)(x+la-dx);
               v3[1]=(GLfloat)(y+lb-dy);
               v3[2]=(GLfloat)(z+dz);      
               
               v4[0]=(GLfloat)(x+dx);
               v4[1]=(GLfloat)(y+lb-dy);
               v4[2]=(GLfloat)(z+dz);         
               
               v5[0]=(GLfloat)(x+dx);
               v5[1]=(GLfloat)(y+dy);
               v5[2]=(GLfloat)(z+lc-dz);    
               
               v6[0]=(GLfloat)(x+la-dx);
               v6[1]=(GLfloat)(y+dy);
               v6[2]=(GLfloat)(z+lc-dz);     
               
               v7[0]=(GLfloat)(x+la-dx);
               v7[1]=(GLfloat)(y+lb-dy);
               v7[2]=(GLfloat)(z+lc-dz);      
               
               v8[0]=(GLfloat)(x+dx);
               v8[1]=(GLfloat)(y+lb-dy);
               v8[2]=(GLfloat)(z+lc-dz);  
               
               val=Post1;col=PickColor(NumColor, minimum, maximum, val); 
               vcol1[0]=r[col]; vcol1[1]=g[col]; vcol1[2]=b[col];
               
               val=Post2;col=PickColor(NumColor, minimum, maximum, val); 
               vcol2[0]=r[col]; vcol2[1]=g[col]; vcol2[2]=b[col];
               
               val=Post3;col=PickColor(NumColor, minimum, maximum, val); 
               vcol3[0]=r[col]; vcol3[1]=g[col]; vcol3[2]=b[col];

               val=Post4;col=PickColor(NumColor, minimum, maximum, val); 
               vcol4[0]=r[col]; vcol4[1]=g[col]; vcol4[2]=b[col];

               val=Post5;col=PickColor(NumColor, minimum, maximum, val); 
               vcol5[0]=r[col]; vcol5[1]=g[col]; vcol5[2]=b[col];

               val=Post6;col=PickColor(NumColor, minimum, maximum, val); 
               vcol6[0]=r[col]; vcol6[1]=g[col]; vcol6[2]=b[col];
               
               val=Post7;col=PickColor(NumColor, minimum, maximum, val); 
               vcol7[0]=r[col]; vcol7[1]=g[col]; vcol7[2]=b[col];
               
               val=Post8;col=PickColor(NumColor, minimum, maximum, val); 
               vcol8[0]=r[col]; vcol8[1]=g[col]; vcol8[2]=b[col];              
                                             
                
               if (1==graphblock->DrawPatchFlag)
               {
                   glBegin (GL_POLYGON); 
                       glColor3fv (vcol1);glVertex3fv (v1);              
                       glColor3fv (vcol2);glVertex3fv (v2);    
                       glColor3fv (vcol3);glVertex3fv (v3);
                       glColor3fv (vcol4);glVertex3fv (v4);
                       glColor3fv (vcol1);glVertex3fv (v1);
                    glEnd ();  
                    
                    glBegin (GL_POLYGON); 
                       glColor3fv (vcol5);glVertex3fv (v5);              
                       glColor3fv (vcol6);glVertex3fv (v6);    
                       glColor3fv (vcol7);glVertex3fv (v7);
                       glColor3fv (vcol8);glVertex3fv (v8);
                       glColor3fv (vcol5);glVertex3fv (v5);
                    glEnd ();
                    
                    glBegin (GL_POLYGON); 
                       glColor3fv (vcol3);glVertex3fv (v3);              
                       glColor3fv (vcol7);glVertex3fv (v7);    
                       glColor3fv (vcol8);glVertex3fv (v8);
                       glColor3fv (vcol4);glVertex3fv (v4);
                       glColor3fv (vcol3);glVertex3fv (v3);
                    glEnd ();
                    
                    glBegin (GL_POLYGON); 
                       glColor3fv (vcol2);glVertex3fv (v2);              
                       glColor3fv (vcol6);glVertex3fv (v6);    
                       glColor3fv (vcol7);glVertex3fv (v7);
                       glColor3fv (vcol3);glVertex3fv (v3);
                       glColor3fv (vcol2);glVertex3fv (v2);
                    glEnd ();
                    
                    glBegin (GL_POLYGON); 
                       glColor3fv (vcol1);glVertex3fv (v1);              
                       glColor3fv (vcol5);glVertex3fv (v5);    
                       glColor3fv (vcol8);glVertex3fv (v8);
                       glColor3fv (vcol4);glVertex3fv (v4);
                       glColor3fv (vcol1);glVertex3fv (v1);
                    glEnd ();
                    
                    glBegin (GL_POLYGON); 
                       glColor3fv (vcol1);glVertex3fv (v1);              
                       glColor3fv (vcol2);glVertex3fv (v2);    
                       glColor3fv (vcol6);glVertex3fv (v6);
                       glColor3fv (vcol5);glVertex3fv (v5);
                       glColor3fv (vcol1);glVertex3fv (v1);
                    glEnd ();
                    
                    
               }

             
               glLineWidth((GLfloat)(graphblock->LineThickness)); 
               
               if (1==graphblock->DrawGridFlag)
               {
               v1[0]=(GLfloat)(x);
               v1[1]=(GLfloat)(y);
               v1[2]=(GLfloat)(z);    
               
               v2[0]=(GLfloat)(x+la);
               v2[1]=(GLfloat)(y);
               v2[2]=(GLfloat)(z);     
               
               v3[0]=(GLfloat)(x+la);
               v3[1]=(GLfloat)(y+lb);
               v3[2]=(GLfloat)(z);      
               
               v4[0]=(GLfloat)(x);
               v4[1]=(GLfloat)(y+lb);
               v4[2]=(GLfloat)(z);         
               
               v5[0]=(GLfloat)(x);
               v5[1]=(GLfloat)(y);
               v5[2]=(GLfloat)(z+lc);    
               
               v6[0]=(GLfloat)(x+la);
               v6[1]=(GLfloat)(y);
               v6[2]=(GLfloat)(z+lc);     
               
               v7[0]=(GLfloat)(x+la);
               v7[1]=(GLfloat)(y+lb);
               v7[2]=(GLfloat)(z+lc);      
               
               v8[0]=(GLfloat)(x);
               v8[1]=(GLfloat)(y+lb);
               v8[2]=(GLfloat)(z+lc); 
                  
                  glColor3f(0.0, 0.0, 0.0);   
                  glBegin (GL_LINE_STRIP); 
                      glVertex3fv (v1);          
                      glVertex3fv (v2);          
                      glVertex3fv (v3);          
                      glVertex3fv (v4);          
                      glVertex3fv (v1);          
                  glEnd(); 
                  
                  glBegin (GL_LINE_STRIP); 
                      glVertex3fv (v5);          
                      glVertex3fv (v6);          
                      glVertex3fv (v7);          
                      glVertex3fv (v8);          
                      glVertex3fv (v5);          
                  glEnd();
                  
                  glBegin (GL_LINES); 
                      glVertex3fv (v3);          
                      glVertex3fv (v7);                    
                  glEnd();
                  
                   glBegin (GL_LINES); 
                      glVertex3fv (v2);          
                      glVertex3fv (v6);                    
                  glEnd();
                  
                   glBegin (GL_LINES); 
                      glVertex3fv (v1);          
                      glVertex3fv (v5);                    
                  glEnd();
                  
                   glBegin (GL_LINES); 
                      glVertex3fv (v4);          
                      glVertex3fv (v8);                    
                  glEnd();               
                  
               }     
return;
}
    


